/*  Sjaak, a program for playing chess variants
 *  Copyright (C) 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <ctype.h>
#include "sjaak.h"

game_t *create_standard_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags;
   flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
   flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Bishop", "B,b", "B");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Queen", "Q,q", "Q");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL|PF_CASTLE, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)board_rank8, "QRBN", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)board_rank1, "QRBN", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   flags = define_stepper("step 2N");
   add_special_move(game, "P", (ubitboard_t)board_rank2, flags);
   flags = define_stepper("step 2S");
   add_special_move(game, "p", (ubitboard_t)board_rank7, flags);

   /* Set castle flags (these actually reproduce the default) */
   deduce_castle_flags(game->large_board, WHITE, E1, G1, H1);
   deduce_castle_flags(game->large_board, WHITE, E1, C1, A1);
   deduce_castle_flags(game->large_board, BLACK, E8, G8, H8);
   deduce_castle_flags(game->large_board, BLACK, E8, C8, A8);

   set_piece_value(game, "Q", 950);
   set_piece_value(game, "R", 500);
   set_piece_value(game, "B", 325);
   set_piece_value(game, "N", 320);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq -");
   game->name = strdup("Chess");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_berolina_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags;
   flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
   flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Bishop", "B,b", "B");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Queen", "Q,q", "Q");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL|PF_CASTLE, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step NE, NW");
   cflags = define_stepper("step N");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)board_rank8, "QRBN", "Pawn", "P", " ");
   flags = define_stepper("step SE, SW");
   cflags = define_stepper("step S");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)board_rank1, "QRBN", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   flags = define_stepper("step 2NE,2NW");
   add_special_move(game, "P", (ubitboard_t)board_rank2, flags);
   flags = define_stepper("step 2SE,2SW");
   add_special_move(game, "p", (ubitboard_t)board_rank7, flags);

   set_piece_value(game, "Q", 950);
   set_piece_value(game, "R", 500);
   set_piece_value(game, "B", 325);
   set_piece_value(game, "N", 320);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq -");
   game->name = strdup("Berolina chess");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_pocketknight_game(void)
{
   game_t *game = create_standard_game();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR[Nn] w KQkq -");
   game->xb_setup  = strdup("(PNBRQ................Kpnbrq................k) 8x8+2");
   game->name = strdup("Pocket Knight Chess");
   game->board.use_holdings = true;

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_chinese_game(void)
{
   game_t *game = create_game();

   /* Play on a 9x10 board */
   set_board_size(game, 9, 10);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags;
   flags = define_leaper("leap (1,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Guard", "A,a", "G");
   flags = define_leaper("leap ((0,1)+(1,1))&(2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Horse", "H,h", "H");
   flags = define_leaper("leap ((1,1)+(1,1))&(2,2)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Elephant", "E", "E");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Elephant", ",e", "E");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V)");
   cflags = define_slider("hop (H,V)");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)large_board_empty, "", "Cannon", "C,c", "C");
   flags = define_leaper("leap (0,1)");
   add_piece_type(game, flags, flags, PF_ROYAL, (ubitboard_t)large_board_empty, "", "King", "K,k", "K");
   flags = define_leaper("aleap (0,1)|(-1,0)|(1,0)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Pawn", "P", " ");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   large_bitboard_t south = large_board_empty;
   large_bitboard_t north = large_board_empty;
   large_bitboard_t pawnf = large_board_empty;
   south = large_board_rank[0] | large_board_rank[1] | large_board_rank[2] | large_board_rank[3] | large_board_rank[4];
   north = large_board_rank[5] | large_board_rank[6] | large_board_rank[7] | large_board_rank[8] | large_board_rank[9];
   pawnf = large_board_file[0] | large_board_file[2] | large_board_file[4] | large_board_file[6] | large_board_file[8];

   /* Guards and kings are restricted to the castles */
   large_bitboard_t castle = large_square_bitboards[ 3] | large_square_bitboards[ 4] | large_square_bitboards[ 5] | 
                             large_square_bitboards[12] | large_square_bitboards[13] | large_square_bitboards[14] | 
                             large_square_bitboards[21] | large_square_bitboards[22] | large_square_bitboards[23] | 
                             large_square_bitboards[66] | large_square_bitboards[67] | large_square_bitboards[68] | 
                             large_square_bitboards[75] | large_square_bitboards[76] | large_square_bitboards[77] | 
                             large_square_bitboards[84] | large_square_bitboards[85] | large_square_bitboards[86];
   limit_piece_movement_region(game, "A", (ubitboard_t)castle);
   limit_piece_movement_region(game, "K", (ubitboard_t)castle);

   /* Elephants are restricted to their own side of the board */
   limit_piece_movement_region(game, "E", (ubitboard_t)south);
   limit_piece_movement_region(game, "e", (ubitboard_t)north);

   /* Pawns can only move sideways after crossing the river */
   limit_piece_movement_region(game, "P", (ubitboard_t)(north | pawnf));
   limit_piece_movement_region(game, "p", (ubitboard_t)(south | pawnf));

   set_piece_value(game, "R", 950);
   set_piece_value(game, "C", 450);
   set_piece_value(game, "H", 400);
   set_piece_value(game, "A", 150);
   set_piece_value(game, "E", 150);
   set_piece_value(game, "e", 150);
   set_piece_value(game, "P", 175);
   set_piece_value(game, "p", 175);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   initialise_super_tables();

   /* Kings may not face eachother */
   add_rule_flag(game, RF_KING_TABOO);

   /* Stale mate is a win */
   game->stale_score = CHECKMATE;
   game->rep_score   = -ILLEGAL;

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rheakaehr/9/1c5c1/p1p1p1p1p/9/9/P1P1P1P1P/1C5C1/9/RHEAKAEHR w - - 0 1");
   game->name = strdup("Chinese Chess (XiangQi)");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_amazon_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags, knight;
   knight = flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
   flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Bishop", "B,b", "B");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)") | knight;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Amazon", "A,a", "A");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL|PF_CASTLE, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)board_rank8, "ARBN", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)board_rank1, "ARBN", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   flags = define_stepper("step 2N");
   add_special_move(game, "P", (ubitboard_t)board_rank2, flags);
   flags = define_stepper("step 2S");
   add_special_move(game, "p", (ubitboard_t)board_rank7, flags);

   set_piece_value(game, "A",1200);
   set_piece_value(game, "R", 500);
   set_piece_value(game, "B", 325);
   set_piece_value(game, "N", 320);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnbakbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBAKBNR w KQkq -");
   game->xb_setup  = strdup("(PNBR..............A..Kpnbr..............a..k)");
   game->name = strdup("Amazon chess");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_capablanca_game(void)
{
   game_t *game = create_game();

   /* This is played on a large board */
   set_board_size(game, 10, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags, knight, bishop, rook;
   knight = flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Knight", "N,n", "N");
   bishop = flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Bishop", "B,b", "B");
   rook = flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Queen", "Q,q", "Q");
   flags = knight | bishop;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Archbishop", "A,a", "A");
   flags = knight | rook;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Chancellor", "C,c", "C");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL|PF_CASTLE, (ubitboard_t)large_board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)large_board_rank[7], "QACRBN", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)large_board_rank[0], "QACRBN", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   flags = define_stepper("step 2N");
   add_special_move(game, "P", (ubitboard_t)large_board_rank[1], flags);
   flags = define_stepper("step 2S");
   add_special_move(game, "p", (ubitboard_t)large_board_rank[6], flags);

   set_piece_value(game, "Q", 950);
   set_piece_value(game, "C", 875);
   set_piece_value(game, "A", 825);
   set_piece_value(game, "R", 475);
   set_piece_value(game, "B", 350);
   set_piece_value(game, "N", 275);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* 260:    A = 850   (2106)
    * 260M:   A = 825   (2115)
    * 260M2:  A = 800   (2104)
    * 263M3:  A = 700   (2076)
    * 263M4:  A = 875   (2099)
    *
    * 260M:   N = 300   (2100)
    * 263M5:  N = 325   (2095)
    * 263M6:  N = 275   (2105)
    */

   /* Set up castling flags */
   short_castle_king_dest[WHITE] = 8;
   short_castle_king_dest[BLACK] = 8 + 70;
   long_castle_king_dest[WHITE]  = 2;
   long_castle_king_dest[BLACK]  = 2 + 70;

   large_short_castle_mask[WHITE] = large_square_bitboards[ 5] | large_square_bitboards[ 9];
   large_short_castle_mask[BLACK] = large_square_bitboards[75] | large_square_bitboards[79];
   large_short_castle_free[WHITE] = large_square_bitboards[ 6] | large_square_bitboards[ 7] |
                                    large_square_bitboards[ 8];
   large_short_castle_free[BLACK] = large_square_bitboards[76] | large_square_bitboards[77] |
                                    large_square_bitboards[78];
   large_short_castle_safe[WHITE] = large_square_bitboards[ 5] | large_square_bitboards[ 6] |
                                    large_square_bitboards[ 7] | large_square_bitboards[ 8];
   large_short_castle_safe[BLACK] = large_square_bitboards[75] | large_square_bitboards[76] |
                                    large_square_bitboards[77] | large_square_bitboards[78];

   large_long_castle_mask[WHITE]  = large_square_bitboards[ 5] | large_square_bitboards[ 0];
   large_long_castle_mask[BLACK]  = large_square_bitboards[75] | large_square_bitboards[70];
   large_long_castle_free[WHITE]  = large_square_bitboards[ 4] | large_square_bitboards[ 3] |
                                    large_square_bitboards[ 2] | large_square_bitboards[ 1];
   large_long_castle_free[BLACK]  = large_square_bitboards[74] | large_square_bitboards[73] |
                                    large_square_bitboards[72] | large_square_bitboards[71];
   large_long_castle_safe[WHITE]  = large_square_bitboards[ 5] | large_square_bitboards[ 4] |
                                    large_square_bitboards[ 3] | large_square_bitboards[ 2];
   large_long_castle_safe[BLACK]  = large_square_bitboards[75] | large_square_bitboards[74] |
                                    large_square_bitboards[73] | large_square_bitboards[72];

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnabqkbcnr/pppppppppp/10/10/10/10/PPPPPPPPPP/RNABQKBCNR w KQkq -");
   game->name = strdup("Capablanca chess");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_gothic_game(void)
{
   game_t *game = create_capablanca_game();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnbqckabnr/pppppppppp/10/10/10/10/PPPPPPPPPP/RNBQCKABNR w KQkq -");
   game->name = strdup("Capablanca chess (Gothic)");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}


game_t *create_embassy_game(void)
{
   game_t *game = create_capablanca_game();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnbqkcabnr/pppppppppp/10/10/10/10/PPPPPPPPPP/RNBQKCABNR w KQkq -");
   game->name = strdup("Capablanca chess (Embassy)");

   deduce_castle_flags(game->large_board, WHITE,  4,  1,  0);
   deduce_castle_flags(game->large_board, WHITE,  4,  7,  9);
   deduce_castle_flags(game->large_board, BLACK, 74, 71, 70);
   deduce_castle_flags(game->large_board, BLACK, 74, 77, 79);

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_janus_game(void)
{
   game_t *game = create_game();

   /* This is played on a large board */
   set_board_size(game, 10, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags, knight, bishop, rook;
   knight = flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Knight", "N,n", "N");
   bishop = flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Bishop", "B,b", "B");
   rook = flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Queen", "Q,q", "Q");
   flags = knight | bishop;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Janus", "J,j", "J");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL|PF_CASTLE, (ubitboard_t)large_board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)large_board_rank[7], "QJRBN", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)large_board_rank[0], "QJRBN", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   flags = define_stepper("step 2N");
   add_special_move(game, "P", (ubitboard_t)large_board_rank[1], flags);
   flags = define_stepper("step 2S");
   add_special_move(game, "p", (ubitboard_t)large_board_rank[6], flags);

   set_piece_value(game, "Q", 950);
   set_piece_value(game, "J", 850);
   set_piece_value(game, "R", 475);
   set_piece_value(game, "B", 350);
   set_piece_value(game, "N", 300);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Set up castling flags */
   short_castle_king_dest[WHITE] = 8;
   short_castle_king_dest[BLACK] = 8 + 70;
   long_castle_king_dest[WHITE]  = 1;
   long_castle_king_dest[BLACK]  = 1 + 70;

   large_short_castle_mask[WHITE] = large_square_bitboards[ 4] | large_square_bitboards[ 9];
   large_short_castle_mask[BLACK] = large_square_bitboards[74] | large_square_bitboards[79];
   large_short_castle_free[WHITE] = large_square_bitboards[ 5] | large_square_bitboards[ 6] |
                                    large_square_bitboards[ 7] | large_square_bitboards[ 8];
   large_short_castle_free[BLACK] = large_square_bitboards[75] | large_square_bitboards[76] |
                                    large_square_bitboards[77] | large_square_bitboards[78];
   large_short_castle_safe[WHITE] = large_square_bitboards[ 4] | large_square_bitboards[ 5] |
                                    large_square_bitboards[ 6] | large_square_bitboards[ 7] |
                                    large_square_bitboards[ 8];
   large_short_castle_safe[BLACK] = large_square_bitboards[74] | large_square_bitboards[75] |
                                    large_square_bitboards[76] | large_square_bitboards[77] |
                                    large_square_bitboards[78];

   large_long_castle_mask[WHITE]  = large_square_bitboards[ 4] | large_square_bitboards[ 0];
   large_long_castle_mask[BLACK]  = large_square_bitboards[74] | large_square_bitboards[70];
   large_long_castle_free[WHITE]  = large_square_bitboards[ 3] | large_square_bitboards[ 2] |
                                    large_square_bitboards[ 1];
   large_long_castle_free[BLACK]  = large_square_bitboards[73] | large_square_bitboards[72] |
                                    large_square_bitboards[71];
   large_long_castle_safe[WHITE]  = large_square_bitboards[ 4] | large_square_bitboards[ 3] |
                                    large_square_bitboards[ 2];
   large_long_castle_safe[BLACK]  = large_square_bitboards[74] | large_square_bitboards[73] |
                                    large_square_bitboards[72];

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rjnbkqbnjr/pppppppppp/10/10/10/10/PPPPPPPPPP/RJNBKQBNJR w KQkq -");
   game->name = strdup("Janus chess");

   /* Non-standard notation: king side castling is O-O-O, queen side is O-O */
   kingside_castle  = "O-O-O";
   queenside_castle = "O-O";

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_grand_game(void)
{
   game_t *game = create_game();

   /* This is played on a large board */
   set_board_size(game, 10, 10);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   large_bitboard_t white_promo = large_board_rank[9] | large_board_rank[8] | large_board_rank[7];
   large_bitboard_t black_promo = large_board_rank[0] | large_board_rank[1] | large_board_rank[2];

   /* Load piece and rule descriptions */
   uint32_t flags, cflags, knight, bishop, rook;
   knight = flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Knight", "N,n", "N");
   bishop = flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Bishop", "B,b", "B");
   rook = flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Queen", "Q,q", "Q");
   flags = knight | bishop;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Cardinal", "C,c", "C");
   flags = knight | rook;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Marshal", "M,m", "M");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL, (ubitboard_t)large_board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)white_promo, "QMCRBN", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)black_promo, "QMCRBN", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   flags = define_stepper("step 2N");
   add_special_move(game, "P", (ubitboard_t)large_board_rank[2], flags);
   flags = define_stepper("step 2S");
   add_special_move(game, "p", (ubitboard_t)large_board_rank[7], flags);

   /* We can never have more of a piece type than what we start with */
   set_maximum_number_of_pieces(game, WHITE, "Q", 1);
   set_maximum_number_of_pieces(game, WHITE, "M", 1);
   set_maximum_number_of_pieces(game, WHITE, "C", 1);
   set_maximum_number_of_pieces(game, WHITE, "R", 2);
   set_maximum_number_of_pieces(game, WHITE, "N", 2);
   set_maximum_number_of_pieces(game, WHITE, "B", 2);
   set_maximum_number_of_pieces(game, BLACK, "Q", 1);
   set_maximum_number_of_pieces(game, BLACK, "M", 1);
   set_maximum_number_of_pieces(game, BLACK, "C", 1);
   set_maximum_number_of_pieces(game, BLACK, "R", 2);
   set_maximum_number_of_pieces(game, BLACK, "N", 2);
   set_maximum_number_of_pieces(game, BLACK, "B", 2);

   /* Piece values, as for Capablanca */
   set_piece_value(game, "Q", 950);
   set_piece_value(game, "M", 875);
   set_piece_value(game, "C", 825);
   set_piece_value(game, "R", 475);
   set_piece_value(game, "B", 350);
   set_piece_value(game, "N", 275);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("r8r/1nbqkmcbn1/pppppppppp/10/10/10/10/PPPPPPPPPP/1NBQKMCBN1/R8R w - -");
   //XBoard: r8r/1nbqkcabn1/pppppppppp/10/10/10/10/PPPPPPPPPP/1NBQKCABN1/R8R[-] w - 0 1
   game->xb_setup = strdup("(PNBRQ..C........M....Kpnbrq..c........m....k)");
   game->name = strdup("Grand chess");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

/* Turkish (or Indian) Grant Chess, see http://www.chessvariants.org/historic.dir/indiangr1.html */
game_t *create_indiangrand_game(void)
{
   game_t *game = create_game();

   /* This is played on a large board */
   set_board_size(game, 10, 10);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   large_bitboard_t white_promo = large_board_rank[9] | large_board_rank[8] | large_board_rank[7];
   large_bitboard_t black_promo = large_board_rank[0] | large_board_rank[1] | large_board_rank[2];

   /* Load piece and rule descriptions */
   uint32_t flags, cflags, knight, bishop, rook;
   knight = flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Knight", "N,n", "N");
   bishop = flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Bishop", "B,b", "B");
   rook = flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Queen", "Q,q", "Q");
   flags = define_slider("slide (H,V,D,A)") | knight;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Amazon", "A,a", "A");
   flags = knight | bishop;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Cardinal", "C,c", "C");
   flags = knight | rook;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Marshal", "M,m", "M");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL|PF_CASTLE, (ubitboard_t)large_board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)white_promo, "Q", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)black_promo, "Q", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   //flags = define_stepper("step 2N");
   //add_special_move(game, "P", (ubitboard_t)large_board_rank[2], flags);
   //flags = define_stepper("step 2S");
   //add_special_move(game, "p", (ubitboard_t)large_board_rank[7], flags);

   /* Piece values, as for Capablanca */
   set_piece_value(game, "A",1150);
   set_piece_value(game, "Q", 950);
   set_piece_value(game, "M", 875);
   set_piece_value(game, "C", 825);
   set_piece_value(game, "R", 475);
   set_piece_value(game, "B", 350);
   set_piece_value(game, "N", 275);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnbqkacbnr/ppppmmpppp/4pp4/10/10/10/10/4PP4/PPPPMMPPPP/RNBCAKQBNR w - -");
   //XBoard: r8r/1nbqkcabn1/pppppppppp/10/10/10/10/PPPPPPPPPP/1NBQKCABN1/R8R[-] w - 0 1
   game->xb_setup = strdup("(PNBRQ..C........M.A..Kpnbrq..c........m.a..k)");

   game->name = strdup("Indian Grand chess");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_shatranj_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags;
   flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
   flags = define_leaper("leap (2,2)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Elephant", "E,e", "E");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Rook", "R,r", "R");
   flags = define_leaper("leap (1,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Ferz", "F,f", "F");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank8, "F", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank1, "F", "Pawn", ",p", " ");

   set_piece_value(game, "F", 150);
   set_piece_value(game, "R", 500);
   set_piece_value(game, "E", 100);
   set_piece_value(game, "N", 350);
   set_piece_value(game, "P",  80);
   set_piece_value(game, "p",  80);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnekfenr/pppppppp/8/8/8/8/PPPPPPPP/RNEKFENR w - -");

   /* Stale mate is a loss */
   game->stale_score = -CHECKMATE;

   /* A bare king is a loss (except if we can bare the enemy king on the next move, but the search takes care
    * of that).
    */
   game->bare_king_score = -CHECKMATE;
   game->name = strdup("Shatranj");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_xboard_shatranj_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags;
   flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
   flags = define_leaper("leap (2,2)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Elephant", "B,b", "B");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Rook", "R,r", "R");
   flags = define_leaper("leap (1,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Ferz", "Q,q", "Q");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank8, "Q", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank1, "Q", "Pawn", ",p", " ");

   set_piece_value(game, "Q", 150);
   set_piece_value(game, "R", 500);
   set_piece_value(game, "B", 100);
   set_piece_value(game, "N", 350);
   set_piece_value(game, "P",  80);
   set_piece_value(game, "p",  80);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnbkqbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBKQBNR w - -");

   /* Stale mate is a loss */
   game->stale_score = -CHECKMATE;

   /* A bare king is a loss (except if we can bare the enemy king on the next move, but the search takes care
    * of that).
    */
   game->bare_king_score = -CHECKMATE;
   game->name = strdup("Shatranj (XB)");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_makruk_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions
    * FIXME: Because of a disagreement with XBoard about the name of different pieces, the Elephants are
    * defined twice (which XBoard insists are "silver generals") and so is the Ferz (which XBoard calls
    * "met" as is its proper name for this game). This is a bit of a hack, but at least both work this way.
    */
   uint32_t flags, cflags;
   flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
#if 0
   flags = define_stepper("step N,NW,NE,SW,SE");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Elephant", "E", "E");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Elephant", "S", "E");
   flags = define_stepper("step S,NW,NE,SW,SE");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Elephant", ",e", "E");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Elephant", ",s", "E");
#else
   flags = define_leaper("aleap (0,1)|(1,1)|(1,-1)|(-1,-1)|(-1,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Silver general", "S,s", "S");
#endif
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Boat", "R,r", "R");
   flags = define_leaper("leap (1,1)");
   //add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Ferz", "F,f", "F");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Met", "M,m", "M");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank6, "M", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank3, "M", "Pawn", ",p", " ");

   //set_piece_value(game, "F", 150);
   set_piece_value(game, "M", 150);
   set_piece_value(game, "R", 500);
   set_piece_value(game, "S", 250);
   //set_piece_value(game, "s", 200);
   //set_piece_value(game, "E", 200);
   //set_piece_value(game, "e", 200);
   set_piece_value(game, "N", 320);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnsmksnr/8/pppppppp/8/8/PPPPPPPP/8/RNSKMSNR w - -");
   //game->start_fen = "rnefkenr/8/pppppppp/8/8/PPPPPPPP/8/RNEKFENR w - -";
   //game->xb_setup  = "(PN.R.ME............E.Kpn.r.me............e.k)";
   game->name = strdup("Makruk");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_burmese_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags;
   flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
   flags = define_leaper("aleap (0,1)|(1,1)|(1,-1)|(-1,-1)|(-1,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Elephant", "E,e", "E");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Rook", "R,r", "R");
   flags = define_leaper("leap (1,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Ferz", "F,f", "F");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)(A8_MASK|B7_MASK|C6_MASK|D5_MASK|E5_MASK|F6_MASK|G7_MASK|H8_MASK), "", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)(A1_MASK|B2_MASK|C3_MASK|D4_MASK|E4_MASK|F3_MASK|G2_MASK|H1_MASK), "", "Pawn", ",p", " ");

   set_piece_value(game, "F", 150);
   set_piece_value(game, "R", 500);
   set_piece_value(game, "E", 250);
   set_piece_value(game, "e", 250);
   set_piece_value(game, "N", 320);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);

   /* Setup the correct drop zones for both players and for all the pieces */
   bitboard_t drop_zone = board_all ^ (board_rank1 | board_rank8);
   set_piece_drop_zone(game, "K", (ubitboard_t)drop_zone);
   set_piece_drop_zone(game, "F", (ubitboard_t)drop_zone);
   set_piece_drop_zone(game, "R", (ubitboard_t)~drop_zone);
   set_piece_drop_zone(game, "E", (ubitboard_t)drop_zone);
   set_piece_drop_zone(game, "e", (ubitboard_t)drop_zone);
   set_piece_drop_zone(game, "N", (ubitboard_t)drop_zone);
   set_player_drop_zone(game, WHITE, (ubitboard_t)(board_rank1 | board_rank2 | board_rank3));
   set_player_drop_zone(game, BLACK, (ubitboard_t)(board_rank6 | board_rank7 | board_rank8));
   game->board.use_holdings = true;

   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("8/8/4pppp/pppp4/4PPPP/PPPP4/8/8[KFRREENNkfrreenn] w - - 0 1");
   game->xb_setup  = strdup("(PN.R.FE..............Kpn.r.fe..............k)");
   //game->start_fen = "4r1r1/2knen2/2nfpppp/pppp4/4PPPP/PPPPFNE1/3E1NK1/2R1R3 w - -";

   /* It is illegal to give stale mate */
   game->stale_score = CHECKMATE;

   /* Drops are forced if they're possible */
   add_rule_flag(game, RF_FORCE_DROPS);

   /* FIXME: promotions are OPTIONAL while in the promotion zone, and apply only when the orginal Ferz has
    * been taken. This needs to be implemented.
    * FIXME: for the first part of the game, the players must place their pieces on the board first. The above
    * starting position is just one possibility.
    */
   game->name = strdup("Burmese chess");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_spartan_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags, knight;

   /* The Persian army (standard chess pieces) */
   knight = flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
   flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Bishop", "B,b", "B");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Queen", "Q,q", "Q");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL|PF_CASTLE, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank8, "QRBN", "Pawn", "P", " ");
   set_piece_value(game, "Q", 975);
   set_piece_value(game, "R", 500);
   set_piece_value(game, "B", 325);
   set_piece_value(game, "N", 320);
   set_piece_value(game, "P", 100);

   /* The Spartan army */
   int ferz = define_leaper("leap (1,1)");
   flags = define_slider("slide (H,V)") | ferz;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "General", "G,g", "G");
   flags = define_slider("slide (D,A)") | knight;
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Warlord", "W,w", "W");
   flags = define_leaper("leap (1,0)|leap (2,0)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Captain", "C,c", "C");
   cflags = define_leaper("leap (1,1)|(2,2)");
   flags = cflags | define_stepper("step W,E");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_empty, "", "Lieutenant", "L,l", "L");
   flags = define_stepper("step SE, SW");
   cflags = define_stepper("step S");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank1, "KGWCL", "Hoplite", ",h", " ");
   set_piece_value(game, "G", 725);
   set_piece_value(game, "W", 825);
   set_piece_value(game, "C", 300);
   set_piece_value(game, "L", 310);
   set_piece_value(game, "h", 100);
   set_piece_value(game, "K", 475);


   /* Add special move types for pawns on their initial squares */
   flags = define_stepper("step 2N");
   add_special_move(game, "P", (ubitboard_t)board_rank2, flags);
   //flags = define_stepper("step 2SE, 2SW");
   flags = define_leaper("leap (2,2)");      /* Ok, since we can't leap two squares back from the 7th rank */
   add_special_move(game, "h", (ubitboard_t)board_rank7, flags);

   /* Spartans can have two kings */
   set_maximum_number_of_kings(game, BLACK, 2);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Override the default heuristic for the Hoplites; we don't care so much about pawn shields. */
   set_evaluation_terms(game, "h", 2, 0, 2);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);

   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("lgkcckwl/hhhhhhhh/8/8/8/8/PPPPPPPP/RNBQKBNR w KQ -");
   game->name = strdup("Spartan chess");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_maharaja_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags;
   flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
   flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Bishop", "B,b", "B");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Queen", "Q,q", "Q");
   flags = define_slider("slide (H,V,D,A)") | define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, PF_ROYAL, (ubitboard_t)board_empty, "", "Maharaja", "M,m", "M");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL|PF_CASTLE, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank8, "", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank1, "", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   flags = define_stepper("step 2N");
   add_special_move(game, "P", (ubitboard_t)board_rank2, flags);
   flags = define_stepper("step 2S");
   add_special_move(game, "p", (ubitboard_t)board_rank7, flags);

   set_piece_value(game, "M", 1200);
   set_piece_value(game, "Q", 950);
   set_piece_value(game, "R", 500);
   set_piece_value(game, "B", 325);
   set_piece_value(game, "N", 320);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnbqkbnr/pppppppp/8/8/8/8/8/4M3/ w KQkq -");
   game->xb_setup  = strdup("(.....................Mpnbrq................k)");
   game->name = strdup("Maharaja and the Sepoys");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_knightmate_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags;
   flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, PF_ROYAL|PF_CASTLE, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
   flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Bishop", "B,b", "B");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Queen", "Q,q", "Q");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)board_rank8, "QRBK", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, PF_SET_EP|PF_TAKE_EP, (ubitboard_t)board_rank1, "QRBK", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   flags = define_stepper("step 2N");
   add_special_move(game, "P", (ubitboard_t)board_rank2, flags);
   flags = define_stepper("step 2S");
   add_special_move(game, "p", (ubitboard_t)board_rank7, flags);

   set_piece_value(game, "Q", 950);
   set_piece_value(game, "R", 500);
   set_piece_value(game, "B", 325);
   set_piece_value(game, "N", 320);
   set_piece_value(game, "K", 220);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);

   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rkbqnbkr/pppppppp/8/8/8/8/PPPPPPPP/RKBQNBKR/ w KQkq -");
   game->name = strdup("Knightmate");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_losers_game(void)
{
   game_t *game = create_game();

   /* Play on a normal 8x8 board */
   set_board_size(game, 8, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags;
   flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Knight", "N,n", "N");
   flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Bishop", "B,b", "B");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Rook", "R,r", "R");
   flags = define_slider("slide (H,V,D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Queen", "Q,q", "Q");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "King", "K,k", "K");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank8, "QRBNK", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)board_rank1, "QRBNK", "Pawn", ",p", " ");

   /* Add special move types for pawns on their initial squares */
   flags = define_stepper("step 2N");
   add_special_move(game, "P", (ubitboard_t)board_rank2, flags);
   flags = define_stepper("step 2S");
   add_special_move(game, "p", (ubitboard_t)board_rank7, flags);

   set_piece_value(game, "Q", -950);
   set_piece_value(game, "R", -500);
   set_piece_value(game, "B", -325);
   set_piece_value(game, "N", -320);
   set_piece_value(game, "P", -100);
   set_piece_value(game, "p", -100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq -");
   game->name = strdup("Losers chess");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

game_t *create_courier_game(void)
{
   game_t *game = create_game();

   /* This is played on a large board */
   set_board_size(game, 12, 8);

   /* Unload any previous piece and rule descriptions */
   reset_piece_definitions();

   /* Load piece and rule descriptions */
   uint32_t flags, cflags;
   flags = define_leaper("leap (2,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Knight", "N,n", "N");
   flags = define_slider("slide (D,A)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Bishop", "B,b", "B");
   flags = define_slider("slide (H,V)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Rook", "R,r", "R");
   flags = define_leaper("leap (1,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Ferz", "F,f", "F");
   flags = define_leaper("leap (2,2)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)board_empty, "", "Elephant", "E,e", "E");
   flags = define_leaper("leap (0,1)|(1,1)");
   add_piece_type(game, flags, flags, PF_ROYAL, (ubitboard_t)large_board_empty, "", "King", "K,k", "K");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Man", "M,m", "M");
   flags = define_leaper("leap (0,1)");
   add_piece_type(game, flags, flags, 0, (ubitboard_t)large_board_empty, "", "Wazir", "W,w", "W");
   flags = define_stepper("step N");
   cflags = define_stepper("step NE, NW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)large_board_rank[7], "F", "Pawn", "P", " ");
   flags = define_stepper("step S");
   cflags = define_stepper("step SE, SW");
   add_piece_type(game, flags, cflags, 0, (ubitboard_t)large_board_rank[0], "F", "Pawn", ",p", " ");

   set_piece_value(game, "R", 475);
   set_piece_value(game, "B", 350);
   set_piece_value(game, "M", 300);
   set_piece_value(game, "N", 200);
   set_piece_value(game, "W", 125);
   set_piece_value(game, "E", 125);
   set_piece_value(game, "F", 125);
   set_piece_value(game, "P", 100);
   set_piece_value(game, "p", 100);

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);


   /* Now all piece types have been added, define the attack tables for a "super piece" */
   /* FIXME: for games where both sides have different pieces, a different super piece could be constructed
    * for each side. This would be a small optimisation.
    */
   initialise_super_tables();

   /* Set the FEN string for the starting position */
   game->start_fen = strdup("rnebmk1wbenr/1ppppp1pppp1/6f5/p5p4p/P5P4P/6F5/1PPPPP1PPPP1/RNEBMK1WBENR w - -");
   game->name = strdup("Courier chess");

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}



typedef struct file_piece_description_t {
   ubitboard_t special_zone;
   ubitboard_t promotion_zone;
   char *name;
   char *abbr;
   char *symbol;
   char *promotion;
   move_flag_t move, capture, special;
   uint32_t flags;
   int value;
   bool set_capture;
} file_piece_description_t;

typedef struct board_zone_description_t {
   char *name;
   ubitboard_t zone;
} board_zone_description_t;

static void add_piece_to_game(game_t *game, file_piece_description_t *pd)
{
   uint32_t cflags = pd->move;
   if (pd->set_capture) cflags = pd->capture;
   add_piece_type(game, pd->move, cflags, pd->flags, pd->promotion_zone, pd->promotion, pd->name, pd->symbol, pd->abbr);

   char ss[4];
   char *s = ss;
   int n;
   for (n=0; n<5; n++)
      ss[n] = pd->symbol[n];
   if (s[0] == ',') s++;
   s[1] = '\0';
   if (pd->special)
      add_special_move(game, s, pd->special_zone, pd->special);

   set_piece_value(game, s, pd->value);
   free(pd->name);
   free(pd->abbr);
   free(pd->symbol);
   free(pd->promotion);
   pd->name = NULL;
   pd->abbr = NULL;
   pd->symbol = NULL;
   pd->promotion = NULL;
}

game_t *create_game_from_file(const char *filename, const char *variant_name)
{
   board_zone_description_t zone[128];
   file_piece_description_t *pd = NULL;
   game_t *game = NULL;
   int num_zones = 0;
   FILE *f;

   f = fopen(filename, "r");
   if (!f) return game;
   char line[4096];

   while (!feof(f)) {
      fgets(line, sizeof line, f);
      /* Strip away comments */
      char *s = strstr(line, "#");
      if (s) s[0] = '\0';

      /* Snip end-of-line */
      s = strstr(line, "\n");
      if (s) s[0] = '\0';

      /* Strip trailing space */
      s = line+strlen(line)-1;
      while (s > line && isspace(s[0])) { s[0] = '\0'; s--; }

      /* New variant */
      if (strstr(line, "Variant:") == line) {
         if (game) break;                          /* We're done loading the variant we're looking for */
         if (strstr(line+8, variant_name)) {       /* We've found the variant we're looking for */
            game = create_game();
            game->name = strdup(variant_name);
         }
         continue;
      }

      /* Have we found the correct variant? */
      if (!game) continue;

      /* Empty line, terminates a piece description */
      if (line[0] == '\0' && pd) {
            add_piece_to_game(game, pd);
            free(pd);
            pd = NULL;
      }

      /* Set board size */
      if (strstr(line, "Board:") == line) {
         s = line+6;
         int files, ranks;
         sscanf(s, "%dx%d", &files, &ranks);

         /* Check whether the input is valid */
         if (files > 16 || ranks > 16 || (files < 8 && ranks < 8)) {
            end_game(game);
            fclose(f);
            return NULL;
         }

         /* Set the requested board size */
         set_board_size(game, files, ranks);

         /* Unload any previous piece and rule descriptions */
         reset_piece_definitions();
         continue;
      }

      /* Starting position (FEN) */
      if (strstr(line, "FEN:") == line) {
         char *s = line+4;
         s = strstr(s, "\"");
         if (!s) continue;
         s++;
         char *eof = strstr(s, "\"");
         if (!eof) continue;
         eof[0] = '\0';
         game->start_fen = strdup(s);
         if (game->xb_setup == NULL) game->xb_setup = strdup("(PNBRQFEACWMOHIJGDVLSUKpnbrqfeacwmohijgdvlsuk)");

         continue;
      }

      /* Piece description for XBoard */
      if ((strstr(line, "XBoard pieces:") == line) || (strstr(line, "WinBoard pieces:") == line)) {
         char *s = line+14;
         s = strstr(s, "\"");
         if (!s) continue;
         s++;
         char *eof = strstr(s, "\"");
         if (!eof) continue;
         eof[0] = '\0';
         
         free(game->xb_setup);
         game->xb_setup = malloc(50);
         snprintf(game->xb_setup, 50, "(%s)", s);
         
         continue;
      }

      /* Define a region of the board */
      if (strstr(line, "Zone:") == line) {
         char *name = line+5;
         while(isspace(*name)) name++;
         char *s = strstr(name, "=");
         char *sq;
         if (!s) continue;
         sq = s+1;
         s[0] = '\0'; s--;
         while(isspace(*s)) s--;
         s[1] = '\0';
         zone[num_zones].name = strdup(name);
         if (game->large_board) {
            s = strtok(sq, ",");
            while(isspace(*s)) s++;
            char file;
            int rank;
            sscanf(s, "%c%d", &file, &rank);
            file -= 'a';
            rank--;
            zone[num_zones].zone.lbb = large_square_bitboards[pack_row_file(rank, file)];
            while (s = strtok(NULL, ",")) {
               sscanf(s, "%c%d", &file, &rank);
               file -= 'a';
               rank--;
               zone[num_zones].zone.lbb |= large_square_bitboards[pack_row_file(rank, file)];
            }
            num_zones++;
         } else {
            s = strtok(sq, ",");
            while(isspace(*s)) s++;
            char file;
            int rank;
            sscanf(s, "%c%d", &file, &rank);
            file -= 'a';
            rank--;
            zone[num_zones].zone.bb = square_bitboards[pack_row_file(rank, file)];
            while (s = strtok(NULL, ",")) {
               sscanf(s, "%c%d", &file, &rank);
               file -= 'a';
               rank--;
               zone[num_zones].zone.bb |= square_bitboards[pack_row_file(rank, file)];
            }
            num_zones++;
         }
         continue;
      }

      /* Exclude a region on the board (for the purpose of move generation) */
      if (strstr(line, "Exclude:") == line) {
         char *s = line+8;
         while(isspace(*s)) s++;
         char *sq = s;
         char file;
         int rank;

         s = strtok(sq, ",");
         while(isspace(*s)) s++;
         sscanf(s, "%c%d", &file, &rank);
         file -= 'a';
         rank--;
         remove_square(game, pack_row_file(rank, file));
         while (s = strtok(NULL, ",")) {
            sscanf(s, "%c%d", &file, &rank);
            file -= 'a';
            rank--;
            remove_square(game, pack_row_file(rank, file));
         }
         continue;
      }

      /* Capture-the-flag, white flag (for black to capture) */
      if (strstr(line, "WhiteFlag:") == line) {
         char *s = line+10;
         while(isspace(*s)) s++;
         char *sq = s;
         char file;
         int rank;

         s = strtok(sq, ",");
         sscanf(sq, "%c%d", &file, &rank);
         file -= 'a';
         rank--;
         place_flag(game, BLACK, pack_row_file(rank, file));
         while (s = strtok(NULL, ",")) {
            sscanf(s, "%c%d", &file, &rank);
            file -= 'a';
            rank--;
            place_flag(game, BLACK, pack_row_file(rank, file));
         }
         continue;
      }

      /* Capture-the-flag, black flag (for white to capture) */
      if (strstr(line, "BlackFlag:") == line) {
         char *s = line+10;
         while(isspace(*s)) s++;
         char *sq = s;
         char file;
         int rank;

         s = strtok(sq, ",");
         sscanf(sq, "%c%d", &file, &rank);
         file -= 'a';
         rank--;
         place_flag(game, WHITE, pack_row_file(rank, file));
         while (s = strtok(NULL, ",")) {
            sscanf(s, "%c%d", &file, &rank);
            file -= 'a';
            rank--;
            place_flag(game, WHITE, pack_row_file(rank, file));
         }
         continue;
      }

      /* Define a new piece */
      if (strstr(line, "Piece:") == line) {
         if (pd) {
            add_piece_to_game(game, pd);
            free(pd);
         }
         pd = calloc(1, sizeof *pd);
         s = line + 6;
         while(isspace(*s)) s++;
         pd->name = strdup(s);
         continue;
      }

      if (strstr(line, "Move:") == line && pd) {
         s = line + 5;
         while(*s && isspace(*s)) s++;
         pd->move |= define_piece_move(s);
         continue;
      }

      if (strstr(line, "Special:") == line && pd) {
         s = line + 8;
         while (isspace(*s)) s++;
         char *p = strstr(s, ",");
         if (!p) continue;
         p[0] = '\0'; p++;

         /* Find the corresponding zone */
         int n;
         for(n=0; n<num_zones; n++)
            if (strstr(s, zone[n].name) == s) {
               pd->special_zone = zone[n].zone;
               break;
            }

         pd->special |= define_piece_move(p);
         continue;
      }

      if (strstr(line, "Capture:") == line && pd) {
         s = line + 8;
         while(isspace(*s)) s++;
         pd->capture |= define_piece_move(s);
         pd->set_capture = true;
         continue;
      }

      if (strstr(line, "Castle:") == line && pd) {
         sides side = WHITE;
         int from, to, rfrom;
         char file;
         int rank;

         s = line + 7;
         while(isspace(*s)) s++;

         /* Field 1: side */
         if (strstr(s, "black") == s) side = BLACK;
         s += 5;

         /* From square */
         while(isspace(*s)) s++;
         sscanf(s, "%c%d", &file, &rank);
         file -= 'a';
         rank--;
         from = pack_row_file(rank, file);

         /* To square */
         s = strstr(s, "-");
         if (!s) continue;
         s++;
         while(isspace(*s)) s++;
         sscanf(s, "%c%d", &file, &rank);
         file -= 'a';
         rank--;
         to = pack_row_file(rank, file);

         /* Rook from-square */
         s = strstr(s, "with");
         if (!s) continue;
         s+=4;
         while(isspace(*s)) s++;
         sscanf(s, "%c%d", &file, &rank);
         file -= 'a';
         rank--;
         rfrom = pack_row_file(rank, file);

         /* Calculate castle masks */
         deduce_castle_flags(game->large_board, side, from, to, rfrom);

         continue;
      }

      if (strstr(line, "Symbol:") == line && pd) {
         s = line + 7;
         while (isspace(*s)) s++;
         char *p = strstr(s, "\"");
         if (!p) continue;
         p++;
         s = strstr(p, "\"");
         if (!s) continue;
         s++; s[-1] = '\0';
         pd->abbr = strdup(p);

         p = strstr(s, "\"");
         if (!p) continue;
         p++;
         s = strstr(p, "\"");
         if (!s) continue;
         s++; s[-1] = '\0';
         pd->symbol = strdup(p);
         continue;
      }

      if (strstr(line, "Promotion:") == line && pd) {
         s = line + 10;
         while (isspace(*s)) s++;
         char *p = strstr(s, ",");
         if (!p) continue;
         p[0] = '\0'; p++;

         /* Find the corresponding zone */
         int n;
         for(n=0; n<num_zones; n++)
            if (strstr(s, zone[n].name) == s) {
               pd->promotion_zone = zone[n].zone;
               break;
            }

         p = strstr(p, "\"");
         if(!p) continue;
         p++;
         s = strstr(p, "\"");
         if (s) s[0] = '\0';

         pd->promotion = strdup(p);
         continue;
      }

      if (strstr(line, "Flags:") == line && pd) {
         char *p;
         s = line + 6;
         while(isspace(*s)) s++;
         while(p = strtok(s, ",")) {
            s = NULL;
            while(isspace(*p)) p++;
            if (strstr(p, "royal") == p) {
               pd->flags |= PF_ROYAL;
            } else if (strstr(p, "castle") == p) {
               pd->flags |= PF_CASTLE;
            } else if (strstr(p, "set_ep") == p) {
               pd->flags |= PF_SET_EP;
            } else if (strstr(p, "take_ep") == p) {
               pd->flags |= PF_TAKE_EP;
            }
         }
         continue;
      }

      if (strstr(line, "Value:") == line && pd) {
         sscanf(line+6, "%d", &pd->value);
         continue;
      }

      if (strstr(line, "Rule:") == line) {
         s = line + 5;
         while(isspace(*s)) s++;
         if (strstr(s, "checkmate") == s) {
            if (strstr(s, "loss")) {
               game->mate_score = CHECKMATE;
            } else if (strstr(s, "win")) {
               game->mate_score = -CHECKMATE;
            } else if (strstr(s, "draw")) {
               game->mate_score = STALEMATE;
            } else if (strstr(s, "illegal")) {
               game->mate_score = ILLEGAL;
            }
         }
         if (strstr(s, "stalemate") == s) {
            if (strstr(s, "loss")) {
               game->stale_score = CHECKMATE;
            } else if (strstr(s, "win")) {
               game->stale_score = -CHECKMATE;
            } else if (strstr(s, "draw")) {
               game->stale_score = STALEMATE;
            } else if (strstr(s, "illegal")) {
               game->stale_score = ILLEGAL;
            }
         }
         if (strstr(s, "repeat3") == s) {
            if (strstr(s, "loss")) {
               game->rep_score = CHECKMATE;
            } else if (strstr(s, "win")) {
               game->rep_score = -CHECKMATE;
            } else if (strstr(s, "draw")) {
               game->rep_score = STALEMATE;
            } else if (strstr(s, "illegal")) {
               game->rep_score = ILLEGAL;
            }
         }
         if (strstr(s, "loneking") == s) {
            if (strstr(s, "loss")) {
               game->bare_king_score = -CHECKMATE;
            } else if (strstr(s, "win")) {
               game->bare_king_score = CHECKMATE;
            } else if (strstr(s, "draw")) {
               game->bare_king_score = STALEMATE;
            } else if (strstr(s, "illegal")) {
               game->bare_king_score = ILLEGAL;
            }
         }
         if (strstr(s, "nopieces") == s) {
            if (strstr(s, "loss")) {
               game->no_piece_score = CHECKMATE;
            } else if (strstr(s, "win")) {
               game->no_piece_score = -CHECKMATE;
            } else if (strstr(s, "draw")) {
               game->no_piece_score = STALEMATE;
            } else if (strstr(s, "illegal")) {
               game->no_piece_score = ILLEGAL;
            }
         }
         if (strstr(s, "captureanyflag") == s) {
            game->board.rule_flags |= RF_CAPTURE_ANY_FLAG;
            if (strstr(s, "loss")) {
               game->flag_score = CHECKMATE;
            } else if (strstr(s, "win")) {
               game->flag_score = -CHECKMATE;
            } else if (strstr(s, "draw")) {
               game->flag_score = STALEMATE;
            } else if (strstr(s, "illegal")) {
               game->flag_score = ILLEGAL;
            }
         }
         if (strstr(s, "captureallflags") == s) {
            game->board.rule_flags |= RF_CAPTURE_ALL_FLAG;
            if (strstr(s, "loss")) {
               game->flag_score = CHECKMATE;
            } else if (strstr(s, "win")) {
               game->flag_score = -CHECKMATE;
            } else if (strstr(s, "draw")) {
               game->flag_score = STALEMATE;
            } else if (strstr(s, "illegal")) {
               game->flag_score = ILLEGAL;
            }
         }

         if (strstr(s, "taboo")) game->board.rule_flags |= RF_KING_TABOO;
      }
   }
   fclose(f);

   /* Add the last remaining piece */
   if (pd) {
      add_piece_to_game(game, pd);
      free(pd);
   }

   /* Free memory */
   int n;
   for (n=0; n<num_zones; n++)
      free(zone[n].name);

   /* Finalise game description */

   /* Setup permutation array for piece values (least->most valuable) */
   sort_piece_values(game);

   /* Get piece mobility information */
   get_mobility_statistics(game);

   /* Initialise evaluation terms, based on heuristics */
   load_piece_evaluation_terms(game);

   /* Construct piece square tables for evaluation */
   initialise_evaluation(game);

   /* Define the attack tables for a "super piece" */
   initialise_super_tables();

   /* Determine the castle piece */
   identify_castle_partner(game);

   return game;
}

#define streq(s1, s2) (strcmp((s1), (s2)) == 0)

new_variant_game_t create_variant_from_name(const char * const name)
{
   if (streq(name, "normal")) {
      return create_standard_game;
   } else if (streq(name, "berolina")) {
      return create_berolina_game;
   } else if (streq(name, "pocketknight")) {
      return create_pocketknight_game;
   } else if (streq(name, "amazon")) {
      return create_amazon_game;
   } else if (streq(name, "spartan")) {
      return create_spartan_game;
   } else if (streq(name, "knightmate")) {
      return create_knightmate_game;
   } else if (streq(name, "maharaja")) {
      return create_maharaja_game;
   } else if (streq(name, "makruk")) {
      return create_makruk_game;
   } else if (streq(name, "burmese")) {
      return create_burmese_game;
   } else if (streq(name, "shatranj")) {
      return create_shatranj_game;
   } else if (streq(name, "capablanca")) {
      return create_capablanca_game;
   } else if (streq(name, "gothic")) {
      return create_gothic_game;
   } else if (streq(name, "embassy")) {
      return create_embassy_game;
   } else if (streq(name, "janus")) {
      return create_janus_game;
   } else if (streq(name, "courier")) {
      return create_courier_game;
   } else if (streq(name, "xiangqi")) {
      return create_chinese_game;
   } else if (streq(name, "grand")) {
      return create_grand_game;
   } else if (streq(name, "indiangrand")) {
      return create_indiangrand_game;
   }

   return NULL;
}
