/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef board_t_H
#define board_t_H

#include <stdbool.h>
#include "assert.h"
#include "bits.h"
#include "square.h"
#include "bitboard.h"
#include "squares.h"
#include "pieces.h"
#include "inline/hashkey.h"
#include "static_piece_tables.h"
#include "config.h"

typedef struct board_t {
   /* Bitboard representation of the chess board.
    *  Index 0 is for pawns
    *  Index 1 is for knights
    *  Index 2 is for bishops
    *  Index 3 is for rooks
    *  Index 4 is for queens
    *  Index 5 is for kings
    */
   bitboard_t bbp[6];

   /* Bitboard encoding initial positions, used in move generation for
    * castling moves.
    */
   bitboard_t init;

   /* Board occupancy bitboards
    *  Index 0 is for white pieces
    *  Index 1 is for black pieces 
    */
   bitboard_t bbc[2];

#ifdef USE_ROTATED_BITBOARDS
   /* Flipped occupancy bitboard (rows & files interchanged), used to find
    * file occupation numbers, which are used in move generation. We make
    * no distinction between black and white here (it's not needed)
    */
   bitboard_t bbf;

   /* Rotated occupancy bitboards (+/- 45 degrees), used to find diagonal
    * occupancy numbers, which are used in move generation. Again, we make
    * no distinction between black and white here (not needed)
    */
   bitboard_t bbr[2];
#endif

   /* Hash key for this board */
   uint64_t hash, pawn_hash;

   /* En-passant square */
   int8_t ep_square;

   /* Whether the current player is in check or not.
    * CAUTION: this is detected by the move generator, which cannot (should
    * not) modify the state of the board. That means this value need to be
    * set manually immediately AFTER the move generator is called if it is
    * to be used.
    */
   bool in_check;
   bool did_castle[2];

   int piece_count;
} board_t;

extern bool ansi_colours;
extern bool unicode_board;
extern void clear_board(board_t *board);

extern void print_bitboards_file(FILE *file, const board_t *board);
extern void print_bitboards(const board_t *board);
extern void print_board_file(FILE *file, const board_t *board);
extern void print_board(const board_t *board);

static inline bitboard_t get_occupied(const board_t *board)
{
   assert(board);
   return board->bbc[0] | board->bbc[1];
}

static inline bitboard_t get_rook_movers(const board_t *board)
{
   return board->bbp[ROOK] | board->bbp[QUEEN];
}

static inline bitboard_t get_bishop_movers(const board_t *board)
{
   return board->bbp[BISHOP] | board->bbp[QUEEN];
}


static inline bitboard_t get_normal_pieces(const board_t *board)
{
   return board->bbp[KNIGHT] | board->bbp[BISHOP] | board->bbp[ROOK] | board->bbp[QUEEN];
}

static inline uint8_t get_row_occupancy_number(const board_t *board, const int square)
{
   assert(board);

   return get_bitboard_rank_occupancy_number(get_occupied(board), square);
}

static inline uint8_t get_file_occupancy_number(const board_t *board, const int square)
{
   assert(board);
#ifdef USE_ROTATED_BITBOARDS
   return get_bitboard_row(board->bbf, unpack_file(square));
#else
   return get_bitboard_file_occupancy_number(get_occupied(board), square);
#endif
}

/* Get the occupancy number for the diagonal parallel to h1-a8 that passes
 * through the indicated square.
 * The number of bits used by the occupancy number are given by the length
 * of the diagonal, which is 1+(diag_nr&7) (diag_nr starting from 0).
 * Any extra high bits are *not* cleared, however. If this is considered
 * necessary, the caller should do this explicitly.
 */
static inline uint8_t get_diagonal_a8h1_occupancy_number(const board_t *board, const int square)
{
   assert(board);

#ifdef USE_ROTATED_BITBOARDS
   int shift = get_a8h1_diagonal_offset(square);
   return (board->bbr[0] >> shift) & 0xFF;
#else
   return get_bitboard_antidiag_occupancy_number(get_occupied(board), square);
#endif
}

/* Get the occupancy number for the diagonal parallel to a1-a8 that passes
 * through the indicated square.
 * The number of bits used by the occupancy number are given by the length
 * of the diagonal, which is 1+(diag_nr&7) (diag_nr starting from 0).
 * Any extra high bits are *not* cleared, however. If this is considered
 * necessary, the caller should do this explicitly.
 */
static inline uint8_t get_diagonal_a1h8_occupancy_number(const board_t *board, const int square)
{
   assert(board);

#ifdef USE_ROTATED_BITBOARDS
   int shift = get_a1h8_diagonal_offset(square);
   return (board->bbr[1] >> shift) & 0xFF;
#else
   return get_bitboard_diagonal_occupancy_number(get_occupied(board), square);
#endif
}

static inline uint8_t get_masked_row_occupancy_number(const board_t *board, bitboard_t mask, const int square)
{
   assert(board);

   return get_bitboard_rank_occupancy_number(get_occupied(board) & mask, square);
}

static inline uint8_t get_masked_file_occupancy_number(const board_t *board, bitboard_t mask, const int square)
{
   assert(board);
   return get_bitboard_file_occupancy_number(get_occupied(board) & mask, square);
}

/* Get the occupancy number for the diagonal parallel to h1-a8 that passes
 * through the indicated square.
 * The number of bits used by the occupancy number are given by the length
 * of the diagonal, which is 1+(diag_nr&7) (diag_nr starting from 0).
 * Any extra high bits are *not* cleared, however. If this is considered
 * necessary, the caller should do this explicitly.
 */
static inline uint8_t get_masked_diagonal_a8h1_occupancy_number(const board_t *board, bitboard_t mask, const int square)
{
   assert(board);

   return get_bitboard_antidiag_occupancy_number(get_occupied(board)&mask, square);
}

/* Get the occupancy number for the diagonal parallel to a1-a8 that passes
 * through the indicated square.
 * The number of bits used by the occupancy number are given by the length
 * of the diagonal, which is 1+(diag_nr&7) (diag_nr starting from 0).
 * Any extra high bits are *not* cleared, however. If this is considered
 * necessary, the caller should do this explicitly.
 */
static inline uint8_t get_masked_diagonal_a1h8_occupancy_number(const board_t *board, bitboard_t mask, const int square)
{
   assert(board);

   return get_bitboard_diagonal_occupancy_number(get_occupied(board)&mask, square);
}

static inline void get_occupancy_numbers(const board_t *board, int square, int *row, int *file)
{
   assert(board);
   assert(row);
   assert(file);

   *row = get_row_occupancy_number(board, square);
   *file = get_file_occupancy_number(board, square);
}

/* Functions used to manipulate a chess board - written as inlineble functions */
static inline uint8_t get_piece(const board_t *board, const int where)
{
   assert(where<=H8);
   assert(board);
   assert((board->bbc[0]|board->bbc[1]) & make_bitboard_square(where));
   bitboard_t bb_where = make_bitboard_square(where);
   int piece = ((board->bbc[1]>>where)&1)<<7;
#if 0
   bitboard_t bb = get_occupied(board) ^ board->bbp[KING];

   if (bb & bb_where) {
      bb ^= board->bbp[PAWN];
      if (bb & bb_where) {
         bb ^= board->bbp[ROOK] | board->bbp[QUEEN];
         if (bb & bb_where) {
            if (board->bbp[BISHOP])
               return piece | BISHOP;
            return piece | KNIGHT;
         }
         if (board->bbp[QUEEN])
            return piece | QUEEN;
         return piece | ROOK;
      }
      return piece | PAWN;
   }
#else
   if (board->bbp[PAWN] & bb_where)
      return piece | PAWN;
   if ((board->bbp[KNIGHT] | board->bbp[BISHOP]) & bb_where) {
      if (board->bbp[BISHOP] & bb_where)
         return piece | BISHOP;
      return piece | KNIGHT;
   }
   if ((board->bbp[ROOK] | board->bbp[QUEEN]) & bb_where) {
      if (board->bbp[QUEEN] & bb_where)
         return piece | QUEEN;
      return piece | ROOK;
   }
#if 0
   if (board->bbp[PAWN] & bb_where)
      return piece | PAWN;
   if (board->bbp[KNIGHT] & bb_where)
      return piece | KNIGHT;
   if (board->bbp[BISHOP] & bb_where)
      return piece | BISHOP;
   if (board->bbp[ROOK] & bb_where)
      return piece | ROOK;
   if (board->bbp[QUEEN] & bb_where)
      return piece | QUEEN;
#endif
#endif

   /* Only one possibility left - this should be a king */
   assert(board->bbp[KING] & bb_where)
   return piece | KING;
}


static inline void remove_known_piece(board_t *board, int piece, const int where)
{
   int colour;

   assert(where<=H8);
   assert(board);
   assert(get_occupied(board) & make_bitboard_square(where));

   const int piece_idx = piece&PIECE;
   bitboard_t bb = make_bitboard_square(where);

   colour = piece_colour(piece);
   board->bbc[colour] ^= bb;
   board->init &= ~bb;
   board->bbp[piece_idx] ^= bb;
#ifdef USE_ROTATED_BITBOARDS
   board->bbf ^= xchg_row_file(where);
   board->bbr[0] ^= get_a8h1_diagonal_square(where);
   board->bbr[1] ^= get_a1h8_diagonal_square(where);
#endif
   /* Update hash key */
   board->hash ^= piece_key[colour][piece_idx][where];
   if ((piece & PIECE) == PAWN) {
      board->pawn_hash ^= piece_key[colour][piece_idx][where];
   }

   /* Sanity checks */
   assert((board->bbc[colour] & bb) == 0);
   assert((board->bbp[piece_idx] & bb) == 0);
}


static inline void remove_piece(board_t *board, const int where)
{
   int piece;
   
   assert(where<=H8);
   assert(board);
   assert(get_occupied(board) & make_bitboard_square(where));

   piece = get_piece(board, where);
   assert((piece & PIECE) <= KING);
   remove_known_piece(board, piece, where);
}

/* Place the piece, assuming it isn't a pawn and therefore the pawn hash
 * doesn't need to be updated (saves some time).
 */
static inline void place_promotion_piece(board_t *board, const int where, const int piece)
{
   int colour;
   const int piece_idx = piece&PIECE;
   bitboard_t bb = make_bitboard_square(where);
   assert(where<=H8);

   colour = piece_colour(piece);

   /* Sanity checks */
   assert((board->bbc[colour] & bb) == 0);
   assert((board->bbp[piece_idx] & bb) == 0);

   board->bbc[colour] |= bb;
   board->bbp[piece_idx] |= bb;
#ifdef USE_ROTATED_BITBOARDS
   set_bitboard(&board->bbf, xchg_row_file(where));
   set_bitboard(&board->bbr[0], get_a8h1_diagonal_square(where));
   set_bitboard(&board->bbr[1], get_a1h8_diagonal_square(where));
#endif
   
   /* Update hash key */
   board->hash ^= piece_key[colour][piece_idx][where];
}

static inline void place_piece(board_t *board, const int where, const int piece)
{
   int colour = piece_colour(piece);
   const int piece_idx = piece&PIECE;

   assert((board->bbc[colour] & make_bitboard_square(where)) == 0);
   assert((board->bbp[piece_idx] & make_bitboard_square(where)) == 0);

   /* Place the piece */
   place_promotion_piece(board, where, piece);
   
   /* Update pawn hash key */
   board->pawn_hash ^= piece_key[colour][piece_idx][where];
}

/* Combine some of the operations in the above functions for better use of
 * the computer power that comes from using bitboards.
 * NB: the destination square must be *empty* for this to work correctly,
 * in other words, this does NOT handle captures.
 */
static inline void move_piece(board_t *board, int piece, const int from, const int to)
{
   bitboard_t mask;
   const int piece_idx = piece&PIECE;
   int colour;
   assert(from<=H8);
   assert(to<=H8);

   colour = piece_colour(piece);

   /* Sanity checks */
   assert((board->bbc[colour] & make_bitboard_square(to)) == 0);
   assert(board->bbc[colour] & make_bitboard_square(from));
   assert((board->bbp[piece_idx] & make_bitboard_square(to)) == 0);
   assert(board->bbp[piece_idx] & make_bitboard_square(from));

   mask = or_bitboards(make_bitboard_square(from), make_bitboard_square(to));

   board->bbc[colour] ^= mask;
   board->bbp[piece_idx] ^= mask;
   board->init &= ~mask;

#ifdef USE_ROTATED_BITBOARDS
   /* Flipped bitboard, rows and files interchanged */
   mask = or_bitboards(make_bitboard_square(xchg_row_file(from)),
                       make_bitboard_square(xchg_row_file(to)));
   board->bbf ^= mask;

   /* Rotated bitboards */
   mask = or_bitboards(make_bitboard_square(get_a8h1_diagonal_square(from)),
                       make_bitboard_square(get_a8h1_diagonal_square(to)));
   board->bbr[0] ^= mask;

   mask = or_bitboards(make_bitboard_square(get_a1h8_diagonal_square(from)),
                       make_bitboard_square(get_a1h8_diagonal_square(to)));
   board->bbr[1] ^= mask;
#endif

   /* Update hash key */
   board->hash ^= piece_key[colour][piece_idx][from];
   board->hash ^= piece_key[colour][piece_idx][to];
   if ((piece & PIECE) == PAWN) {
      board->pawn_hash ^= piece_key[colour][piece_idx][from];
      board->pawn_hash ^= piece_key[colour][piece_idx][to];
   }
}

/* cidx should be 0 or 1, for white and black respectively */
static inline void castle_rook(board_t *board, int cidx, uint8_t rook_bits)
{
   bitboard_t bb = make_bitboard_row(rook_bits, cidx*7);
#ifdef USE_ROTATED_BITBOARDS
   bitboard_t bbf = 0ll, bbr1 = 0ll, bbr2 = 0ll;
#endif

   /* Move rook by flipping bits on the relevant bitboards. This flips both
    * bits simultaneously.
    */
   board->bbp[ROOK] ^= bb;
   board->bbc[cidx] ^= bb;
   board->init = mask_bitboards(board->init, bb);


#ifdef USE_ROTATED_BITBOARDS
   /* Now for the tricky part: the rotated bitboards. What bits to flip
    * there we have to look up from a table containing the 4 possible
    * combinations (black/white, king/queen).
    * FIXME: for now, we calculate the bitboards here as they are needed.
    * Probably doesn't hurt so much as castling doesn't happen that often.
    */
   if (cidx == 0) {     /* White */
      if ( (rook_bits & 1) ) {      /* Queen side */
         bbf = or_bitboards(make_bitboard_square(xchg_row_file(A1)),
                            make_bitboard_square(xchg_row_file(D1)));
         bbr1 = or_bitboards( 
                        make_bitboard_square(get_a8h1_diagonal_square(A1)),
                        make_bitboard_square(get_a8h1_diagonal_square(D1)));
         bbr2 = or_bitboards( 
                        make_bitboard_square(get_a1h8_diagonal_square(A1)),
                        make_bitboard_square(get_a1h8_diagonal_square(D1)));
      } else {                      /* King side */
         bbf = or_bitboards(make_bitboard_square(xchg_row_file(H1)),
                            make_bitboard_square(xchg_row_file(F1)));
         bbr1 = or_bitboards( 
                        make_bitboard_square(get_a8h1_diagonal_square(H1)),
                        make_bitboard_square(get_a8h1_diagonal_square(F1)));
         bbr2 = or_bitboards( 
                        make_bitboard_square(get_a1h8_diagonal_square(H1)),
                        make_bitboard_square(get_a1h8_diagonal_square(F1)));
      }
   } else {          /* Black */
      if ( (rook_bits & 1) ) {      /* Queen side */
         bbf = or_bitboards(make_bitboard_square(xchg_row_file(A8)),
                            make_bitboard_square(xchg_row_file(D8)));
         bbr1 = or_bitboards( 
                        make_bitboard_square(get_a8h1_diagonal_square(A8)),
                        make_bitboard_square(get_a8h1_diagonal_square(D8)));
         bbr2 = or_bitboards( 
                        make_bitboard_square(get_a1h8_diagonal_square(A8)),
                        make_bitboard_square(get_a1h8_diagonal_square(D8)));
      } else {                      /* King side */
         bbf = or_bitboards(make_bitboard_square(xchg_row_file(H8)),
                            make_bitboard_square(xchg_row_file(F8)));
         bbr1 = or_bitboards( 
                        make_bitboard_square(get_a8h1_diagonal_square(H8)),
                        make_bitboard_square(get_a8h1_diagonal_square(F8)));
         bbr2 = or_bitboards( 
                        make_bitboard_square(get_a1h8_diagonal_square(H8)),
                        make_bitboard_square(get_a1h8_diagonal_square(F8)));
      }
   }

   /* Flip bits on rotated bitboards */
   board->bbf = xor_bitboards(board->bbf, bbf);
   board->bbr[0] = xor_bitboards(board->bbr[0], bbr1);
   board->bbr[1] = xor_bitboards(board->bbr[1], bbr2);
#endif

   /* Update hash key */
   board->hash ^= castle_rook_key[cidx][rook_bits&1];
}

/* Returns TRUE or FALSE depending on whether to given side may still
 * castle or not in the given position.
 * NB: side = 0 or 1 for white or black!
 */
static inline bool may_castle(const board_t *board, int side)
{
   if (board->did_castle[side])
      return false;


   if (!(board->init & board->bbp[KING] & board->bbc[side]))
      return false;

   if (!(board->init & board->bbp[ROOK] & board->bbc[side]))
      return false;

   return true;
}
#endif
