/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef MOVE_H
#define MOVE_H

#include <stdbool.h>
#include <stdint.h>
#include "pieces.h"
#include "assert.h"
#include "bitboard.h"

/* The information needed to store a move fits in a 32 bit integer */
typedef uint32_t move_t;

/* Offsets and test masks */
#define MOVE_SQUARE_MASK      0x3f
#define MOVE_PIECE_MASK       0x07
#define MOVE_PLAYER_MASK      0x80
#define MOVE_IS_CAPTURE_MASK  0x01
#define MOVE_IS_DPAWN_MASK    0x01
#define MOVE_IS_CASTLE_MASK   0x01

#define MOVE_FROM_SHIFT       0           /* From square    (6 bits) */
#define MOVE_TO_SHIFT         6           /* To square      (6 bits) */
#define MOVE_PIECE_SHIFT      12          /* Piece          (3 bits) */
#define MOVE_PLAYER_SHIFT     8           /* Player bit     (1 bit, offset by 7; it's really bit 15) */
#define MOVE_CAPTURE_SHIFT    16          /* Capture square (6 bits) */
#define MOVE_EP_SHIFT         16          /* En-passant square (6 bits), doubles with the capture square field */
#define MOVE_TPIECE_SHIFT     22          /* Piece on destination square (promotion) (3 bits) */
#define MOVE_IS_CAPTURE_SHIFT 25          /* Whether the move is a capture (1 bit) */
#define MOVE_IS_DPAWN_SHIFT   26          /* Whether the move is a double pawn move and stores an EP square */
#define MOVE_IS_CASTLE_SHIFT  27          /* Whether the move is a castling move */
#define MOVE_PTAKEN_SHIFT     28          /* Captured piece (3 bits) */

/* datastructure to hold the number of legal moves in a position */
#define MAX_MOVES    218   /* According to Wikipedia; seems safe... */
typedef struct movelist_t {
   move_t move[MAX_MOVES];
   int score[MAX_MOVES];
   int perm[MAX_MOVES];
   int num_moves;
   int cur_move;
} movelist_t;

char *short_move_string(const move_t move, movelist_t *movelist, char *buffer);
extern char *move_string(const move_t move, char *buffer);

/**********************************************************************
 *                      static inline functions                       *
 **********************************************************************/
/* Inlinable functions */
/* Get the player that makes the move */
static inline uint8_t get_move_player(const move_t move)
{
   return (move >> MOVE_PLAYER_SHIFT) & MOVE_PLAYER_MASK;
}

/* Get the piece that moves */
static inline uint8_t get_move_piece(const move_t move)
{
   uint8_t piece;

   piece = (move >> MOVE_PIECE_SHIFT) & MOVE_PIECE_MASK; 
   assert(piece <= KING);
   piece |= (move >> MOVE_PLAYER_SHIFT) & MOVE_PLAYER_MASK;
   return piece;
}

/* Get the origin square */
static inline uint8_t get_move_origin(const move_t move)
{
   return (move >> MOVE_FROM_SHIFT) & MOVE_SQUARE_MASK;
}

/* Get the destination square */
static inline uint8_t get_move_destination(const move_t move)
{
   return (move >> MOVE_TO_SHIFT) & MOVE_SQUARE_MASK;
}

/* Get the en-passant square */
static inline uint8_t get_ep_square(const move_t move)
{
   if ( (move >> MOVE_IS_DPAWN_SHIFT) & MOVE_IS_DPAWN_MASK )
      return (move >> MOVE_EP_SHIFT) & MOVE_SQUARE_MASK;
   else
      return 0;
}

/* Get the captured piece (if any) */
static inline uint8_t get_captured_piece(const move_t move)
{
   uint8_t piece;

   piece = (move >> MOVE_PTAKEN_SHIFT) & MOVE_PIECE_MASK; 
   assert(piece < KING);
   piece |= (move >> MOVE_PLAYER_SHIFT) & MOVE_PLAYER_MASK;
   return piece ^ BLACK;
}

/* Encode a standard move. Standard moves are everything apart from:
 *  1. Castling
 *  2. En-passant capture
 *  3. Promotion
 */
static inline move_t encode_normal_move(uint8_t piece, const uint8_t from, const uint8_t to)
{
   move_t move;
   uint8_t side;

   /* Sanity checks */
   assert(from <= H8);
   assert(to <= H8);

   /* Initialise all elements to 0 */
   move = 0;

   side = decode_piece_colour(piece);
   piece = decode_piece_type(piece);

   assert(piece <= KING);

   move |= (uint32_t)piece << MOVE_PIECE_SHIFT;
   move |= (uint32_t)side << MOVE_PLAYER_SHIFT;
   move |= (uint32_t)from << MOVE_FROM_SHIFT;
   move |= (uint32_t)to << MOVE_TO_SHIFT;
   move |= (uint32_t)to << MOVE_CAPTURE_SHIFT;

   return move;
}

/* Encode a double pawn move; normal apart from that it stores the
 * en-passant square.
 */
static inline move_t encode_doublep_move(uint8_t piece, const uint8_t from, const uint8_t to, const uint8_t ep)
{
   move_t move;
   uint8_t side;

   /* Sanity checks */
   assert(from <= H8);
   assert(to <= H8);

   /* Initialise all elements to 0 */
   move = 0;

   side = decode_piece_colour(piece);
   piece = decode_piece_type(piece);

   assert(piece <= KING);

   move |= (uint32_t)piece << MOVE_PIECE_SHIFT;
   move |= (uint32_t)side << MOVE_PLAYER_SHIFT;
   move |= (uint32_t)from << MOVE_FROM_SHIFT;
   move |= (uint32_t)to << MOVE_TO_SHIFT;
   //move |= (uint32_t)to << MOVE_CAPTURE_SHIFT;
   move |= (uint32_t)ep << MOVE_EP_SHIFT;
   move |= (uint32_t)MOVE_IS_DPAWN_MASK << MOVE_IS_DPAWN_SHIFT;

   return move;
}

/* Encode a promotion move. */
static inline move_t encode_promotion_move(uint8_t tpiece, const uint8_t from, const uint8_t to)
{
   move_t move;
   uint8_t side;

   /* Sanity checks */
   assert(from <= H8);
   assert(to <= H8);

   /* Initialise all elements to 0 */
   move = 0;

   side = decode_piece_colour(tpiece);
   tpiece = decode_piece_type(tpiece);

   assert(tpiece <= KING);

   move |= (uint32_t)PAWN << MOVE_PIECE_SHIFT;
   move |= (uint32_t)side << MOVE_PLAYER_SHIFT;
   move |= (uint32_t)tpiece << MOVE_TPIECE_SHIFT;
   move |= (uint32_t)from << MOVE_FROM_SHIFT;
   move |= (uint32_t)to << MOVE_TO_SHIFT;
   move |= (uint32_t)to << MOVE_CAPTURE_SHIFT;

   return move;
}

/* Encode en-passant capture */
static inline move_t encode_enpassant_capture(uint8_t piece, const uint8_t from, const uint8_t to, const uint8_t ep)
{
   move_t move;
   uint8_t side;

   /* Sanity checks */
   assert(from <= H8);
   assert(to <= H8);

   /* Initialise all elements to 0 */
   move = 0;

   side = decode_piece_colour(piece);
   piece = decode_piece_type(piece);
 
   move |= (uint32_t)piece << MOVE_PIECE_SHIFT;
   move |= (uint32_t)side << MOVE_PLAYER_SHIFT;
   move |= (uint32_t)from << MOVE_FROM_SHIFT;
   move |= (uint32_t)to << MOVE_TO_SHIFT;
   move |= (uint32_t)ep << MOVE_CAPTURE_SHIFT;
   move |= (uint32_t)MOVE_IS_CAPTURE_MASK << MOVE_IS_CAPTURE_SHIFT;

   return move;
}

/* Encode a castling move */
static inline move_t encode_castling(const uint8_t piece, const uint8_t from, uint8_t to, uint8_t rook_bits)
{
   move_t move;

   /* Sanity checks */
   assert(from <= H8);
   assert(to <= H8);

   move = encode_normal_move(piece, from, to);
   move |= (uint32_t)MOVE_IS_CASTLE_MASK << MOVE_IS_CASTLE_SHIFT;

   return move;
}

/* Compare two moves */
static inline int moves_are_equal(const move_t m1, const move_t m2)
{
   return m1 == m2;
}

static inline void set_captured_piece(move_t *m, const uint8_t ptaken)
{
   (*m) |= (uint32_t)(ptaken & MOVE_PIECE_MASK) << MOVE_PTAKEN_SHIFT;
}

static inline void set_promotion_piece(move_t *m, uint8_t tpiece)
{
   tpiece &= MOVE_PIECE_MASK;
   (*m) &= ~( (uint32_t)MOVE_PIECE_MASK << MOVE_TPIECE_SHIFT);
   (*m) |= (uint32_t)tpiece << MOVE_TPIECE_SHIFT;
}

/****************/
/*  Move types  */
/****************/
/* Returns true if a move is a promotion move */
static inline int is_promotion_move(const move_t move)
{
   return (move >> MOVE_TPIECE_SHIFT) & MOVE_PIECE_MASK;
}

/* Returns true if a move is a promotion move */
static inline int is_underpromotion_move(const move_t move)
{
   return ((move >> MOVE_TPIECE_SHIFT) & MOVE_PIECE_MASK) < QUEEN;
}

/* Returns TRUE if the move is an en-passant move */
static inline int is_enpassant_move(const move_t move)
{
   return ((move >> MOVE_TO_SHIFT)&MOVE_SQUARE_MASK) !=
          ((move >> MOVE_CAPTURE_SHIFT)&MOVE_SQUARE_MASK);
}

/* Returns TRUE if the move is a double pawn move */
static inline int is_doublepawn(const move_t move)
{
   int piece = get_move_piece(move) & PIECE;
   int from  = get_move_origin(move);
   int to    = get_move_destination(move);
   return (piece == PAWN) && (abs(from-to)==16);
}

/* returns TRUE is a move is a castling move */
static inline int is_castle_move(const move_t move)
{
   return (move >> MOVE_IS_CASTLE_SHIFT) & MOVE_IS_CASTLE_MASK;
}

static inline uint8_t get_castle_flags(const move_t move)
{
   assert(is_castle_move(move));
   if ((get_move_destination(move)&7) == 6)
      return castlek_rook_bits;
   else
      return castleq_rook_bits;
}

/* returns TRUE is a move is a capture move */
static inline int is_capture_move(const move_t move)
{
   return (move >> MOVE_IS_CAPTURE_SHIFT) & MOVE_IS_CAPTURE_MASK;
}

static inline void set_capture_flag(move_t *move)
{
   (*move) |= (uint32_t)MOVE_IS_CAPTURE_MASK << MOVE_IS_CAPTURE_SHIFT;
}

static inline bool is_irreversible_move(move_t move)
{
   return is_capture_move(move) || (get_move_piece(move)&PIECE) == PAWN;
}

/* Get promotion piece */
static inline int get_promotion_piece(const move_t move)
{
   uint8_t piece;

   piece = (move >> MOVE_TPIECE_SHIFT) & MOVE_PIECE_MASK; 
   assert(piece <= KING);
   piece |= (move >> MOVE_PLAYER_SHIFT) & MOVE_PLAYER_MASK;
   return piece;
}

/* Get the square the captured piece was on.
 * This is the destination square for ordinary moves but not for en-passant
 * captures.
 */
static inline int get_capture_square(const move_t move)
{
   return (move >> MOVE_CAPTURE_SHIFT) & MOVE_SQUARE_MASK;
}


/*************************
 * Accessing a move list *
 *************************/
static inline move_t get_next_move(movelist_t *movelist)
{
   assert(movelist);
   assert(movelist->cur_move < movelist->num_moves);
   move_t move = 0;

#if 0
   if (movelist->cur_move < movelist->num_moves) {
      int nm = movelist->cur_move;
      int score = movelist->score[nm];
      int n;
      for (n=movelist->cur_move; n<movelist->num_moves; n++) {
         if (movelist->score[n] > movelist->score[nm])
            nm = n;
      }
      n = movelist->cur_move;
      move = movelist->move[n];
      movelist->move[n] = movelist->move[nm];
      movelist->move[nm] = move;
      score = movelist->score[n];
      movelist->score[n] = movelist->score[nm];
      movelist->score[nm] = score;

      move = movelist->move[movelist->cur_move];
      movelist->cur_move++;
   }
#else
   move = movelist->move[movelist->perm[movelist->cur_move]];
   movelist->cur_move++;
#endif

   return move;
}

static inline int get_move_score(const movelist_t *movelist)
{
   assert(movelist);
   assert(movelist->cur_move > 0);
   assert(movelist->cur_move-1 < movelist->num_moves);

#if 0
   return movelist->score[movelist->cur_move-1];
#else
   return movelist->score[movelist->perm[movelist->cur_move-1]];
#endif
}

#endif

