/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "jazz.h"

typedef struct {
   char move[10];
   int score;
} in_move;

polyglot_position_t *pg_book = NULL;
int pg_book_size = 0;
int pg_book_max_size = 0;

void add_pg_book_entry(polyglot_position_t *pos)
{
   if (pg_book_size >= pg_book_max_size) {
      pg_book_max_size = 2*(pg_book_max_size + 1);
      pg_book = realloc(pg_book, pg_book_max_size * sizeof *pg_book);
   }

   pg_book[pg_book_size] = *pos;
   pg_book_size++;
}

int compare_pg_book_entries(const void *p1, const void *p2)
{
   polyglot_position_t *pos1 = (polyglot_position_t *)p1;
   polyglot_position_t *pos2 = (polyglot_position_t *)p2;

   if (pos1->key < pos2->key)
      return -1;
   if (pos1->key > pos2->key)
      return 1;

   if (pos1->weight < pos2->weight)
      return -1;
   if (pos1->weight > pos2->weight)
      return 1;

   return 0;
}

int main(int argc, char **argv)
{
   const char *input_file = "../bigbook.dat";
   const char *output_file = "bigbook.bin";
   gamestate_t *game = NULL;
   FILE *fin, *fout;
   in_move moves[40];

   printf("Welcome to Jazz\n");
   printf("Memory sizes:\n"
          " * move_t     : %lu bytes\n"
          " * bitboard_t : %lu bytes\n"
          " * hash_table_: %lu bytes\n"
          " * board_t    : %lu bytes\n",
          sizeof(move_t),
          sizeof(bitboard_t),
          sizeof(hash_table_entry_t),
          sizeof(board_t));
   printf("Size of transposition table: %lu bytes\n",
         sizeof(hash_table_entry_t) * HASH_TABLE_SIZE);

   printf("Initialising Jazz engine\n");
   initialise_jazz();

   fin = fopen(input_file, "r");

   game = create_game();
   start_new_game(game);

   while(!feof(fin)) {
      char buffer[4096];
      uint64_t key;
      int score = 0;
      int nm = 0;
      int n = 0;
      char *s;

      /* FEN string */
      fgets(buffer, sizeof buffer, fin);
      setup_fen_position(game, buffer);

      if (buffer[0] == '#')
         break;

      /* Now test: if the EP-square is set, there must be a pawn present
       * for the polyglot key to be set.
       * This is different from the FEN standard!
       */
      if (game->board->ep_square) {
         bitboard_t bb = make_bitboard_square(game->board->ep_square);
         if (game->side_to_move) {
            bb <<= 8;
            bb = shift_bitboard_file_left(bb) | shift_bitboard_file_right(bb);
            if ((game->board->bbp[PAWN] & bb & game->board->bbc[1]) == 0)
               game->board->ep_square = 0;
         } else {
            bb >>= 8;
            bb = shift_bitboard_file_left(bb) | shift_bitboard_file_right(bb);
            if ((game->board->bbp[PAWN] & bb & game->board->bbc[0]) == 0)
               game->board->ep_square = 0;
         }
      }

      key = get_polyglot_key(game->board, game->side_to_move);
      s = strstr(buffer, "\n");
      if (s) *s = '\0';
      printf("%s -> 0x%016llx\n", buffer, key);

      /* Move list */
      fgets(buffer, sizeof buffer, fin);
      s = buffer;
      while (*s) {
         /* Skip space */
         char *p = strstr(s, "{");
         if (!p) break;
         *p = '\0';
         strcpy(moves[nm].move, s);
         p++;
         sscanf(p, "%d}", &moves[nm].score);
         score += moves[nm].score;
         s = p;
         while(*s && (*s != ' ')) s++;
         while(*s && (*s == ' ')) s++;
         nm++;
      }
      for (n=0; n<nm; n++) {
         //printf("%s %d\n", moves[n].move, moves[n].score);
      }

      /* Encode move in polyglot format */
      for (n=0; n<nm; n++) {
         polyglot_position_t pg_pos;
         uint8_t from, to, p;
         uint16_t weight;
         from = pack_row_file(moves[n].move[1] - '1', moves[n].move[0] - 'a');
         to = pack_row_file(moves[n].move[3] - '1', moves[n].move[2] - 'a');
         p = moves[n].move[4];

         /* Promotion moves */
         if (p) {
            const char pp[] = "nbrq";
            int n;
            for (n=0; n<4; n++) {
               if (p == pp[n]) {
                  p = n;
                  break;
               }
            }
         }

         /* Castling moves */
         if (from == E1 && to == G1 && (game->board->bbp[KING] & E1_MASK)) to++;
         if (from == E1 && to == C1 && (game->board->bbp[KING] & E1_MASK)) to-=2;
         if (from == E8 && to == G8 && (game->board->bbp[KING] & E8_MASK)) to++;
         if (from == E8 && to == C8 && (game->board->bbp[KING] & E8_MASK)) to-=2;

         weight = 65535 * (float)moves[n].score / score;

         pg_pos.key = key;
         pg_pos.move = to | (from<<6) | (p<<12);
         pg_pos.weight = weight;
         pg_pos.learn = 0;

         add_pg_book_entry(&pg_pos);
      }
   }
   fclose(fin);

   /* Sort the book */
   qsort(pg_book, pg_book_size, sizeof *pg_book, compare_pg_book_entries);

   fout = fopen(output_file, "w");
   int n;
   for (n=0; n<pg_book_size; n++) {
      entry_to_file(fout, pg_book+n);
      //printf("0x%016llx\n", pg_book[n].key);
   }
   fclose(fout);

   return 0;
}
