/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <ctype.h>
#include <time.h>
#include <inttypes.h>

#include "jazz.h"
#include "keypressed.h"

#ifdef __APPLE__
#define __unix__
#endif
#ifdef __unix__
#include "signal.h"
#endif

#ifdef HAVE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#if defined __LP64__ || defined _WIN64
#define ARCHSTR "(x86_64)"
#elif defined __i386__ || defined _WIN32
#define ARCHSTR "(i386)"
#elif defined POWERPC
#define ARCHSTR "(powerpc)"
#else
#define ARCHSTR "(unknown)"
#endif 

#ifndef SVNVERSION
#define SVNVERSION "unknown revision "
#endif
#define VERSIONSTR SVNVERSION

#ifndef SMP
#define PROGNAME "Jazz"
#else
#define PROGNAME "Jazz Orchestra"
#endif

#define TIME_BUFFER  100

typedef struct help_topic_t {
   char *topic;
   char *cmd;
   char *text;
} help_topic_t;

typedef struct position_signature_t {
   char *fen;
   int depth;
   uint64_t nodes;
} position_signature_t;

static position_signature_t perftests[] = {
   // Martin Sedlak's test positions
   // avoid illegal ep
   { "3k4/3p4/8/K1P4r/8/8/8/8 b - - 0 1",         6, 1134888 },
   { "8/8/8/8/k1p4R/8/3P4/3K4 w - - 0 1",         6, 1134888 },
   { "8/8/4k3/8/2p5/8/B2P2K1/8 w - - 0 1",         6, 1015133 },
   { "8/b2p2k1/8/2P5/8/4K3/8/8 b - - 0 1",         6, 1015133 },
   // en passant capture checks opponent: 
   { "8/8/1k6/2b5/2pP4/8/5K2/8 b - d3 0 1",         6, 1440467 },
   { "8/5k2/8/2Pp4/2B5/1K6/8/8 w - d6 0 1",         6, 1440467 },
   // short castling gives check: 
   { "5k2/8/8/8/8/8/8/4K2R w K - 0 1",            6, 661072 },
   { "4k2r/8/8/8/8/8/8/5K2 b k - 0 1",            6, 661072 },
   // long castling gives check: 
   { "3k4/8/8/8/8/8/8/R3K3 w Q - 0 1",            6, 803711 },
   { "r3k3/8/8/8/8/8/8/3K4 b q - 0 1",            6, 803711 },
   // castling (including losing cr due to rook capture): 
   { "r3k2r/1b4bq/8/8/8/8/7B/R3K2R w KQkq - 0 1",   4, 1274206 },
   { "r3k2r/7b/8/8/8/8/1B4BQ/R3K2R b KQkq - 0 1",    4, 1274206 },
   // castling prevented: 
   { "r3k2r/8/3Q4/8/8/5q2/8/R3K2R b KQkq - 0 1",   4, 1720476 },
   { "r3k2r/8/5Q2/8/8/3q4/8/R3K2R w KQkq - 0 1",   4, 1720476 },
   // promote out of check: 
   { "2K2r2/4P3/8/8/8/8/8/3k4 w - - 0 1",         6, 3821001 },
   { "3K4/8/8/8/8/8/4p3/2k2R2 b - - 0 1",         6, 3821001 },
   // discovered check: 
   { "8/8/1P2K3/8/2n5/1q6/8/5k2 b - - 0 1",         5, 1004658 },
   { "5K2/8/1Q6/2N5/8/1p2k3/8/8 w - - 0 1",         5, 1004658 },
   // promote to give check: 
   { "4k3/1P6/8/8/8/8/K7/8 w - - 0 1",            6, 217342 },
   { "8/k7/8/8/8/8/1p6/4K3 b - - 0 1",            6, 217342 },
   // underpromote to check: 
   { "8/P1k5/K7/8/8/8/8/8 w - - 0 1",            6, 92683 },
   { "8/8/8/8/8/k7/p1K5/8 b - - 0 1",            6, 92683 },
   // self stalemate: 
   { "K1k5/8/P7/8/8/8/8/8 w - - 0 1",            6, 2217 },
   { "8/8/8/8/8/p7/8/k1K5 b - - 0 1",            6, 2217 },
   // stalemate/checkmate: 
   { "8/k1P5/8/1K6/8/8/8/8 w - - 0 1",            7, 567584 },
   { "8/8/8/8/1k6/8/K1p5/8 b - - 0 1",            7, 567584 },
   // double check: 
   { "8/8/2k5/5q2/5n2/8/5K2/8 b - - 0 1",         4, 23527 },
   { "8/5k2/8/5N2/5Q2/2K5/8/8 w - - 0 1",         4, 23527 },

   // short castling impossible although the rook never moved away from its corner 
   { "1k6/1b6/8/8/7R/8/8/4K2R b K - 0 1", 5, 1063513 },
   { "4k2r/8/8/7r/8/8/1B6/1K6 w k - 0 1", 5, 1063513 },

   // long castling impossible although the rook never moved away from its corner 
   { "1k6/8/8/8/R7/1n6/8/R3K3 b Q - 0 1", 5, 346695 },
   { "r3k3/8/1N6/r7/8/8/8/1K6 w q - 0 1", 5, 346695 },

   // From the Wiki
   { "r3k2r/p1ppqpb1/bn2pnp1/3PN3/1p2P3/2N2Q1p/PPPBBPPP/R3K2R w KQkq -", 4, 4085603 },
   { "rnbqkb1r/pp1p1ppp/2p5/4P3/2B5/8/PPP1NnPP/RNBQK2R w KQkq - 0 6", 3, 53392 },

   // John Merlino's test positions, some of these take a long time, only do them
   // in debug mode.
#ifdef DEBUGMODE
   { "r3k2r/8/8/8/3pPp2/8/8/R3K1RR b KQkq e3 0 1", 6, 485647607 },
   { "r3k2r/Pppp1ppp/1b3nbN/nP6/BBP1P3/q4N2/Pp1P2PP/R2Q1RK1 w kq - 0 1", 6, 706045033 },
   { "8/7p/p5pb/4k3/P1pPn3/8/P5PP/1rB2RK1 b - d3 0 28", 6, 38633283 },
   { "8/3K4/2p5/p2b2r1/5k2/8/8/1q6 b - - 1 67", 7, 493407574 },
   { "rnbqkb1r/ppppp1pp/7n/4Pp2/8/8/PPPP1PPP/RNBQKBNR w KQkq f6 0 3", 6, 244063299 },
   { "r3k2r/p1ppqpb1/bn2pnp1/3PN3/1p2P3/2N2Q1p/PPPBBPPP/R3K2R w KQkq -", 5, 193690690 },
   { "8/p7/8/1P6/K1k3p1/6P1/7P/8 w - -", 8, 8103790 },
   { "n1n5/PPPk4/8/8/8/8/4Kppp/5N1N b - -", 6, 71179139 },
   { "r3k2r/p6p/8/B7/1pp1p3/3b4/P6P/R3K2R w KQkq -", 6, 77054993 },
#endif
   { "8/2p5/3p4/KP5r/1R3p1k/8/4P1P1/8 w - -", 7, 178633661 },
   { "8/5p2/8/2k3P1/p3K3/8/1P6/8 b - -", 8, 64451405 },
   { "r3k2r/pb3p2/5npp/n2p4/1p1PPB2/6P1/P2N1PBP/R3K2R w KQkq -", 5, 29179893 },

   { NULL }
};

/* Benchmark positions. For now we just use the Stockfish ones */
static position_signature_t benchtests[] = {
  { "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1" },
  { "r3k2r/p1ppqpb1/bn2pnp1/3PN3/1p2P3/2N2Q1p/PPPBBPPP/R3K2R w KQkq -" },
  { "8/2p5/3p4/KP5r/1R3p1k/8/4P1P1/8 w - -" },
  { "4rrk1/pp1n3p/3q2pQ/2p1pb2/2PP4/2P3N1/P2B2PP/4RRK1 b - - 7 19" },
  { "rq3rk1/ppp2ppp/1bnpb3/3N2B1/3NP3/7P/PPPQ1PP1/2KR3R w - - 7 14" },
  { "r1bq1r1k/1pp1n1pp/1p1p4/4p2Q/4Pp2/1BNP4/PPP2PPP/3R1RK1 w - - 2 14" },
  { "r3r1k1/2p2ppp/p1p1bn2/8/1q2P3/2NPQN2/PPP3PP/R4RK1 b - - 2 15" },
  { "r1bbk1nr/pp3p1p/2n5/1N4p1/2Np1B2/8/PPP2PPP/2KR1B1R w kq - 0 13" },
  { "r1bq1rk1/ppp1nppp/4n3/3p3Q/3P4/1BP1B3/PP1N2PP/R4RK1 w - - 1 16" },
  { "4r1k1/r1q2ppp/ppp2n2/4P3/5Rb1/1N1BQ3/PPP3PP/R5K1 w - - 1 17" },
  { "2rqkb1r/ppp2p2/2npb1p1/1N1Nn2p/2P1PP2/8/PP2B1PP/R1BQK2R b KQ - 0 11" },
  { "r1bq1r1k/b1p1npp1/p2p3p/1p6/3PP3/1B2NN2/PP3PPP/R2Q1RK1 w - - 1 16" },
  { "3r1rk1/p5pp/bpp1pp2/8/q1PP1P2/b3P3/P2NQRPP/1R2B1K1 b - - 6 22" },
  { "r1q2rk1/2p1bppp/2Pp4/p6b/Q1PNp3/4B3/PP1R1PPP/2K4R w - - 2 18" },
  { "4k2r/1pb2ppp/1p2p3/1R1p4/3P4/2r1PN2/P4PPP/1R4K1 b - - 3 22" },
  { "3q2k1/pb3p1p/4pbp1/2r5/PpN2N2/1P2P2P/5PP1/Q2R2K1 b - - 4 26" },

  { NULL }
};

static help_topic_t help_topic[] = {
   /* .........|.........|.........|.........|.........|.........|.........|.........| */
   { "all", NULL, NULL },

   { "analyse", "analyse, analyze",
     "  Analyse the current position.\n" },

   { "colour", "colour on|off",
     "  Switches use of colour to represent the different squares on the\n"
     "  printed board on or off. Useful for terminals that cannot display colour\n"
     "  or that don't understand ANSI escape codes.\n" }, 

   { "bench", "test benchmark [depth]",
     "  Perform a benchmark test to the specified depth. The returned node-count\n"
     "  can be used as a validation that the program is working correctly while\n"
     "  the reported time can be used as a performance measure on the current\n"
     "  system.\n" },

   { "board", "board [on|off]",
     "  Print the current board position, or toggles automatic printing of the\n"
     "  current position on and off.\n" },

   { "book", "book [filename|off]",
     "  Set the name of the (polyglot) opening book file to use.\n"
     "  'book off' switches off the opening book.\n" },

#ifdef SMP
   { "cores", "cores N, threads N",
     "  Use N cores/threads for the search.\n" },
#endif

   { "fen", NULL,
     "  Print the FEN representation of the current board position.\n" },

   { "force", NULL,
     "  Switch to 'force' mode: Jazz will play neither side in the game until\n"
     "  it receives a 'go' command\n" },

   { "go", NULL,
     "  Switch off force mode, tell Jazz to start playing from the current position.\n" },

   { "help", "help [topic]",
     "  Display a list of help topics, or detailed information about a particular\n"
     "  topic.\n" },

   { "hint", NULL,
     "  Print the move that Jazz is pondering on. If there is no ponder move, no\n"
     "  hint is displayed.\n" },

   { "level", "level moves time inc",
     "  Set time control options: number of moves per session, time per session\n"
     "  (either in minutes or minutes:seconds) and time increment (in seconds)\n" },

   { "maxnodes", "maxnodes n",
     "  Set the maximum number of nodes that can be searched before a move is\n"
     "  returned. Setting maxnodes to 0 disables it and restores normal time\n"
     "  control.\n" },

   { "memory", "memory [MB]",
     "  Set the (approximate) amount of memory the program can use, in MB. The\n"
     "  actual amount of memory used will be different from this and may be\n"
     "  slightly larger or smaller.\n" },

   { "new", NULL,
     "  Start a new game from the starting position.\n" },

   { "perft", "perft [depth] [print depth]",
     "  Perform a 'perft' (performance test) on the current position: count all\n"
     "  positions that can result from this position to the specified depth.\n"
     "  If 'print depth' is specified then the total will be sub-divided per move\n"
     "  upto 'print depth'\n" },

   { "ponder", "ponder [on|off]",
     "  Switches ponder mode on or off (Jazz will think while it is not on move)\n" }, 

   { "post", "(no)post",
     "  Display thinking output (post) or not (nopost)\n" },

   { "prompt", "prompt [on|off]",
     "  Switch the prompt on or off\n" },

   { "quit", "quit, exit, bye",
     "  Exit Jazz\n" },

   { "readepd", "readepd filename",
     "  Read (and setup) a position from an EPD file.\n" },

   { "setboard", "setboard FEN",
     "  Setup a position on the board from a FEN string.\n" },

   { "sd", "sd depth",
     "  Specify the maximum search depth\n" },

   { "shell", "!command",
     "  Run a shell command.\n" },

   { "st", "st time",
     "  Specify the maximum time (in seconds) to think on a single move\n" }, 

   { "takeback", "takeback, remove",
     "  Reverses the last two moves in the game, if any.\n" }, 

   { "test", "test [movegen|benchmark [depth]]",
     "  Perform tests on the move generator, or the search.\n" },

   { "time", "time csec",
     "  Set the remaining time on the engine's clock, in centi-seconds.\n" },

   { "uci", "uci [on|off]",
     "  Toggle UCI mode on or off. In UCI mode commands from the UCI protocol\n"
     "  are recognised in addition to the standard commands described here.\n"
     "  'uci' is equivalent to 'uci on'\n"},

   { "undo", NULL,
     "  Unmakes the last move in the game, if any. If it is Jazz' turn, it will\n"
     "  start thinking again immediately.\n" },

   { "unicode", "unicode on|off",
     "  Switches use of unicode characters to represent the chess pieces on the\n"
     "  printed board on or off. Useful for terminals that cannot display unicode\n"
     "  or that don't have the required glyphs.\n" }, 

   { "xboard", "xboard [on|off]",
     "  Switch to xboard mode: prompt off, don't display board, don't print SAN\n"
     "  moves, don't trap interrupts (ctrl-C).\n"
     "  'xboard' is equivalent to 'xboard on'\n" },

   { "MOVE", NULL,
     "  Play a move on the board. Moves can be entered as 'long algebraic'\n"
     "  (e2e4, g1f3) or SAN (e4, Nf3).\n" },

   { NULL }
};

static bool prompt = true;
static bool show_board = true;
static bool san = true;
static bool trapint = true;

#ifdef __APPLE__
#define __unix__
#endif
#ifdef __unix__
static sig_t old_signal_handler;

void interrupt_computer(int i)
{
   abort_search = true;
   if (old_signal_handler)
      old_signal_handler(i);
}
#endif


#ifdef HAVE_READLINE
static bool stdin_is_terminal(void)
{
#ifdef __unix__
   return isatty(fileno(stdin));
#else
   return true;
#endif
}
#endif

char *trim(char *s)
{
   char *p = s;
   while (*p && isspace(*p)) p++;
   if (p!= s) {
      for (int n = 0; n<=strlen(p); n++)
         s[n] = p[n];
   }
   return s;
}

char *chomp(char *s)
{
   char *p;
   p = strstr(s, "\n");
   if (p) *p = '\0';
   return s;
}

static bool streq(const char *s1, const char *s2)
{
   return strcmp(s1, s2) == 0;
}

static FILE *f = NULL;
static char *buf;

static bool may_ponder = false;

static void log_engine_output(const char *msg, ...) __attribute__((unused));
static void log_engine_output(const char *msg, ...)
{
   va_list ap;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   if (f) {
      fprintf(f, "  %s", buf);
      fflush(f);
   }
}

static void log_xboard_output(const char *msg, ...)
{
   va_list ap;
   static bool newline = true;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   if (f) {
      if (newline)
         fprintf(f, "> ");
      newline = false;
      fprintf(f, "%s", buf);
      if (strstr(buf, "\n"))
         newline = true;
   }

   printf("%s", buf);
}

int num_games = 0;

/* Play a sequence of moves from the initial position */
void replay_game(gamestate_t *game, char *moves)
{
   movelist_t movelist;
   int from, to, n;
   char *s;
   /* First rewind to the beginning if not there yet */
   while (game->moves_played)
      takeback(game);

#ifndef SMP
   /* Clear the repetition hash table - it will be rebuild as we play
    * through the game.
    */
   memset(game->repetition_hash_table, 0, sizeof game->repetition_hash_table);
   game->repetition_hash_table[game->board->hash&0xFFFF] = 1;
#endif

   s = moves;
   while(s && *s) {
      generate_moves(&movelist, game->board, game->side_to_move);
      while (*s == ' ') s++;
      if (*s) {
         move_t move;
         int col = *s - 'a'; s++;
         int row = *s - '1'; s++;
         from = pack_row_file(row, col);
         col = *s - 'a'; s++;
         row = *s - '1'; s++;
         to = pack_row_file(row, col);

         n = validate_move(&movelist, from, to);
         if (n<0) {
            int k;
            for (k=0; k<movelist.num_moves; k++) {
               fprintf(stderr, "%s ", move_string(movelist.move[k], NULL));
            }
            fprintf(stderr, "\n");
            break;
         }
         move = movelist.move[n];
         if (is_promotion_move(move)) {
            switch (*s) {
               case 'q':
                  set_promotion_piece(&move, get_move_player(move) | QUEEN);
                  break;
               case 'r':
                  set_promotion_piece(&move, get_move_player(move) | ROOK);
                  break;
               case 'n':
                  set_promotion_piece(&move, get_move_player(move) | KNIGHT);
                  break;
               case 'b':
                  set_promotion_piece(&move, get_move_player(move) | BISHOP);
                  break;
            }
            s++;
         }
         playmove(game, move);
      }
   }
}

/* Play a sequence of moves from the initial position */
bool input_move(gamestate_t *game, char *move_str)
{
   char *s;

   if (!game)
      return false;

   s = move_str;
   if(s && *s) {
      while (*s == ' ') s++;
      move_t move = move_string_to_move(game, s);
      if (move == 0) {
         fprintf(stderr, "\n");
         if (f) {
            printf("Report illegal move %s in position:\n", move_str);
            print_bitboards_file(f, game->board);
         }
         return false;
      }
      playmove(game, move);
   } else {
      return false;
   }

   //write_epd_file(game, "xb_next.epd");
   return true;
}

static char ponder_input[65536];
static bool keyboard_input_on_move(gamestate_t *game)
{
   if (keyboard_input_waiting() && fgets(ponder_input, sizeof ponder_input, stdin)) {
      if (strstr(ponder_input, "random")) {
         sgenrand(time(NULL));
      } else if (strstr(ponder_input, "easy")) {
         may_ponder = false;
      } else if (strstr(ponder_input, "hard")) {
         may_ponder = true;
      } else if (strstr(ponder_input, "post")) {
         set_xboard_output_function(log_xboard_output);
      } else if (strstr(ponder_input, "nopost")) {
         set_xboard_output_function(NULL);
      } else if (strstr(ponder_input, "?") == ponder_input) {
         return true;
      } else if (strstr(ponder_input, "stop") == ponder_input) {
         return true;
      } else if (strstr(ponder_input, "otim")) {
      } else if (strstr(ponder_input, "ping")) {
         log_xboard_output("pong %s\n", ponder_input+5);
      } else if (strstr(ponder_input, "result")) {
      } else if (strstr(ponder_input, "time")) {
      } else if (strstr(ponder_input, "pause")) {
         uint64_t start_pause = get_timer();
         /* Sleep until keyboard input */
         while(!fgets(ponder_input, sizeof ponder_input, stdin) || !strstr(ponder_input, "resume"));
         uint64_t stop_pause = get_timer();
         game->start_time += stop_pause - start_pause;
      } else if (streq(ponder_input, "quit")) {
         if (game)
            end_game(game);
         free(buf);
         exit(0);
      }
   }
   return false;
}

static bool uci_keyboard_input_on_move(gamestate_t *game)
{
   if (keyboard_input_waiting() && fgets(ponder_input, sizeof ponder_input, stdin)) {
      if (strstr(ponder_input, "ponderhit")) {
         set_time_for_game(game);
         return false;
      } else if (strstr(ponder_input, "stop") == ponder_input) {
         return true;
      } else if (streq(ponder_input, "quit")) {
         exit(0);
      }
   }
   return false;
}


static bool keyboard_input_analyse(gamestate_t *game)
{
   bool restart = false;
   if (keyboard_input_waiting() && fgets(ponder_input, sizeof ponder_input, stdin)) {
      /* FIXME: for some of these the search should be at the root
       * position, for others the search should continue uninterrupted.
       * This means we need to push the command on a stack and set a flag
       * to interrupt the current search.
       * At the root and in the main analysis function we should then pop
       * commands off the stack before deciding whether to continue or not.
       */
      if (strstr(ponder_input, "undo")) {
         takeback(game);
         restart = true;
      } else if (strstr(ponder_input, "new")) {
         start_new_game(game);
         restart = true;
      } else if (strstr(ponder_input, "exit")) {
         game->analysing = false;
         restart = true;
      } else if (strstr(ponder_input, "setboard")) {
         char *p = strstr(ponder_input, " ");
         while (p && *p && p[0]==' ') p++;
         setup_fen_position(game, p);
         restart = true;
      } else if (strstr(ponder_input, ".") == ponder_input) {
      } else if (strstr(ponder_input, "hint")) {
         if (game && game->ponder_move)
            log_xboard_output("Hint: %s\n", move_string(game->ponder_move, NULL));
      } else if (strstr(ponder_input, "bk")) {
      } else if (streq(ponder_input, "quit")) {
         if (game)
            end_game(game);
         free(buf);
         exit(0);
      } else {
         if (ponder_input[0] && !input_move(game, ponder_input))
            log_xboard_output("Illegal move: %s\n", ponder_input);
         else
            restart = true;
      }
   }
   return restart;
}

static uint64_t perft(gamestate_t *game, int depth, int root)
{
   movelist_t movelist;
   int me = game->side_to_move;
   uint64_t nodes = 0;
   bool check = game->board->in_check;
   int n;

   if (abort_search) return 0;

   if (depth == 0) return 1;

   generate_moves(&movelist, game->board, game->side_to_move);
   for (n=0; n<movelist.num_moves; n++) {
      uint64_t count = 0;

      playmove(game, movelist.move[n]);
      if (check || !move_leaves_player_in_check(game, movelist.move[n], me))  /* Don't count illegal moves */
         count = perft(game, depth-1, root - 1);
      nodes += count;
      if (root > 0)
         printf("%8s %10"PRIu64" %10"PRIu64"\n", move_string(movelist.move[n], NULL), count, nodes);
      takeback_no_check(game);
   }
   return nodes;
}

static uint64_t legal_perft(gamestate_t *game, int depth, int root)
{
   movelist_t movelist;
#ifdef DEBUGMODE
   int me = game->side_to_move;
#endif
   uint64_t nodes = 0;
   int n;

   if (abort_search) return 0;

   if (depth == 0) return 1;

   generate_legal_moves(&movelist, game->board, game->side_to_move);
   if (depth == 1 && root <= 0) return movelist.num_moves;

   for (n=0; n<movelist.num_moves; n++) {
      uint64_t count = 0;

      playmove(game, movelist.move[n]);
      assert(!move_leaves_player_in_check(game, movelist.move[n], me));
      count = legal_perft(game, depth-1, root - 1);
      nodes += count;
      if (root > 0)
         printf("%8s %10"PRIu64" %10"PRIu64"\n", move_string(movelist.move[n], NULL), count, nodes);
      takeback_no_check(game);
   }
   return nodes;
}

static void test_movegen(bool legal)
{
   gamestate_t *game = NULL;
   int n = 0;
   while (perftests[n].fen) {
      uint64_t nodes;
      game = create_game();
      start_new_game(game);

      setup_fen_position(game, perftests[n].fen);
      printf(".");
      fflush(stdout);
      if (legal)
         nodes = legal_perft(game, perftests[n].depth, 0);
      else
         nodes = perft(game, perftests[n].depth, 0);
      if (nodes != perftests[n].nodes) {
         printf("\n");
         printf("*** Failed at position %d (%s):\n", n, perftests[n].fen);
         printf("    Expected %"PRIu64" nodes, got %"PRIu64" nodes\n", perftests[n].nodes, nodes);
         return;
      }

      end_game(game);
      n++;
   }
   printf("\nOk.\n");
}

static void test_checking_movegen(void)
{
   const char *fen = "1B5N/8/3P4/R1N1k3/1Q6/4K3/3P4/4R3 w - - 0 1";
   gamestate_t *game = NULL;
   movelist_t movelist;

   game = create_game();
   start_new_game(game);

   setup_fen_position(game, fen);

   print_board(game->board);
   generate_checking_moves(&movelist, game->board, game->side_to_move, false);
   for (int k=0; k<movelist.num_moves; k++)
      printf("%s ", move_string(movelist.move[k], NULL));
   printf("\n");

   end_game(game);
}

static void test_benchmark(int depth)
{
   gamestate_t *game = NULL;
   int n = 0;
   uint64_t nodes = 0;
   uint64_t t = get_timer();
   unsigned long long int moves_searched = 0;
#ifdef TRACK_PRUNING_STATISTICS
   memset(branches_pruned_by_move, 0, sizeof branches_pruned_by_move);
#endif

   while (benchtests[n].fen) {
      game = create_game();
      start_new_game(game);
      game->random_ok = false;
      game->output_iteration = NULL;
      game->uci_output = NULL;
      game->xboard_output = NULL;

      //printf("%s\n", benchtests[n].fen);
      setup_fen_position(game, benchtests[n].fen);
      printf(".");
      fflush(stdout);

#ifdef __unix__
      if (trapint) old_signal_handler = signal(SIGINT, interrupt_computer);
#endif
      computer_play(game, depth);
#ifdef __unix__
      if (trapint) signal(SIGINT, old_signal_handler);
#endif
      moves_searched = game->moves_searched;
      end_game(game);

      if (abort_search) {
         printf("\n*** Aborted");
         break;
      }

      //printf("%"PRIu64"\n", nodes);
      nodes += moves_searched;
      n++;
   }

   uint64_t tt = get_timer();

   printf("\n");
   printf("%"PRIu64" nodes searched\n", nodes);
   printf("Elapsed time %"PRIu64" ms\n", (tt - t) / 1000);
   printf("%g nodes / s\n", 1.0e6*nodes / (tt - t));

#ifdef TRACK_PRUNING_STATISTICS
   uint64_t total_pruned = 0;
   uint64_t avg_move_nr = 0;
   for (int n = 0; n<256; n++)
      total_pruned += branches_pruned_by_move[n];

   if (total_pruned) {
      for (int n = 0; n<256; n++) {
         avg_move_nr += (n+1) * branches_pruned_by_move[n];
         if (branches_pruned_by_move[n])
            printf("%3d : %8"PRIu64" (%5.2f%%)\n", n+1, branches_pruned_by_move[n], 100. * branches_pruned_by_move[n] / total_pruned);
      }

      printf("Average move number that produced a cut-off: %.2f\n", (double)avg_move_nr / total_pruned);
   }

#endif
}

#ifdef SMP
/* Test the SMP search.
 * Test position is the first from Hyatt (1994), after White's 9th move in a game
 * Mchess Pro - Cray Blitz.
 */
static void test_smp(int cores, int depth)
{
   gamestate_t *game = NULL;
   int n = get_number_of_threads();
   uint64_t t_start[2];
   uint64_t t_end[2];
   unsigned long long int moves_searched[2];
   char *fen = "r2qkbnr/ppp2p1p/2n5/3P4/2BP1pb1/2N2p2/PPPQ2PP/R1B2RK1/ b - - 2 9";//benchtests[1].fen;

   init_threads(1);
   game = create_game();
   start_new_game(game);
   game->random_ok = false;
   setup_fen_position(game, fen);
   print_board(game->board);
   printf("Single core analysis:\n");
   t_start[0] = get_timer();
   computer_play(game, depth);
   t_end[0] = get_timer();
   moves_searched[0] = game->moves_searched;
   end_game(game);

   init_threads(cores);
   printf("Analysis using %d cores:\n", cores);
   game = create_game();
   start_new_game(game);
   game->random_ok = false;
   setup_fen_position(game, fen);
   t_start[1] = get_timer();
   computer_play(game, depth);
   t_end[1] = get_timer();
   moves_searched[1] = game->moves_searched;
   end_game(game);

   init_threads(n);

   printf("\n");
   printf("Single core:     %-10"PRIu64" nodes searched\n", moves_searched[0]);
   printf("Multi-core:      %-10"PRIu64" nodes searched\n", moves_searched[1]);
   printf("Single core time %"PRIu64" ms\n", (t_end[0]-t_start[0]) / 1000);
   printf("Multi-core time  %"PRIu64" ms\n", (t_end[1]-t_start[1]) / 1000);
   printf("Relative node counts (multi/single): %.2f\n", (float)moves_searched[1] / moves_searched[0]);
   printf("Parallel speed-up (time-to-depth):   %.2f\n", (float)(t_end[0]-t_start[0]) / (t_end[1]-t_start[1]));

}
#endif

static void print_help(const char *topic)
{
   if (!topic) topic = "help";
   while (isspace(*topic)) topic++;
   if(!strlen(topic)) topic = "help";

   int n = 0;
   if (streq(topic, "all")) {
   while (help_topic[n].topic) {
      if (help_topic[n].text) {
         char *s = help_topic[n].cmd;
         if (!s) s = help_topic[n].topic;
         printf("%s\n%s\n", s, help_topic[n].text);
      }
      n++;
   }
      return;
   }
   while (help_topic[n].topic) {
      if (streq(help_topic[n].topic, topic)) {
         char *s = help_topic[n].cmd;
         if (!s) s = help_topic[n].topic;
         printf("%s\n%s\n", s, help_topic[n].text);
         if (streq(topic, "help")) break;
         return;
      }
      n++;
   }

   printf("Help topics: \n");
   n = 0;
   while (help_topic[n].topic) {
      if (n) printf(", ");
      printf("%s", help_topic[n].topic);
      n++;
   }
   printf("\n");
}

int main(int argc, char **argv)
{
   bool uci_mode = false;
   bool in_play = true;
   int my_colour = BLACK;
   gamestate_t *game = NULL;
   char input[65536];
   char prompt_str[65536];
   const char *lower_piece_str[] = {
      "p", "n", "b", "r", "q", "k"
   };
   int depth = 60;
   size_t hash_size = HASH_TABLE_SIZE;
   char *pgbook_file = NULL;
   int time_per_move = 0;
   int nps = -1;

#if defined SMP && defined _WIN32
   pthread_win32_process_attach_np();
   pthread_win32_thread_attach_np();
#endif

   buf = malloc(65536);

   printf("%s version %s, built %s\n", PROGNAME, VERSIONSTR " " ARCHSTR, __DATE__);
   printf("Type 'help' for a list of commands and help topics\n");
   initialise_jazz();

#ifdef SMP
   int ncore = get_number_of_cores();
   printf("Machine has %d core%s\n", ncore, (ncore > 1)? "s" : "");
   atexit(kill_threads);

   printf("Locks are implemented as %s\n", LOCK_DESCRIPTION);
#endif

#ifdef HAVE_READLINE
   if (stdin_is_terminal()) {
      rl_initialize();
      //rl_read_init_file(readline_path);
      using_history();
      stifle_history(256);
      //read_history(history_path);
      //atexit(write_history_file_on_exit);
   }
#endif

   int tb = initialise_tablebase("gtb", 0, 0);
   if (tb) {
      char *msg[] = {
         "No %d piece tablebases available\n",
         "Some %d piece tablebases available\n",
         "All %d piece tablebases available\n",
         "All %d piece tablebases available\n"
      };
      printf(msg[(tb & 3)], 3);
      printf(msg[((tb >> 2) & 3)], 4);
      printf(msg[((tb >> 4) & 3)], 5);
   }

   for (int n = 1; n<argc; n++) {
      if (strstr(argv[n], "-book")) {
         if (n+1 >= argc) {
            fprintf(stderr, "error: no book specified\n");
            exit(0);
         }
         n++;
         pgbook_file = strdup(argv[n]);
         printf("Using opening book %s\n", pgbook_file);
         //pgbook_file = strdup("bigbook.bin");
      } else if (strstr(argv[n], "-log")) {
         char *logfile = "jazz.log";
         if (n+1 < argc && argv[n+1][0] != '-') {
            logfile = argv[n+1];
            n++;
         }
         if (!f) f = fopen(logfile, "a");
      } else {
         fprintf(stderr, "Unknown option: %s\n", argv[n]);
         exit(0);
      }
   }

   set_default_output_function(NULL);
   //set_default_output_function(log_engine_output);
   set_xboard_output_function(log_xboard_output);

   /* Turn off buffering for stdout and stdin */
   setvbuf(stdout, NULL, _IONBF, 0);
   setvbuf(stdin, NULL, _IONBF, 0);

   /* Write log output (for debugging purposes) */
   //if (!f) f = fopen("xb.log", "a");
   game = create_game();
   game->book = open_opening_book(pgbook_file);
   set_transposition_table_size(game, hash_size);
   start_new_game(game);

   while (true) {
      input[0] = '\0';
      prompt_str[0] = '\0';
      if (show_board && game)
         print_board(game->board);
      if (prompt) {
         int n = 0;
         n += snprintf(prompt_str+n, (sizeof prompt_str) - n, "#");
         if (game) {
            n += snprintf(prompt_str+n, (sizeof prompt_str) - n, "%d", (int)game->moves_played);
            n += snprintf(prompt_str+n, (sizeof prompt_str) - n, "%s", (game->side_to_move == WHITE)?"w":"b");
            if (!in_play) n += snprintf(prompt_str+n, (sizeof prompt_str) - n, " (f)");
            if (player_in_check(game, game->side_to_move)) n += snprintf(prompt_str+n, (sizeof prompt_str) - n, "+");
         } else {
            n += snprintf(prompt_str+n, (sizeof prompt_str) - n, "[-]");
         }
         n += snprintf(prompt_str+n, (sizeof prompt_str) - n, ">");

#ifdef HAVE_READLINE
         if (!stdin_is_terminal())
#endif
         printf("%s", prompt_str);
      }
      if (!(may_ponder && game && game->ponder_move)|| keyboard_input_waiting()) {
#ifdef HAVE_READLINE
         if (stdin_is_terminal()) {
            char *readline_input = readline(prompt_str);
            if (readline_input && strlen(readline_input))
               add_history(readline_input);
            snprintf(input, sizeof input, "%s", readline_input);
            free(readline_input);
         } else
#endif
         if (!fgets(input, sizeof input, stdin))
            break;
      }
      chomp(input);
      trim(input);
      if (f) {
         fprintf(f, "< %s\n", input);
         fflush(f);
      }
      if (streq(input, "xboard") || streq(input, "xboard on")) {
         prompt = false;
         show_board = false;
         san = false;
         trapint = false;
         set_xboard_output_function(log_xboard_output);
         set_uci_output_function(NULL);
         log_xboard_output("\n");
      } else if (streq(input, "xboard off")) {
         prompt = true;
         show_board = true;
         san = true;
         trapint = true;
         log_xboard_output("\n");
      } else if (streq(input, "uci") || streq(input, "uci on")) {
         uci_mode = true;
         prompt = false;
         show_board = false;
         san = false;
         trapint = false;
         in_play = false;
         set_xboard_output_function(NULL);
         set_uci_output_function(log_xboard_output);
         log_xboard_output("\n");
         log_xboard_output("id name %s %s %s\n", PROGNAME, VERSIONSTR, ARCHSTR);
         log_xboard_output("id author Evert Glebbeek\n");
         //log_xboard_output("option name OwnBook type check default true\n");
         /* Hash size: 1MB - 1 GB */
         log_xboard_output("option name Hash type spin default 48 min 1 max 4096\n");
         log_xboard_output("option name Cores type spin default 1 min 1 max %d\n", MAX_THREADS);
         log_xboard_output("option name Ponder type check default true\n");
         log_xboard_output("uciok\n");
      } else if (streq(input, "uci off")) {
         uci_mode = true;
         set_xboard_output_function(log_xboard_output);
         set_uci_output_function(NULL);
      } else if (streq(input, "isready") && uci_mode) {
         log_xboard_output("readyok\n");
      } else if (strstr(input, "debug") == input && uci_mode) {
      } else if (strstr(input, "setoption name Hash") == input && uci_mode) {
         /* TODO: set the size of the hash table */
         size_t memory_size;
         char *s = strstr(input, "value");
         if (s) {
            sscanf(s+6, "%lu", &memory_size);

            /* Convert to bytes */
            memory_size <<= 20;

            /* Reserve default */
            if (memory_size > 10<<20)
               memory_size -= 5<<20;

            hash_size = memory_size / sizeof(hash_table_entry_t);
         }
      } else if (strstr(input, "setoption name Ponder") == input && uci_mode) {
      } else if (strstr(input, "setoption name Cores") == input && uci_mode) {
         int threads = 0;
         char *s = strstr(input, "value");
         if (s) {
            s += 6;
            sscanf(s, "%d", &threads);
            kill_threads();
            if (threads > 0)
               init_threads(threads);
            if (show_board) printf("Started %d threads\n", get_number_of_threads());
         }
      } else if (streq(input, "ucinewgame") && uci_mode) {
         if (game) {
            end_game(game);
         }
         game = create_game();
         set_transposition_table_size(game, hash_size);
         start_new_game(game);
         num_games++;
         if (f) fprintf(f, "# game %d\n", num_games);
      } else if (strstr(input, "position") && uci_mode) {
         char *p = strstr(input, " ");
         while (p && *p && p[0]==' ') p++;

         if (!game) {
            game = create_game();
            set_transposition_table_size(game, hash_size);
            start_new_game(game);
         }

         if (strstr(p, "startpos") == p) {   /* Game from start position */
            /* We need to return to the first node in the game struct and
             * then replay all the moves from the beginning to the current
             * position.
             */
            p = strstr(p, "moves");
            if (p && *p) p+=6;
            replay_game(game, p);
         } else if (strstr(p, "fen") == p) { /* Game from FEN string */
            p+=4;
            setup_fen_position(game, p);

            p = strstr(p, "moves");
            if (p && *p) p+=6;
            replay_game(game, p);
         }
      } else if (streq(input, "stop") && uci_mode) {
      } else if (strstr(input, "go") == input && uci_mode) {
         move_t move;
         const char *p = "";
         int depth = 60;
         char *s;

         /* Set defaults */
         set_time_per_move(game, 5000);
         game->movestogo = 0;
         game->time_inc[0] = game->time_inc[1] = 0;

         /* parse options */
         if ((s = strstr(input, "ponder"))) {   /* Can't ponder */
            computer_ponder(game);
            continue;
         }
         if ((s = strstr(input, "movetime"))) {
            sscanf(s+9, "%d", &time_per_move);
            set_time_per_move(game, time_per_move);
         }
         if ((s = strstr(input, "movestogo"))) {
            sscanf(s+10, "%d", &game->movestogo);
         }
         if ((s = strstr(input, "infinite"))) {
            set_infinite_time(game);
         }
         if ((s = strstr(input, "wtime"))) {
            sscanf(s+6, "%d", &game->time_left[0]);
            set_time_for_game(game);
         }
         if ((s = strstr(input, "btime"))) {
            sscanf(s+6, "%d", &game->time_left[1]);
            set_time_for_game(game);
         }
         if ((s = strstr(input, "winc"))) {
            sscanf(s+5, "%d", &game->time_inc[0]);
         }
         if ((s = strstr(input, "binc"))) {
            sscanf(s+5, "%d", &game->time_inc[1]);
         }
         if ((s = strstr(input, "mate"))) {  /* Just do a normal N-ply search instead */
            sscanf(s+5, "%d", &depth);
            set_infinite_time(game);
         }
         if ((s = strstr(input, "depth"))) {
            sscanf(s+6, "%d", &depth);
            set_infinite_time(game);
         }

         game->check_keyboard = uci_keyboard_input_on_move;
         computer_play(game, depth);

         move = game->move_list[game->moves_played-1];
         if (is_promotion_move(move)) {
            p = lower_piece_str[get_promotion_piece(move) & PIECE];
         }
         if (game->ponder_move) {
            const char *pp = "";
            move_t pmove = game->ponder_move;
            if (is_promotion_move(pmove))
               pp = lower_piece_str[get_promotion_piece(pmove) & PIECE];
            printf("bestmove %s%s%s ponder %s%s%s\n",
                  square_str[get_move_origin(move)], square_str[get_move_destination(move)], p,
                  square_str[get_move_origin(pmove)], square_str[get_move_destination(pmove)], pp);
         } else {
            printf("bestmove %s%s%s\n", square_str[get_move_origin(move)],
                  square_str[get_move_destination(move)], p);
         }
      } else if (strstr(input, "ponderhit") && uci_mode) {  /* Nothing special yet... */
      } else if (strstr(input, "help")) {
         print_help(input+4);
      } else if (strstr(input, "protover")) {
         log_xboard_output("feature setboard=1"
                                       " smp=1"
                                      " time=1"
                                    " sigint=0"
                                    " colors=0"
                                      " ping=1"
                                    " memory=1"
                                   " analyze=1"
                                     " pause=1"
                                       " nps=1"
#ifdef WANT_GTB
                                       " egt=\"gaviota\""
#endif
                                    " myname=\"%s %s\""
                                 " myversion=\"[%s]\"\n", PROGNAME, VERSIONSTR, VERSIONSTR " " __DATE__ " " ARCHSTR);
         log_xboard_output("feature option=\"Opening book (polyglot) -file %s\"\n", pgbook_file?pgbook_file:"");
#ifdef WANT_GTB
         log_xboard_output("feature option=\"GaviotaTbPath -file .\"\n");
#endif
         log_xboard_output("feature done=1\n");
      } else if (streq(input, "new")) {
         if (game) {
            end_game(game);
         }
         game = create_game();
         game->book = open_opening_book(pgbook_file);
         set_transposition_table_size(game, hash_size);
         start_new_game(game);
         my_colour = BLACK;
         depth = 60;
         num_games++;
         in_play = true;
         if (f) fprintf(f, "# game %d\n", num_games);
      } else if (strstr(input, "accepted") == input) {
      } else if (strstr(input, "rejected") == input) {
      } else if (strstr(input, "option") == input) {
         log_xboard_output("# %s\n", input);
         if (strstr(input+7, "Opening book (polyglot)")) {
            char *s = strstr(input, "=");
            char *eol = input + strlen(input)-1;
            if (s) s++;

            /* Strip leading and trailing spaces */
            while (*s && isspace(*s)) s++;
            while (isspace(*eol)) { *eol='\0'; eol--; }

            log_xboard_output("# %s\n", s);

            free(pgbook_file);
            pgbook_file = strdup(s);

            if (game) {
               close_opening_book(game->book);
               game->book = open_opening_book(pgbook_file);
            }
         } else if (strstr(input+7, "GaviotaTbPath")) {
            char *s = strstr(input, "=");
            char *eol = input + strlen(input)-1;
            if (s) s++;

            /* Strip leading and trailing spaces */
            while (*s && isspace(*s)) s++;
            while (isspace(*eol)) { *eol='\0'; eol--; }
            add_tablebase_path(s);
         }
      } else if (strstr(input, "egtpath gaviota") == input) {
         char *s = input+16;
         while (*s && isspace(*s)) s++;
         add_tablebase_path(s);

         int tb = tablebase_flags();
         if (tb && prompt) {
            char *msg[] = {
               "No %d piece tablebases available.\n",
               "Some %d piece tablebases available.\n",
               "All %d piece tablebases available.\n",
               "All %d piece tablebases available.\n"
            };
            printf(msg[(tb & 3)], 3);
            printf(msg[((tb >> 2) & 3)], 4);
            printf(msg[((tb >> 4) & 3)], 5);
         }
      } else if (strstr(input, "egtbpath") == input) {
         char *s = input+8;
         while (*s && isspace(*s)) s++;
         add_tablebase_path(s);

         int tb = tablebase_flags();
         if (tb) {
            char *msg[] = {
               "No %d piece tablebases available.\n",
               "Some %d piece tablebases available.\n",
               "All %d piece tablebases available.\n",
               "All %d piece tablebases available.\n"
            };
            printf(msg[(tb & 3)], 3);
            printf(msg[((tb >> 2) & 3)], 4);
            printf(msg[((tb >> 4) & 3)], 5);
         } else {
            printf("No endgame tablebases available.\n");
         }

      } else if (strstr(input, "book") == input) {
            char *s = input+4;
            char *eol = input + strlen(input)-1;
            if (s) s++;

            /* Strip leading and trailing spaces */
            while (*s && isspace(*s)) s++;
            while (isspace(*eol)) { *eol='\0'; eol--; }

            if (streq(s, "off")) {
               free(pgbook_file);
               pgbook_file = NULL;
               if (game) {
                  close_opening_book(game->book);
                  game->book = NULL;
               }
            } else {
               free(pgbook_file);
               pgbook_file = strdup(s);

               if (game) {
                  close_opening_book(game->book);
                  game->book = open_opening_book(pgbook_file);
               }
            }
      } else if (strstr(input, "memory") == input) {
         unsigned long int memory_size;
         sscanf(input+7, "%lu", &memory_size);

         /* Convert to bytes */
         memory_size <<= 20;

         /* Reserve default */
         if (memory_size > 10<<20)
            memory_size -= 5<<20;

         hash_size = memory_size / sizeof(hash_table_entry_t);
         if (game)
            set_transposition_table_size(game, hash_size);
      } else if (strstr(input, "analyze") || strstr(input, "analyse")) {
         if (game) {
            in_play = false;
            set_xboard_output_function(log_xboard_output);
            game->analysing = true;
         }
      } else if (strstr(input, "force setboard")) {
         char *p = strstr(input+7, " ");
         while (p && *p && p[0]==' ') p++;

         if (!game) {
            game = create_game();
            game->book = open_opening_book(pgbook_file);
            start_new_game(game);
         }

         setup_fen_position(game, p);
         in_play = false;
      } else if (strstr(input, "force") == input) {
         in_play = false;
      } else if (strstr(input, "undo") == input) {
         takeback(game);
      } else if (strstr(input, "remove") || strstr(input, "takeback")) {
         takeback(game);
         takeback(game);
      } else if (strstr(input, "setboard") == input) {
         char *p = strstr(input, " ");
         while (p && *p && p[0]==' ') p++;

         if (!game) {
            game = create_game();
            game->book = open_opening_book(pgbook_file);
            start_new_game(game);
         }

         setup_fen_position(game, p);
      } else if (strstr(input, "seteval") == input) {
         /* Set evaluation parameter, for parameter tuning */
         char *name = input+8;
         while (*name && isspace(*name)) name++;
         char *s;
         s = strstr(name, "=");
         *s = '\0'; s++;
         while (*s && isspace(*s)) s++;
         int value = 0;
         sscanf(s, "%d", &value);
         if (strstr(name, "PAWN_BASE")) {
            PAWN_BASE = value;
         } else if (strstr(name, "MINOR_BASE")) {
            MINOR_BASE = value;
         } else if (strstr(name, "ROOK_BASE")) {
            ROOK_BASE = value;
         } else if (strstr(name, "QUEEN_BASE")) {
            QUEEN_BASE = value;
         } else if (strstr(name, "EXCHANGE_BASE")) {
            EXCHANGE_BASE = value;
         } else if (strstr(name, "MEXCHANGE_BASE")) {
            MEXCHANGE_BASE = value;
         } else if (strstr(name, "EVAL_BISHOP_PAIR")) {
            EVAL_BISHOP_PAIR = value;
         } else if (strstr(name, "ROOK_ADVANTAGE_BASE")) {
            ROOK_ADVANTAGE_BASE = value;
         } else if (strstr(name, "MINOR_ADVANTAGE_BASE")) {
            MINOR_ADVANTAGE_BASE = value;
         } else if (strstr(name, "KNIGHT_VALUE_SCALE")) {
            KNIGHT_VALUE_SCALE = value;
         } else if (strstr(name, "ROOK_VALUE_SCALE")) {
            ROOK_VALUE_SCALE = value;
         } else if (strstr(name, "RAZOR_MARGIN_UNIT")) {
            RAZOR_MARGIN_UNIT = value;
         } else if (strstr(name, "FUTILITY1")) {
            FUTILITY1 = value;
         } else if (strstr(name, "FUTILITY2")) {
            FUTILITY2 = value;
         } else if (strstr(name, "FUTILITY3")) {
            FUTILITY3 = value;
         } else if (strstr(name, "FUTILITY4")) {
            FUTILITY4 = value;
         } else if (strstr(name, "FUTILITY5")) {
            FUTILITY5 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST0")) {
            PAWN_COUNT_VALUE_ADJUST0 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST1")) {
            PAWN_COUNT_VALUE_ADJUST1 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST2")) {
            PAWN_COUNT_VALUE_ADJUST2 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST3")) {
            PAWN_COUNT_VALUE_ADJUST3 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST4")) {
            PAWN_COUNT_VALUE_ADJUST4 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST5")) {
            PAWN_COUNT_VALUE_ADJUST5 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST6")) {
            PAWN_COUNT_VALUE_ADJUST6 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST7")) {
            PAWN_COUNT_VALUE_ADJUST7 = value;
         } else if (strstr(name, "PAWN_COUNT_VALUE_ADJUST8")) {
            PAWN_COUNT_VALUE_ADJUST8 = value;
         }

         initialise_evaluation();
      } else if (strstr(input, "lperft") == input) {
         if (game) {
            int depth = 6;
            int root = 0;
            char *s = input + 6;
            while (*s && isspace(*s)) s++;
            if (*s) {
               sscanf(s, "%d", &depth);
               while(*s && isdigit(*s)) s++;
            }
            while (*s && isspace(*s)) s++;
            if (*s) {
               sscanf(s, "%d", &root);
            }
#ifdef __unix__
            if (trapint) old_signal_handler = signal(SIGINT, interrupt_computer);
#endif
            abort_search = false;
            uint64_t t = get_timer();
            for (int n = 1; n<depth+1; n++) {
               uint64_t nodes = legal_perft(game, n, root);
               uint64_t tt = get_timer();

               if (tt == t) tt++;

               if (abort_search) break;
               printf("%2d %10lld %5.2f %12.2fnps\n", n, (long long int)nodes,
                     (double)(tt - t)/1000000.0,nodes*1.0e6/(tt-t));

               t = tt;
            }
#ifdef __unix__
            if (trapint) signal(SIGINT, old_signal_handler);
#endif
         }
      } else if (strstr(input, "perft") == input) {
         if (game) {
            int depth = 6;
            int root = 0;
            char *s = input + 5;
            while (*s && isspace(*s)) s++;
            if (*s) {
               sscanf(s, "%d", &depth);
               while(*s && isdigit(*s)) s++;
            }
            while (*s && isspace(*s)) s++;
            if (*s) {
               sscanf(s, "%d", &root);
            }
#ifdef __unix__
            if (trapint) old_signal_handler = signal(SIGINT, interrupt_computer);
#endif
            abort_search = false;
            uint64_t t = get_timer();
            for (int n = 1; n<depth+1; n++) {
               uint64_t nodes = perft(game, n, root);
               uint64_t tt = get_timer();

               if (tt == t) tt++;

               if (abort_search) break;
               printf("%2d %10lld %5.2f %12.2fnps\n", n, (long long int)nodes,
                     (double)(tt - t)/1000000.0,nodes*1.0e6/(tt-t));

               t = tt;
            }
#ifdef __unix__
            if (trapint) signal(SIGINT, old_signal_handler);
#endif
         }
      } else if (strstr(input, "test movegen") == input) {
         test_movegen(false);
      } else if (strstr(input, "test legal movegen") == input) {
         test_movegen(true);
      } else if (strstr(input, "test checkmovegen") == input) {
         test_checking_movegen();
      } else if (strstr(input, "test benchmark") == input) {
         int depth = 10;
         char *s = input + 15;
         while (*s && isspace(*s)) s++;
         if (*s) sscanf(s, "%d", &depth);
         //printf("Benchmark %d\n", depth);
         test_benchmark(depth);
      } else if (strstr(input, "test smp") == input) {
#ifdef SMP
         int cores = ncore/2;
         int depth = 12;
         char *s = input + 8;
         while (*s && isspace(*s)) s++;
         if (*s == '-') {
            if (*s) sscanf(s, "- %d", &depth);
         } else {
            if (*s) sscanf(s, "%d %d", &cores, &depth);
         }
         test_smp(cores, depth);
#endif
      } else if (strstr(input, "test") == input) {
         printf("Unknown test: %s\n", input+4);
      } else if (strstr(input, "sd")) {
         sscanf(input, "sd %d", &depth);
      } else if (strstr(input, "threads") || strstr(input, "cores")) {
#ifdef SMP
         int threads = 0;
         char *s = input;
         while (*s && isalpha(*s)) s++;
         sscanf(s, "%d", &threads);
         kill_threads();
         if (threads > 0)
            init_threads(threads);

         if (show_board) printf("Started %d threads\n", get_number_of_threads());
#endif
      } else if (strstr(input, "go")) {
         my_colour = game->side_to_move;
         in_play = true;

         /* Correctly set the number of moves to go */
         if (game->movestotc) {
            size_t moves_played = game->moves_played / 2;
            if (my_colour == WHITE)
               moves_played += (game->moves_played % 2);
            moves_played %= game->movestotc;
            game->movestogo = game->movestotc - moves_played;
         }
      } else if (strstr(input, "st inf")) {
         if (game) set_infinite_time(game);
      } else if (strstr(input, "st ")) {
         float tpm = 5;
         sscanf(input+3, "%g", &tpm);
         time_per_move = tpm * 1000;
         set_time_per_move(game, time_per_move);
         if (nps > 0) game->max_nodes = nps * time_per_move;
      } else if (strstr(input, "computer") == input) {
      } else if (strstr(input, "name") == input) {
      } else if (strstr(input, "rating") == input) {
      } else if (strstr(input, "random")) {
         sgenrand(time(NULL));
      } else if (strstr(input, "easy") || streq(input, "ponder off")) {
         may_ponder = false;
      } else if (strstr(input, "hard") || streq(input, "ponder on")) {
         may_ponder = true;
      } else if (strstr(input, "post") == input) {
         set_xboard_output_function(log_xboard_output);
      } else if (strstr(input, "nopost") == input) {
         set_xboard_output_function(NULL);
      } else if (strstr(input, "?") == input) {
      } else if (strstr(input, "hint") == input) {
         if (game && game->ponder_move) {
            log_xboard_output("Hint: %s\n", move_string(game->ponder_move, NULL));
         }
      } else if (strstr(input, "otim") == input) {
      } else if (strstr(input, "ping") == input) {
         log_xboard_output("pong %s\n", input+5);
      } else if (strstr(input, "result") == input) {
      } else if (strstr(input, "time") == input) {
         float milliseconds;
         sscanf(input+5, "%g", &milliseconds);
         milliseconds *= 10;

         /* Reserve a bit of time for when we reach the time control, so we don't get penalised by a time
          * loss. We don't need this for a Fischer clock.
          */
         if (milliseconds > TIME_BUFFER)
            milliseconds -= TIME_BUFFER;

         game->time_left[0] = game->time_left[1] = milliseconds;
         set_time_for_game(game);
      } else if (strstr(input, "nps")) {
         /* This is a convoluted mess... */
         sscanf(input+4, "%d", &nps);
         /* If we have a fixed time per move, then we can set a maximum number of nodes.
          * Otherwise... things get complicated.
          */
         game->max_nodes = 0;
         if (time_per_move > 0 && nps > 0) game->max_nodes = nps * time_per_move;
      } else if (strstr(input, "maxnodes")) {
         unsigned int nodes = 0;
         sscanf(input+9, "%u", &nodes);
         game->max_nodes = nodes;
      } else if (strstr(input, "level") == input) {
         int moves, milliseconds;
         float minutes, seconds, inc;
         /* Set defaults */
         set_time_per_move(game, 5000);
         game->movestotc = 0;
         game->movestogo = 0;
         game->time_inc[0] = game->time_inc[1] = 0;

         if (strstr(input, ":")) {
            sscanf(input+6, "%d %g:%g %g", &moves, &minutes, &seconds, &inc);
         } else {
            sscanf(input+6, "%d %g %g", &moves, &minutes, &inc);
            seconds = 0;
         }
         seconds += minutes*60;
         milliseconds = seconds * 1000;

         /* Reserve a bit of time for when we reach the time control, so we don't get penalised by a time
          * loss. We don't need this for a Fischer clock.
          */
         if (inc == 0 && milliseconds > TIME_BUFFER)
            milliseconds -= TIME_BUFFER;

         game->movestotc = moves;
         game->movestogo = moves;
         game->time_inc[0] = game->time_inc[1] = inc*1000;
         game->time_left[0] = game->time_left[1] = milliseconds;
         set_time_for_game(game);
      } else if (streq(input, "quit") || streq(input, "exit") || streq(input, "bye")) {
         if (game)
            end_game(game);
         free(buf);
         exit(0);
      } else if (streq(input, "eval")) {
         if (game) {
            game->root_board = game->board;
            printf("Static evaluation: %d\n", static_evaluation(game, game->side_to_move, -CHECKMATE, CHECKMATE));
         }
      } else if (strstr(input, "prompt on") == input) {
         prompt = true;
      } else if (strstr(input, "prompt off") == input) {
         printf("\n");
         prompt = false;
      } else if (strstr(input, "unicode on") == input) {
         unicode_board = true;
      } else if (strstr(input, "unicode off") == input) {
         unicode_board = false;
      } else if (strstr(input, "colour on") == input) {
         ansi_colours = true;
      } else if (strstr(input, "colour off") == input) {
         ansi_colours = false;
      } else if (strstr(input, "board on") == input) {
         show_board = true;
      } else if (strstr(input, "board off") == input) {
         show_board = false;
      } else if (streq(input, "board")) {
         print_bitboards(game->board);
         print_board(game->board);
      } else if (streq(input, "fen")) {
         if (game)
         printf("%s\n", make_fen_string(game, NULL));
      } else if (streq(input, "moves")) {
         if (game) {
            movelist_t movelist;
            generate_legal_moves(&movelist, game->board, game->side_to_move);
            int k;
            printf("%d moves\n", movelist.num_moves);
            for (k=0; k<movelist.num_moves; k++)
               printf("%s ", move_string(movelist.move[k], NULL));
         }
         printf("\n");
      } else if (streq(input, "pseudomoves")) {
         if (game) {
            movelist_t movelist;
            generate_moves(&movelist, game->board, game->side_to_move);
            int k;
            printf("%d moves\n", movelist.num_moves);
            for (k=0; k<movelist.num_moves; k++)
               printf("%s ", move_string(movelist.move[k], NULL));
         }
         printf("\n");
      } else if (strstr(input, "readfen") || strstr(input, "readepd")) {
         FILE *f = fopen(input+8, "r");
         if (f) {
            char s[4096];
            fgets(s, sizeof s, f);
            fclose(f);

            if (!game) {
               game = create_game();
               game->book = open_opening_book(pgbook_file);
               start_new_game(game);
            }

            setup_fen_position(game, s);
            print_board(game->board);
         } else {
            printf("Can't open file: %s\n", input+8);
         }
      } else if (prompt && input[0] == '!') {
         system(input+1);
      } else {
         if (input[0] && !input_move(game, input)) {
            log_xboard_output("Illegal move: %s\n", input);
         }
      }

      if (game && in_play) {
         game->check_keyboard = keyboard_input_on_move;
         if (game->side_to_move == my_colour) {
            const char *p = "";
            move_t move;
            bool played;

            /* Check whether NPS time control is sane. */
            if (nps>-1 && !time_per_move) {
               log_xboard_output("telluser Warning: specifying NPS without fixed time per move is not implemented\n");
               game->max_nodes = nps;
            }

            game->pondering = false;
            log_engine_output("Searching position %s\n", make_fen_string(game, NULL));

#ifdef __unix__
            if (trapint) old_signal_handler = signal(SIGINT, interrupt_computer);
#endif
            played = computer_play(game, depth);
#ifdef __unix__
            if (trapint) signal(SIGINT, old_signal_handler);
#endif
            if (played) {
               move = game->move_list[game->moves_played-1];
               if (is_promotion_move(move)) {
                  p = lower_piece_str[get_promotion_piece(move) & PIECE];
               }
               if (san) {
                  movelist_t movelist;
                  generate_legal_moves(&movelist, game->board, game->side_to_move);
                  log_xboard_output("move %s\n", short_move_string(move, &movelist, NULL));
               } else
                  log_xboard_output("move %s%s%s\n", square_str[get_move_origin(move)],
                        square_str[get_move_destination(move)], p);
               if (game->movestogo) {
                  game->movestogo--;
                  if (!game->movestogo)
                     game->movestogo = game->movestotc;
               }
            } else {
               if (root_is_mate) {
                  log_xboard_output("%s\n", game->side_to_move ? "1-0 {White mates}" : "0-1 {Black mates}");
               } else {
                  if (game->fifty_counter[game->moves_played] >= 100) {
                     log_xboard_output("1/2-1/2 {Draw by 50 move rule}\n");
                  } else if (draw_by_repetition(game)) {
                     log_xboard_output("1/2-1/2 {Draw by repetition rule}\n");
                  } else {
                     log_xboard_output("1/2-1/2 {Stalemate}\n");
                  }
               }
            }
         } else {    /* Ponder ? */
            if (may_ponder && ponder_ok && game->ponder_move)
               computer_ponder(game);
         }
      }

      if (game && game->analysing) {
         game->check_keyboard = keyboard_input_analyse;
         computer_analyse(game);
      }
   }

   if (prompt)
      printf("\n");

   return 0;
}

