/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <ctype.h>
#include "move.h"
#include "names.h"
#include "pieces.h"
#include "squares.h"
#include "square.h"
#include "game.h"
#include "movegen.h"

static bool move_is_long_algebraic(const char *move)
{
   if (strlen(move) < 4) return false;

   if (isalpha(move[0]) && isalpha(move[2]) && isdigit(move[1]) && isdigit(move[3]))
      return true;

   return false;
}

move_t move_string_to_move(const gamestate_t *game, const char *move_str)
{
   movelist_t movelist;
   int src = 0, dest = 0, ppiece = 0;
   char *s;
   int n;
   move_t move;

   /* First, generate the list of moves for this position */
   generate_legal_moves(&movelist, game->board, game->side_to_move);

   move = 0;

   /* Test if the move is simplified long algebraic (from-to) */
   if (move_is_long_algebraic(move_str)) {
      const char *s = move_str;
      int col = *s - 'a'; s++;
      int row = *s - '1'; s++;
      int from = pack_row_file(row, col);
      col = *s - 'a'; s++;
      row = *s - '1'; s++;
      int to = pack_row_file(row, col);

      n = validate_move(&movelist, from, to);
      if (n<0)
         return move;
      move = movelist.move[n];
      if (is_promotion_move(move)) {
         switch (*s) {
            case 'q':
               set_promotion_piece(&move, get_move_player(move) | QUEEN);
               break;
            case 'r':
               set_promotion_piece(&move, get_move_player(move) | ROOK);
               break;
            case 'n':
               set_promotion_piece(&move, get_move_player(move) | KNIGHT);
               break;
            case 'b':
               set_promotion_piece(&move, get_move_player(move) | BISHOP);
               break;
         }
      }

      return move;
   }

   //printf("Input: %s\n", move_str);
   /* Now, work out source and destination squares from the move */
   if (move_str[0] == 'O') {     // Castling
      if (game->side_to_move == WHITE) {
         src = E1;
         dest = G1;
         if (strlen(move_str) >= 5) {
            dest = C1;
         }
      } else {
         src = E8;
         dest = G8;
         if (strlen(move_str) >= 5) {
            dest = C8;
         }
      }
   } else if (isupper(move_str[0])) {  // Piece
      //printf("Piece move: %s\n", move_str);

      if (move_str[3] == '-' || move_str[3] == 'x') { // Long notation
         src = pack_row_file(move_str[2] - '1', move_str[1] - 'a');
         dest = pack_row_file(move_str[5] - '1', move_str[4] - 'a');
      } else {                                        // Short notation
         /* Find piece type */
         switch (move_str[0]) {
            case 'K':
               ppiece = KING;
               break;
            case 'Q':
               ppiece = QUEEN;
               break;
            case 'R':
               ppiece = ROOK;
               break;
            case 'B':
               ppiece = BISHOP;
               break;
            case 'N':
               ppiece = KNIGHT;
               break;
         }

         /* What type of move is this? */
         if (isdigit(move_str[2])) {   // Unambiguous regular move
            dest = pack_row_file(move_str[2] - '1', move_str[1] - 'a');
            for (n=0; n<movelist.num_moves; n++) {
               if ((get_move_piece(movelist.move[n])&PIECE) == ppiece && get_move_destination(movelist.move[n])==dest){
                  move = movelist.move[n];
                  return move;
               }
            }
         } else if (move_str[1] == 'x') { // Unambiguous capture
            dest = pack_row_file(move_str[3] - '1', move_str[2] - 'a');
            for (n=0; n<movelist.num_moves; n++) {
               if (get_move_piece(movelist.move[n])&ppiece && get_move_destination(movelist.move[n])==dest){
                  move = movelist.move[n];
                  return move;
               }
            }
         } else if (isalpha(move_str[1])) {  // Use file to disambiguate
            if (move_str[2] == 'x')          // Capture
               dest = pack_row_file(move_str[4] - '1', move_str[3] - 'a');
            else
               dest = pack_row_file(move_str[3] - '1', move_str[2] - 'a');

            for (n=0; n<movelist.num_moves; n++) {
               if (get_move_piece(movelist.move[n])&ppiece && get_move_destination(movelist.move[n]) == dest
                     &&
                   unpack_file(get_move_origin(movelist.move[n])) == move_str[1]-'a') {
                  move = movelist.move[n];
                  return move;
               }
            }
         } else if (isdigit(move_str[1])) {  // Use row to disambiguate
            if (move_str[2] == 'x')          // Capture
               dest = pack_row_file(move_str[4] - '1', move_str[3] - 'a');
            else
               dest = pack_row_file(move_str[3] - '1', move_str[2] - 'a');

            for (n=0; n<movelist.num_moves; n++) {
               if (get_move_piece(movelist.move[n])&ppiece && get_move_destination(movelist.move[n]) == dest
                     &&
                   unpack_rank(get_move_origin(movelist.move[n])) == move_str[1]-'1') {
                  move = movelist.move[n];
                  return move;
               }
            }
         }

      }

   } else {                            // Pawn
      //printf("Pawn move: %s\n", move_str);

      if (move_str[2] == '-' || move_str[2] == 'x') { // Long notation
         src = pack_row_file(move_str[1] - '1', move_str[0] - 'a');
         dest = pack_row_file(move_str[4] - '1', move_str[3] - 'a');
         if (move_str[5] == '=') {     // Promotion
            switch (move_str[6]) {
               case 'Q':
                  ppiece = QUEEN;
                  break;
               case 'R':
                  ppiece = ROOK;
                  break;
               case 'B':
                  ppiece = BISHOP;
                  break;
               case 'N':
                  ppiece = KNIGHT;
                  break;
            }
         }
      } else { // Short notation
         s = strstr(move_str, "=");
         if (s) {                // Promotion
            //printf("Promotion: %s %c\n", move_str, s[1]);
            switch (s[1]) {
               case 'Q':
                  ppiece = QUEEN;
                  break;
               case 'R':
                  ppiece = ROOK;
                  break;
               case 'B':
                  ppiece = BISHOP;
                  break;
               case 'N':
                  ppiece = KNIGHT;
                  break;
            }
         }

         /* Push or capture? */
         if (isdigit(move_str[1])) {   // Simple pawn push
            dest = pack_row_file(move_str[1] - '1', move_str[0] - 'a');
            for (n=0; n<movelist.num_moves; n++) {
               if ((get_move_piece(movelist.move[n])&PIECE)==PAWN && get_move_destination(movelist.move[n]) == dest) {
                  move = movelist.move[n];
                  if (!is_promotion_move(move) || get_promotion_piece(move)&ppiece)
                     return move;
               }
            }
         } else {                      // Capture
            dest = pack_row_file(move_str[3] - '1', move_str[2] - 'a');
            for (n=0; n<movelist.num_moves; n++) {
               if ((get_move_piece(movelist.move[n])&PIECE)==PAWN && get_move_destination(movelist.move[n]) == dest
                     &&
                   unpack_file(get_move_origin(movelist.move[n])) == move_str[0]-'a') {
                  move = movelist.move[n];
                  if (!is_promotion_move(move) || get_promotion_piece(move)&ppiece)
                     return move;
               }
            }
         }
      }
   }

   for (n=0; n<movelist.num_moves; n++) {
      //printf("%c%c-%c%c %s\n",
      //   unpack_file(src)+'a', unpack_rank(src)+'1',
      //   unpack_file(dest)+'a', unpack_rank(dest)+'1',
      //   move_string(movelist.move[n], NULL));
      if (get_move_origin(movelist.move[n]) == src && get_move_destination(movelist.move[n]) == dest) {
         move = movelist.move[n];
         if (!is_promotion_move(move) || get_promotion_piece(move) & ppiece)
            break;
      }
   }

   return move;
}

