/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <unistd.h>
#if defined _POSIX_TIMERS && _POSIX_TIMERS>0 && defined _POSIX_MONOTONIC_CLOCK
#define USE_CLOCK_GETTIME
#include <time.h>
#else
#include <sys/time.h>
#endif
#include <stdint.h>
#include <limits.h>
#include "assert.h"
#include "game.h"
#include "timer.h"
#include "keypressed.h"

/* returns the current value of the counter, in micro seconds */
uint64_t get_timer(void)
{
#ifdef USE_CLOCK_GETTIME
   struct timespec t;
   clock_gettime(0, &t);

   return t.tv_sec*1000000 + t.tv_nsec/1000; 
#else
   struct timeval t;
   gettimeofday(&t, NULL);
   
   return t.tv_sec*1000000 + t.tv_usec; 
#endif
}

/* Start the clock for the current player */
void start_clock(gamestate_t *game)
{
   assert(game);
   game->start_time = get_timer();
}

/* Get the time elapsed since the last call to start_clock(), in ms */
int peek_timer(const gamestate_t *game)
{
   return (get_timer() - game->start_time) / 1000;
}

/* Calculate time allocated for this a move */
int get_game_time_for_move(const gamestate_t *game)
{
   int num_moves = game->movestogo;
   int time_for_move;

   if (game->pondering) {
      if (!keyboard_input_waiting())
         return INT_MAX;
      else
         return 0;
   }

   if (game->time_per_move)
      return game->time_per_move;

   if (game->check_clock == NULL)
      return INT_MAX;


   /* No set number of moves to be played within the time limit, make some
    * sort of estimate for how many moves we may want to play in the time
    * remaining.
    */
   if (num_moves == 0) {
      //num_moves = 40 - game->root_moves_played % 40;
      if (game->root_moves_played < 20)
         num_moves = 15+game->root_moves_played;
      else
         num_moves = 20;
   }

   if (num_moves < 0) num_moves = 10;

   /* Base time for this move: the total time left divided by the number of
    * moves.
    */
   time_for_move = game->time_left[game->running_clock]/num_moves;

   if (game->time_inc[game->running_clock]) {
      time_for_move = 4*time_for_move / 5;
      time_for_move += 31*game->time_inc[game->running_clock]/32;
   }

   /* We want to preserve a small buffer to avoid time losses */
   if (num_moves < 5) 
      time_for_move *= 0.95;

   /* Adjust: we can spare at least half the increment */
#if 0
   if (game->time_left[game->running_clock] > game->time_inc[game->running_clock]/2)
      time_for_move += game->time_inc[game->running_clock]/2;

   /* We may have allocated some extra time for this move, for instance
    * because our best move failed low.
    */
   if (time_for_move + game->extra_time < game->time_left[game->running_clock])
      time_for_move += game->extra_time;

   /* If we're short on time and have an increment, then play quickly so we
    * gain some extra time to finish the game.
    */
   if (game->time_left[game->running_clock] < game->time_inc[game->running_clock] * 5)
      time_for_move = 3*game->time_inc[game->running_clock]/4;
#endif

   return time_for_move;
}

/* Time management: check if a fixed time per move has passed since the
 * clock was started.
 */
static bool check_time_per_move_clock(const gamestate_t *game)
{
   int time_passed = peek_timer(game);

   if (time_passed >= game->time_per_move)
      return true;

   return false;
}

static bool check_time_for_game(const gamestate_t *game)
{
   int time_passed = peek_timer(game);
   int time_for_move = get_game_time_for_move(game);

   /* If we've expended our time for this move, then abort */
   if (time_passed >= time_for_move)
      return true;

   return false;

   if (game->time_left[game->running_clock] >
      2*game->time_inc[game->running_clock])
      time_for_move += game->time_inc[game->running_clock];

   /* Very little time left at all */
   if (game->time_left[game->running_clock] < 500)
      return true;

   if (time_passed && game->time_left[game->running_clock] < 1000)
      return true;

   /* Check if we've used our aallocated time for this move */
   if ( time_passed >= (time_for_move+game->extra_time) )
      return true;

   return false;
}

static bool check_keyboard(const gamestate_t *game)
{
   return keyboard_input_waiting();
}

void set_time_per_move(gamestate_t *game, int msec)
{
   game->time_per_move = msec;
   game->check_clock = check_time_per_move_clock;
}

void set_ponder_timer(gamestate_t *game)
{
   game->check_clock = check_keyboard;
}

void set_infinite_time(gamestate_t *game)
{
   game->check_clock = NULL;
}

void set_time_for_game(gamestate_t *game)
{
   game->time_per_move = 0;
   game->check_clock = check_time_for_game;
}
