/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef bitboard_t_H
#define bitboard_t_H

#include <stdint.h>
#include "assert.h"
#include "square.h"
#include "squares.h"
#include "bits.h"

typedef uint64_t bitboard_t;

/* Special bitboards, used for masking and move generation:
 * The "board_empty" and "board_all" bitboards speak for themselves.
 * The "board_minus_(a|h)" boards are used in move generation when
 * shifting pawn bit boards.
 * The centre, xcentre, kwing and qwing boards can be used in tactical
 * situations to specifically look for moves in that part of the board.
 * Rank encodings are mainly used in move generation, but could also be
 * used in function evaluations (think rooks on the 7th row).
 * The "board_ep" bitboard is used in move generation: valid en-passant
 * squares occur along these rows.
 * The "castlek" and "castleq" bitboards are bitmasks that have to fit the
 * king, rook and inibit bitboards; used in move generation. Their "i"
 * counter parts mark the squares that have to be empty.
 */
static const bitboard_t board_empty   = 0x0000000000000000ll;
static const bitboard_t board_all     = 0xFFFFFFFFFFFFFFFFll;
static const bitboard_t board_edge    = 0xFF818181818181FFll;
static const bitboard_t board_minus_a = 0xFEFEFEFEFEFEFEFEll;
static const bitboard_t board_minus_h = 0x7F7F7F7F7F7F7F7Fll;
static const bitboard_t board_centre  = 0x0000001818000000ll;
static const bitboard_t board_xcentre = 0x00003C24243C0000ll;
static const bitboard_t board_offedge = 0x007e424242427e00ll;
static const bitboard_t board_kwing   = 0xF0F0F0F0F0F0F0F0ll;
static const bitboard_t board_qwing   = 0x0F0F0F0F0F0F0F0Fll;
static const bitboard_t board_dark    = 0xAA55AA55AA55AA55ll;
static const bitboard_t board_light   = 0x55AA55AA55AA55AAll;
static const bitboard_t board_rank1   = 0x00000000000000FFll;
static const bitboard_t board_rank2   = 0x000000000000FF00ll;
static const bitboard_t board_rank3   = 0x0000000000FF0000ll;
static const bitboard_t board_rank4   = 0x00000000FF000000ll;
static const bitboard_t board_rank5   = 0x000000FF00000000ll;
static const bitboard_t board_rank6   = 0x0000FF0000000000ll;
static const bitboard_t board_rank7   = 0x00FF000000000000ll;
static const bitboard_t board_rank8   = 0xFF00000000000000ll;
static const bitboard_t board_afile   = 0x0101010101010101ll;
static const bitboard_t board_bfile   = 0x0202020202020202ll;
static const bitboard_t board_cfile   = 0x0404040404040404ll;
static const bitboard_t board_dfile   = 0x0808080808080808ll;
static const bitboard_t board_efile   = 0x1010101010101010ll;
static const bitboard_t board_ffile   = 0x2020202020202020ll;
static const bitboard_t board_gfile   = 0x4040404040404040ll;
static const bitboard_t board_hfile   = 0x8080808080808080ll;
static const bitboard_t board_ep      = 0x0000FF0000FF0000ll;
static const bitboard_t board_castlek = 0x9000000000000090ll;
static const bitboard_t board_castleki= 0x6000000000000060ll;
static const bitboard_t board_castleq = 0x1100000000000011ll;
static const bitboard_t board_castleqi= 0x0E0000000000000Ell;
static const bitboard_t board_minors  = 0x6600000000000066ll;
static const bitboard_t board_knights = 0x4200000000000042ll;
static const bitboard_t board_bishops = 0x2400000000000024ll;
static const bitboard_t board_rooks   = 0x8100000000000081ll;
static const bitboard_t board_kings   = 0x1000000000000010ll;
static const bitboard_t board_white   = 0x00000000FFFFFFFFll;
static const bitboard_t board_black   = 0xFFFFFFFF00000000ll;

/* Rook castling masks: these are to be xor'ed with the rook's position
 * bitboards to move the rook to its proper square.
 * For kingside castling, this flips bits 5 & 7 (h->f), so 0xA0
 * For queenside castling, this flips bits 0 & 3 (a->d), so 0x09
 */
static const uint8_t castlek_rook_bits = 0xA0;
static const uint8_t castleq_rook_bits = 0x09;

/* Array-based indexing for rank and file bitboards */
static const bitboard_t board_rank[8] = {
   0x00000000000000FFll, 0x000000000000FF00ll,
   0x0000000000FF0000ll, 0x00000000FF000000ll,
   0x000000FF00000000ll, 0x0000FF0000000000ll,
   0x00FF000000000000ll, 0xFF00000000000000ll,
};

static const bitboard_t board_file[8] = {
   0x0101010101010101ll, 0x0202020202020202ll,
   0x0404040404040404ll, 0x0808080808080808ll,
   0x1010101010101010ll, 0x2020202020202020ll,
   0x4040404040404040ll, 0x8080808080808080ll,
};

/* Regions of the board that are north, east, south or west of a given
 * rank/file.
 */
static const bitboard_t board_north_of_rank[8] = {
   0xffffffffffffff00ll, 0xffffffffffff0000ll,
   0xffffffffff000000ll, 0xffffffff00000000ll,
   0xffffff0000000000ll, 0xffff000000000000ll,
   0xff00000000000000ll, 0x0000000000000000ll,
};

static const bitboard_t board_south_of_rank[8] = {
   0x0000000000000000ll, 0x00000000000000ffll,
   0x000000000000ffffll, 0x0000000000ffffffll,
   0x00000000ffffffffll, 0x000000ffffffffffll,
   0x0000ffffffffffffll, 0x00ffffffffffffffll,
};

static const bitboard_t board_east_of_file[8] = {
   0xfefefefefefefefell, 0xfcfcfcfcfcfcfcfcll,
   0xf8f8f8f8f8f8f8f8ll, 0xf0f0f0f0f0f0f0f0ll,
   0xe0e0e0e0e0e0e0e0ll, 0xc0c0c0c0c0c0c0c0ll,
   0x8080808080808080ll, 0x0000000000000000ll
};

static const bitboard_t board_west_of_file[8] = {
   0x0000000000000000ll, 0x0101010101010101ll,
   0x0303030303030303ll, 0x0707070707070707ll,
   0x0f0f0f0f0f0f0f0fll, 0x1f1f1f1f1f1f1f1fll,
   0x3f3f3f3f3f3f3f3fll, 0x7f7f7f7f7f7f7f7fll
};

/* Diagonal masks */
static const bitboard_t board_a1h8_diagonal[16] = {
   H1_MASK,
   G1_MASK|H2_MASK,
   F1_MASK|G2_MASK|H3_MASK,
   E1_MASK|F2_MASK|G3_MASK|H4_MASK,
   D1_MASK|E2_MASK|F3_MASK|G4_MASK|H5_MASK,
   C1_MASK|D2_MASK|E3_MASK|F4_MASK|G5_MASK|H6_MASK,
   B1_MASK|C2_MASK|D3_MASK|E4_MASK|F5_MASK|G6_MASK|H7_MASK,
   A1_MASK|B2_MASK|C3_MASK|D4_MASK|E5_MASK|F6_MASK|G7_MASK|H8_MASK,
   A2_MASK|B3_MASK|C4_MASK|D5_MASK|E6_MASK|F7_MASK|G8_MASK,
   A3_MASK|B4_MASK|C5_MASK|D6_MASK|E7_MASK|F8_MASK,
   A4_MASK|B5_MASK|C6_MASK|D7_MASK|E8_MASK,
   A5_MASK|B6_MASK|C7_MASK|D8_MASK,
   A6_MASK|B7_MASK|C8_MASK,
   A7_MASK|B8_MASK,
   A8_MASK,
};

static const bitboard_t board_a8h1_diagonal[16] = {
   A1_MASK,
   B1_MASK|A2_MASK,
   C1_MASK|B2_MASK|A3_MASK,
   D1_MASK|C2_MASK|B3_MASK|A4_MASK,
   E1_MASK|D2_MASK|C3_MASK|B4_MASK|A5_MASK,
   F1_MASK|E2_MASK|D3_MASK|C4_MASK|B5_MASK|A6_MASK,
   G1_MASK|F2_MASK|E3_MASK|D4_MASK|C5_MASK|B6_MASK|A7_MASK,
   H1_MASK|G2_MASK|F3_MASK|E4_MASK|D5_MASK|C6_MASK|B7_MASK|A8_MASK,
   H2_MASK|G3_MASK|F4_MASK|E5_MASK|D6_MASK|C7_MASK|B8_MASK,
   H3_MASK|G4_MASK|F5_MASK|E6_MASK|D7_MASK|C8_MASK,
   H4_MASK|G5_MASK|F6_MASK|E7_MASK|D8_MASK,
   H5_MASK|G6_MASK|F7_MASK|E8_MASK,
   H6_MASK|G7_MASK|F8_MASK,
   H7_MASK|G8_MASK,
   H8_MASK,
};

extern const bitboard_t square_bitboards[64];
extern bitboard_t white_pawn_square[64];
extern bitboard_t black_pawn_square[64];
extern bitboard_t north_pawn_attack[64];
extern bitboard_t south_pawn_attack[64];
extern bitboard_t north_pawn_attack_span[64];
extern bitboard_t south_pawn_attack_span[64];

extern void compute_pawn_bitboard_masks(void);

extern void printf_bitboard(bitboard_t b);
extern void print_bitboard_squares(bitboard_t b);

/*******************************************/
/* Bitboard manipulation routines (inline) */
/*******************************************/

/* Set a bit on the bitboard */
static inline void set_bitboard(bitboard_t *board, const int square)
{
   assert(board);
   (*board) |= ((uint64_t)1)<<square;
}

/* Clears a square on the bitboard */
static inline void unset_bitboard(bitboard_t *board, const int square)
{
   assert(board);
   bitboard_t mask = ~((uint64_t)1<<square);
   (*board) &= mask;
}

/* Returns true if a square on the bitboard is set */
static inline bool test_bitboard(const bitboard_t board, const int square)
{
   return board & ((uint64_t)1)<<square;
}

/* ANDs two bitboards */
static inline bitboard_t and_bitboards(const bitboard_t b1, const bitboard_t b2)
{
   return b1&b2;
}

/* MASKs two bitboards: set bits in first bitboard to 0 if set in second */
static inline bitboard_t mask_bitboards(const bitboard_t b1, const bitboard_t b2)
{
   return b1&~b2;
}

/* XORs two bitboards */
static inline bitboard_t xor_bitboards(const bitboard_t b1, const bitboard_t b2)
{
   return b1^b2;
}

/* ORs two bitboards */
static inline bitboard_t or_bitboards(const bitboard_t b1, const bitboard_t b2)
{
   return b1|b2;
}

/* Inverts a bitboard */
static inline bitboard_t invert_bitboard(const bitboard_t b)
{
   return ~b;
}

/* Retrieve one row from a bitboard */
static inline uint8_t get_bitboard_row(const bitboard_t b, const int row)
{
   return b >> (row*8);
}

/* Return a bitboard representation of a given square */
static inline bitboard_t make_bitboard_square(const int square)
{
#if defined(__LP64__)
   return ((uint64_t)1)<<square;
#else
   return square_bitboards[square];
#endif
}

/* Expand one row into a full bitboard */
static inline bitboard_t make_bitboard_row(const uint8_t data, const int row_nr)
{
   return ((uint64_t)data) << (row_nr*8);
}

/* Expand one packed file into a full bitboard */
static inline bitboard_t make_bitboard_file(const uint8_t data, const int file_nr)
{
   bitboard_t b;
   int n;
   b = 0;
   for(n=0;n<8;n++) {
      if (data & (1<<n)) {
         set_bitboard(&b, pack_row_file(n, file_nr));
      }
   }
   return b;
}

static inline bitboard_t make_bitboard_a8h1_diagonal(uint8_t data, int square)
{
   bitboard_t bb = 0;
   int diag_nr = get_a8h1_diagonal(square);
   int offset = get_a8h1_diagonal_offset(square);
   /* Find the length of the diagonal: this is the number of bits in
    * the occupancy number that we actually care about.
    */
   int diag_length = 8 - abs(7 - diag_nr);
   int n;
   for (n=0; n<diag_length; n++)
      if (data & (1<<n))
         set_bitboard(&bb, diagonal_a8h1_inverse_map[offset+n]);

   return bb;
}

static inline bitboard_t make_bitboard_a1h8_diagonal(uint8_t data, int square)
{
   bitboard_t bb = 0;
   int diag_nr = get_a1h8_diagonal(square);
   int offset = get_a1h8_diagonal_offset(square);
   /* Find the length of the diagonal: this is the number of bits in
    * the occupancy number that we actually care about.
    */
   int diag_length = 8 - abs(7 - diag_nr);
   int n;
   for (n=0; n<diag_length; n++)
      if (data & (1<<n))
         set_bitboard(&bb, diagonal_a1h8_inverse_map[offset+n]);

   return bb;
}

static inline uint8_t get_bitboard_diagonal_occupancy_number(bitboard_t bb, const int square)
{
   return ( (bb & board_a1h8_diagonal[get_a1h8_diagonal(square)]) * board_bfile) >> 57;
}

static inline uint8_t get_bitboard_antidiag_occupancy_number(bitboard_t bb, const int square)
{
   return ( (bb & board_a8h1_diagonal[get_a8h1_diagonal(square)]) * board_bfile) >> 57;
}

static inline uint8_t get_bitboard_rank_occupancy_number(bitboard_t bb, const int square)
{
   return get_bitboard_row(bb, unpack_rank(square));
}

static inline uint8_t get_bitboard_file_occupancy_number(bitboard_t bb, const int square)
{
   uint8_t occ;
   int file = unpack_file(square);

   bb = board_afile & (bb >> file);
   occ = (bb * 0x0004081020408000) >> 57;
   return occ;
}

/* Shift a bitboard to the right (that is, a->b, b->c etc.) by 1 file */
/*  and discards what gets shifted in on the left */
/* Note that shifting the board to the right implies shifting the binary */
/*  values to the left, since files to the right have a higher index */
static inline bitboard_t shift_bitboard_file_right(const bitboard_t b)
{
   return and_bitboards(board_minus_a, (b<<1));
}

/* Shift a bitboard to the left (that is, b->a, c->b etc.) by 1 file */
/*  and discards what gets shifted in on the right */
/* Note that shifting the board to the left implies shifting the binary */
/*  values to the right, since files to the left have a lower index */
static inline bitboard_t shift_bitboard_file_left(const bitboard_t b)
{
   return and_bitboards(board_minus_h, (b>>1));
}

/* Shift a bitboard up one row (that is, 1->2, 2->3 etc */
static inline bitboard_t shift_bitboard_row_up(const bitboard_t b)
{
   return b<<8;
}

/* Shift a bitboard down one row (that is, 1->2, 2->3 etc */
static inline bitboard_t shift_bitboard_row_down(const bitboard_t b)
{
   return b>>8;
}

/* Combinations of the above */
static inline bitboard_t shift_bitboard_right_up(const bitboard_t b)
{
   return shift_bitboard_row_up(shift_bitboard_file_right(b));
}

static inline bitboard_t shift_bitboard_right_down(const bitboard_t b)
{
   return shift_bitboard_row_down(shift_bitboard_file_right(b));
}

static inline bitboard_t shift_bitboard_left_up(const bitboard_t b)
{
   return shift_bitboard_row_up(shift_bitboard_file_left(b));
}

static inline bitboard_t shift_bitboard_left_down(const bitboard_t b)
{
   return shift_bitboard_row_down(shift_bitboard_file_left(b));
}

static inline bitboard_t fill_north(bitboard_t bb)
{
   bb |= bb<<8;
   bb |= bb<<16;
   bb |= bb<<32;
   return bb;
}

static inline bitboard_t fill_south(bitboard_t bb)
{
   bb |= bb>>8;
   bb |= bb>>16;
   bb |= bb>>32;
   return bb;
}

static inline bitboard_t north_pawn_attacks(bitboard_t p)
{
   return (p&board_minus_a)<<7 | (p&board_minus_h)<<9;
}

static inline bitboard_t south_pawn_attacks(bitboard_t p)
{
   return (p&board_minus_h)>>7 | (p&board_minus_a)>>9;
}

static inline bitboard_t north_pawn_step(bitboard_t p)
{
   return p << 8;
}

static inline bitboard_t south_pawn_step(bitboard_t p)
{
   return p >> 8;
}

/* Helper function: test whether the horizontal and vertical lines through
 * the square divide the given board.
 * Can be used to test whether the two kings are to either side of a pawn,
 * for instance.
 */
static inline bool square_separates_pieces(int square, bitboard_t bb)
{
   int file = unpack_file(square);
   int rank = unpack_rank(square);
   if ((board_north_of_rank[rank] & bb) && (board_south_of_rank[rank] & bb)) {
      return true;
   }

   if ((board_east_of_file[file] & bb) && (board_west_of_file[file] & bb)) {
      return true;
   }

   return false;
}
#endif
