#ifndef sbr_H
#define sbr_H

#include <stdint.h>
#include <stdbool.h>
#include "assert.h"

#define USE_INTRINSICS

static inline int bitscan16(uint16_t x) {
   assert(x);
#if defined __GNUC__ && defined USE_INTRINSICS
   return __builtin_ctz (x);
#else
#warning Using slow bitscan16
   int i = 0;
   for (x=x<<1; x; x=x<<1)
      i++;
   return i;
#endif
}

static inline int bitscan64(uint64_t bb)
{
#if defined __GNUC__ && defined USE_INTRINSICS
   return 63^__builtin_clzll(bb);
#else
#warning Using slow bitscan64
   int i = 0;
   for (bb=bb>>1; bb; bb=bb>>1)
      i++;
   return i;
#endif
}

static inline int forward_bitscan64(uint64_t bb)
{
#if defined __GNUC__ && defined USE_INTRINSICS
   return __builtin_ctzll(bb);
#else
#warning Using slow forward_bitscan64
   int i = 0;
   for (bb=bb<<1; bb; bb=bb<<1)
      i++;
   return i;
#endif
}

/* Returns true if only a single bit has been set */
static inline bool onebit64(uint64_t n)
{
   return (n & (n-1)) == 0;
}

/* Return the number of bits set on a bitboard
 * From http://chessprogramming.wikispaces.com/Population+Count
 */
static inline int popcount64(uint64_t x)
{
#if defined __GNUC__ && defined USE_INTRINSICS
    return __builtin_popcountll(x);
#else
#warning Using slow popcount64
   int count = 0;
   while (x) {
       count++;
       x &= x - 1;
   }
   return count;
#endif
}

static inline int sparse_popcount64(uint64_t x) {
   return popcount64(x);
}

/* Return the number of bits set on a bitboard
 * From http://chessprogramming.wikispaces.com/Population+Count
 */
static inline int popcount8(uint8_t x)
{
#if defined __GNUC__ && defined USE_INTRINSICS
    return __builtin_popcount(x);
#else
#warning Using slow popcount64
   int count = 0;
   while (x) {
       count++;
       x &= x - 1;
   }
   return count;
#endif
}

static inline uint64_t rol64(uint64_t x, int s)
{
   return (x<<s) | (x>>(64-s));
}

static inline uint64_t ror64(uint64_t x, int s)
{
   return (x>>s) | (x<<(64-s));
}

#endif
