/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef EVALUATE_H
#define EVALUATE_H

#include "game.h"

/* Base value of the pieces as well as material imbalance parameters */
#define PIECE_VALUE_VAR
#ifdef PIECE_VALUE_VAR
extern int PAWN_BASE;
extern int MINOR_BASE;
extern int ROOK_BASE;
extern int QUEEN_BASE;
extern int EXCHANGE_BASE;
extern int MEXCHANGE_BASE;
extern int EVAL_BISHOP_PAIR;
extern int ROOK_ADVANTAGE_BASE;
extern int MINOR_ADVANTAGE_BASE;
extern int KNIGHT_VALUE_SCALE;
extern int ROOK_VALUE_SCALE;
extern int RAZOR_MARGIN_UNIT;
extern int FUTILITY1;
extern int FUTILITY2;
extern int FUTILITY3;
extern int FUTILITY4;
extern int FUTILITY5;
#else

#define PAWN_BASE    100
#define MINOR_BASE  325
#define ROOK_BASE    550
#define QUEEN_BASE   975

/* Penalty for dropping the exchange, on top of the difference in value
 * between rook and a minor.
 */
#define EXCHANGE_BASE   50

/* The minor exchange penalty: difference between two minors and a rook */
#define MEXCHANGE_BASE  50

/* Bonus for having the bishop pair */
#define EVAL_BISHOP_PAIR      50

/* Rook vs pawn advantage */
#define ROOK_ADVANTAGE_BASE   100
#define MINOR_ADVANTAGE_BASE   90

/* Knight value adjustment multiplier (4 according to Kaufman) */
#define KNIGHT_VALUE_SCALE    4

/* Rook value adjustment multiplier (3 according to Kaufman) */
#define ROOK_VALUE_SCALE      0

/* Related: razor and futility margins */
#define RAZOR_MARGIN_UNIT  320
#define FUTILITY1            0
#define FUTILITY2          300
#define FUTILITY3          500
#define FUTILITY4          900
#define FUTILITY5         1500
#endif

extern int PAWN_COUNT_VALUE_ADJUST0;
extern int PAWN_COUNT_VALUE_ADJUST1;
extern int PAWN_COUNT_VALUE_ADJUST2;
extern int PAWN_COUNT_VALUE_ADJUST3;
extern int PAWN_COUNT_VALUE_ADJUST4;
extern int PAWN_COUNT_VALUE_ADJUST5;
extern int PAWN_COUNT_VALUE_ADJUST6;
extern int PAWN_COUNT_VALUE_ADJUST7;
extern int PAWN_COUNT_VALUE_ADJUST8;

/* Lazy evaluation margins, two of them.
 * The idea behind lazy evaluation is that the remaining evaluation terms
 * will not add up to more than the specified margin. Absolutely safe is to
 * set these to the true limit for the remaining evaluation terms, but
 * instead it might be better to make some quick and crude estimates for
 * the more expensive (and significant) terms like passed pawn and king
 * safety evaluation.
 */
#define LAZY_MARGIN1         500
#define LAZY_MARGIN2        1500

/* Bonus for an advanced duo (6th rank or beyond) */
#define EVAL_ADVANCED_DUO     600

/* Bonus for the side to move */
#define SIDE_TO_MOVE_BONUS    0

/* Penalty when castling rights have been lost since the root of the search */
#define CASTLE_LOST_PENALTY   40

/* Bonus for placing a piece on an opponent's stop square */
#define STOP_SQUARE_BONUS     10

/* Penalty for placing an own piece on a stop square */
#define STOP_SQUARE_PENALTY    7

/* Bonus for placing a piece on a strong square */
#define STRONG_SQUARE_BONUS   15

/* Penalty for number of pawn island, multiply by number of islands */
#define PAWN_ISLAND_PENALTY    5

/* Passed pawn candidate bonus */
#define CANDIDATE_BONUS        7

/* Free pawn value bonus, middle game (base) and end game (on top of base) */
#define FREE_PAWN_BONUS_MG     15
#define FREE_PAWN_BONUS_EG      0

/* Advance bonuses for free pawns, middle game (base) and end game (on top
 * of base), multiply by rank number of the free pawn, seventh rank counts
 * double.
 */
#define FREE_PAWN_ADVANCE_BONUS_MG  2
#define FREE_PAWN_ADVANCE_BONUS_EG  8

/* Parameters for passed pawn evaluation table */
#define PASSED_PAWN_CEIL            250      /* Asymptotic value of a passed pawn */
#define PASSED_PAWN_SLOPE             1.0    /* Slope of the passed pawn evaluation term */
#define PASSED_PAWN_INFLECTION        6.5    /* Inflection point where evaluation of the pawn begins to rise steeply */

/* Default values: 420 1 6.5
 * Performance vs Jazz-481 (2140 elo):
 *  485M   420 0.5  6.5: 2050 elo(-90) +/-5
 *  485M-2 350 1.0  6.5: 2127 elo(-13) +/-14
 *  485M-3 300 1.0  6.5: 2137 elo(-3)  +/-14
 *  485M-4 250 1.0  6.5: 2146 elo(+6)  +/-11
 *  485M-5 300 1.0  7.5: 2144 elo(+4)  +/-11
 *  485M-6 250 1.25 6.5: 2138 elo(-2)  +/-14
 *  485M-7 200 1.0  6.5: 2141 elo(+1)  +/-14
 */

/* Parameters for king safety evaluation table */
#define KING_SAFETY_CEIL         625         /* Asymptotic value (uncorrected) */
#define KING_SAFETY_SLOPE          0.15      /* Slope of the king safety evaluation term */
#define KING_SAFETY_INFLECTION    18         /* Inflection point where the evaluation begins to rise steeply */

/* Default parameters: 625 0.2 18
 * Performance vs. Jazz-486: (2098 elo)
 * 486M    625 0.2 18: 2104 elo(+6)
 * 486M-2  500 0.2 18: 2096 elo(-2)
 * 486M-3  625 0.1 18: 2110 elo(+12)
 * 486M-4  625 0.2 14: 2101 elo(+3)
 * 486M-5  625 0.1 20: 2085 elo(-13)
 * 486M-6  625 0.05 18: 2093 elo (-5)
 * 486M-7  625 0.15 18: 2113 elo (+15)
 */

/* Parameters for tuning the weighting of king safety with game phase */
#define KING_WEIGHT_SLOPE           1.0
#define KING_WEIGHT_INFLECTION      6.0

/* Default values: 1.0 8.0
 * Performance vs. Jazz-498 (2109 elo):
 * 498M   1.0 8.0: 2104 elo
 * 498M2  0.5 8.0: 2097 elo
 * 498M3  1.5 8.0: 2088 elo
 * 498M4  1.0 6.0: 2103 elo
 */

#define KING_SHELTER_OPEN           50
#define KING_SHELTER_OTHER_SEMI     10
#define KING_SHELTER_OWN_SEMI       20
#define KING_SHELTER_DIAG            8

/* Knight evaluation terms */
#define KNIGHT_OUTPOST_BASE         15
#define KNIGHT_OUTPOST_CENTRE        8 // divide by 16
#define KNIGHT_OUTPOST_RANK          8 // divide by 16
#define KNIGHT_STOPSQ_ATTACK         3 // or 7
#define KNIGHT_WEAK_PAWN_ATTACK      1
#define KNIGHT_NOMOVES_PENALTY      -50
#define KNIGHT_FEWMOVES_BASE1       -20
#define KNIGHT_FEWMOVES_BASE2       -10
#define KNIGHT_FEWMOVES_BASE3        0
#define KNIGHT_FEWMOVES_CENTRE       8 // divide by 16

/* Rook evaluation terms */
#define ROOK_PAWN_BASE_BONUS        20
#define ROOK_KING_CUTOFF_BONUS      20
#define ROOK_OPEN_FILE_BONUS        25
#define ROOK_HALF_OPEN_BONUS        10
#define ROOK_INTERSPAN_PENALTY       5
#define ROOK_BACKUP_FREE_PAWN_BONUS 20
#define ROOK_CONNECTED_RANK_BONUS    5
#define ROOK_CONNECTED_FILE_BONUS   10
#define ROOK_MOBILITY_WEIGHT        64

/* Bishop evaluation terms */
#define BISHOP_OUTPOST_BASE         15
#define BISHOP_BLOCKED_PENALTY      15
#define BISHOP_SEMI_BLOCKED_PENALTY 10
#define BISHOP_BLOCKING_PENALTY      5

/* Queen evaluation terms */
#define QUEEN_PAWN_BASE_BONUS         5
#define QUEEN_OPEN_FILE_BONUS        10
#define QUEEN_HALF_OPEN_BONUS         5
#define QUEEN_BACKUP_FREE_PAWN_BONUS 10

/* Colour weakness: the difference between control of the light and dark
 * squares. This may be tricky to implement, because if white (say) has a
 * colour weakness on the light squares, then black should try to attack
 * over the light squares - but that may be picked up as colour weakness on
 * the dark squares for black!
 */
#define COLOUR_WEAKNESS_PENALTY      0

/* Pinned pieces are penalised fairly heavily */
#define PINNED_PIECE_PENALTY         0

extern int futility_margins[6];

extern void load_dynamic_psq(gamestate_t *game);

extern int static_evaluation(gamestate_t *game, int player, const int alpha, const int beta);

extern void initialise_evaluation(void);

static inline bool mate_score(int score)
{
   static const int mate = CHECKMATE - 1000;
   if ( score > mate || score < -mate )
      return true;

   return false;
}

static inline int score_to_hashtable(int score, int depth)
{
   if (mate_score(score)) {
      if (score > 0)
         return score + depth;
      else
         return score - depth;
   } else
      return score;
}

static inline int score_from_hashtable(int score, int depth)
{
   if (mate_score(score)) {
      if (score > 0)
         return score - depth;
      else
         return score + depth;
   } else
      return score;
}

#endif
