/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef MOVEGEN_H
#define MOVEGEN_H

#include <stdbool.h>
#include "move.h"
#include "board.h"
#include "expect.h"

/* Attack and move bitboards */
/* Normal moves */
extern bitboard_t king_attack[64];
extern bitboard_t king_attack2[64];
extern bitboard_t knight_attack[64];
extern bitboard_t knight_attack2[64];
extern bitboard_t super_attack[64];
extern bitboard_t orth[64];
extern bitboard_t diag[64];
extern bitboard_t rays[64];
extern bitboard_t attack_row[64][64];
extern bitboard_t attack_file[64][64];
extern bitboard_t attack_diagonal_a8h1[64][64];
extern bitboard_t attack_diagonal_a1h8[64][64];
extern bitboard_t ep_neighbours[16];
extern bitboard_t connecting_ray[64][64];

/* X-ray attacks */
extern bitboard_t xattack_row[64][64];
extern bitboard_t xattack_file[64][64];
extern bitboard_t xattack_diagonal_a8h1[64][64];
extern bitboard_t xattack_diagonal_a1h8[64][64];

/* Population count/mobility */
extern int8_t mobility_row[64][64];
extern int8_t mobility_file[64][64];
extern int8_t mobility_diagonal_a8h1[64][64];
extern int8_t mobility_diagonal_a1h8[64][64];


void generate_movement_tables(void);
void generate_legal_moves(movelist_t *movelist, const board_t *board, int colour);
void generate_moves(movelist_t *movelist, const board_t *board, int colour);
void generate_tactical_moves(movelist_t *movelist, const board_t *board, int colour);
void generate_quiescence_moves(movelist_t *movelist, const board_t *board, int colour);
void generate_legal_quiescence_moves(movelist_t *movelist, const board_t *board, int colour);
void generate_checking_moves(movelist_t *movelist, const board_t *board, int side_to_move, bool include_captures);

bool check_move_exists(move_t move, const board_t *board);

int validate_move(const movelist_t *movelist, int source, int dest);

bitboard_t get_attack_bitboard(const board_t *board, int side);
bitboard_t get_attackers(const board_t *board, int side, int square);
bitboard_t get_all_attackers(const board_t *board, bitboard_t mask, int square);
bitboard_t get_pinned_pieces(const board_t *board, bitboard_t mask, int side, int square);

void print_movelist(const movelist_t *movelist);

static inline bitboard_t get_connecting_ray(int s1, int s2)
{
   return connecting_ray[s1][s2];
}

/* 
 * Primitives for generating orthogonal (rook) attacks
 */
static inline bitboard_t get_file_attacks(bitboard_t occ, int square)
{
   int n;

   n = (get_bitboard_file_occupancy_number(occ, square) >> 1) & 63;
   return attack_file[square][n];
}

static inline bitboard_t get_rank_attacks(bitboard_t occ, int square)
{
   int n;

   n = (get_bitboard_rank_occupancy_number(occ, square) >> 1) & 63;
   return attack_row[square][n];
}

static inline bitboard_t get_rank_xattacks(bitboard_t occ, int square)
{
   int n;

   n = (get_bitboard_rank_occupancy_number(occ, square) >> 1) & 63;
   return xattack_row[square][n];
}

static inline bitboard_t get_orth_attack_bitboard(bitboard_t occ, int square)
{
   return get_file_attacks(occ, square) | get_rank_attacks(occ, square);
}

static inline bitboard_t get_rook_attack_bitboard(const board_t *board, int square)
{
   bitboard_t occ = get_occupied(board);

   return get_orth_attack_bitboard(occ, square);
}

/* 
 * Primitives for generating diagonal (bishop) attacks
 */
static inline bitboard_t get_diagonal_attacks(bitboard_t occ, int square)
{
   int n;

   n = (get_bitboard_diagonal_occupancy_number(occ, square) >> 1) & 63;
   return attack_diagonal_a1h8[square][n];
}

static inline bitboard_t get_antidiag_attacks(bitboard_t occ, int square)
{
   int n;

   n = (get_bitboard_antidiag_occupancy_number(occ, square) >> 1) & 63;
   return attack_diagonal_a8h1[square][n];
}

static inline bitboard_t get_diag_attack_bitboard(bitboard_t occ, int square)
{
   return get_diagonal_attacks(occ, square) | get_antidiag_attacks(occ, square);
}

static inline bitboard_t get_bishop_attack_bitboard(const board_t *board, int square)
{
   bitboard_t occ = get_occupied(board);

   return get_diag_attack_bitboard(occ, square);
}


/* Simplified version of the move generator that only checks if any piece
 * belonging to the given player can reach the square indicated.
 */
static bool inline square_is_attacked(const board_t *board, bitboard_t mask, int square, int player)
{
   bitboard_t bb_where = make_bitboard_square(square);
   bitboard_t occupied;
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops_or_queens;
   bitboard_t rooks_or_queens;
   bitboard_t king;
   bitboard_t bb;
   bitboard_t moves;
   bitboard_t own_movers;
   int c_index = (player != 0);

   /* Initilize appropriate bitboards */
   own_movers = board->bbc[c_index] & mask;
   occupied = get_occupied(board);

   /* Check rook moves and horizontal/vertical queen moves */
   rooks_or_queens = own_movers & get_rook_movers(board);
   if (rooks_or_queens) {
      moves = get_orth_attack_bitboard(occupied, square);
      bb = and_bitboards(moves, rooks_or_queens);
      if (bb) return true;
   }

   /* Check bishop moves and diagonal queen moves */
   bishops_or_queens = own_movers & get_bishop_movers(board);
   if (bishops_or_queens) {
      moves = get_diag_attack_bitboard(occupied, square);
      bb = and_bitboards(moves, bishops_or_queens);
      if (bb) return true;
   }

   /* Check knight and king moves in one go */
   knights = and_bitboards(own_movers, board->bbp[KNIGHT]);
   king = and_bitboards(own_movers, board->bbp[KING]);
   bb = and_bitboards(knights, knight_attack[square]);
   bb = or_bitboards(bb, and_bitboards(king, king_attack[square]));
   if (bb) return true;

   /* Finally check for pawns. */
   pawns = and_bitboards(own_movers, board->bbp[PAWN]);
   if (pawns) {
      if (player==WHITE) {
         bb = and_bitboards(bb_where, shift_bitboard_right_up(pawns)|shift_bitboard_left_up(pawns));
         if (bb) return true;
      } else {
         bb = and_bitboards(bb_where, shift_bitboard_right_down(pawns)|shift_bitboard_left_down(pawns));
         if (bb) return true;
      }
   }

   return false;
}


/* Return true if two squares are on a connecting ray (ie, a super piece
 * can move from one square to the other in one move on an ampty board)
 */
static inline bool squares_on_ray(int square1, int square2)
{
   return (rays[square1] & make_bitboard_square(square2)) != 0;
}

/* Special: get attacks and xray attacks for a rook on a given square */
static inline void get_rook_attacks(const board_t *board, int square, bitboard_t *attack, bitboard_t *xray)
{
   int row_nr, col_nr;

   row_nr = (get_row_occupancy_number(board, square) >> 1) & 63;
   col_nr = (get_file_occupancy_number(board,square) >> 1) & 63;
   *attack = (attack_row[square][row_nr] | attack_file[square][col_nr])&
                           ~make_bitboard_square(square);
   *xray = (xattack_row[square][row_nr] | xattack_file[square][col_nr]);
}

/* Special: get attacks and xray attacks for a bishop on a given square */
static inline void get_bishop_attacks(const board_t *board, int square, bitboard_t *attack, bitboard_t *xray)
{
   int a8h1_nr, a1h8_nr;

   a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
   a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;
   *attack = (attack_diagonal_a8h1[square][a8h1_nr] |
               attack_diagonal_a1h8[square][a1h8_nr])& ~make_bitboard_square(square);
   *xray = xattack_diagonal_a8h1[square][a8h1_nr]|
                           xattack_diagonal_a1h8[square][a1h8_nr];
}

/* Special: get attack and xray attacks for a bishop on a given square and
 * for a given occupancy bitboard.
 */
static inline void get_bishop_attacks_for_occupancy(bitboard_t occ, int square, bitboard_t *attack, bitboard_t *xray)
{
   int a8h1_nr, a1h8_nr;

   a8h1_nr = (get_bitboard_antidiag_occupancy_number(occ, square) >> 1) & 63;
   a1h8_nr = (get_bitboard_diagonal_occupancy_number(occ, square) >> 1) & 63;
   *attack = (attack_diagonal_a8h1[square][a8h1_nr] | attack_diagonal_a1h8[square][a1h8_nr]) & ~make_bitboard_square(square);
   *xray   = xattack_diagonal_a8h1[square][a8h1_nr] | xattack_diagonal_a1h8[square][a1h8_nr];
}



/* Combination of the above two: get queen attacks */
static inline void get_queen_attacks(const board_t *board, int square, bitboard_t *attack, bitboard_t *xray) 
{
   bitboard_t a, x;
   get_rook_attacks(board, square, &a, &x);
   get_bishop_attacks(board, square, attack, xray);
   *attack |= a;
   *xray |= x;
}

/* Pawn attacks */
static inline void get_pawn_attacks(const board_t *board, bitboard_t *white, bitboard_t *black)
{
   bitboard_t pawns = board->bbp[PAWN] & board->bbc[0];

   *white = shift_bitboard_right_up(pawns) | shift_bitboard_left_up(pawns);
   pawns ^= board->bbp[PAWN];
   *black = shift_bitboard_right_down(pawns) | shift_bitboard_left_down(pawns);
}

/* We can check whether the move was a checking move without needing to do
 * the (slightly expensive) attack lookup: if neither the target nor the
 * destination square are in the 'superpiece' attack sphere
 */
static inline bool move_checked_player(const board_t *board, int side, move_t move)
{
   bitboard_t king_bb = board->bbc[side>>7] & board->bbp[KING];
   bitboard_t bb;
   uint8_t king_square = bitscan64(king_bb);

   /* If the move didn't originate on a square connecting to the king, then
    * it can't possibly have been a checking move (exceptions: castling, en-passant).
    * Break out early in this case.
    */
   bb = make_bitboard_square(get_move_origin(move)) | 
        make_bitboard_square(get_move_destination(move)) |
        make_bitboard_square(get_capture_square(move));

   if (!is_castle_move(move) && !(bb & super_attack[king_square]))
      return false;
 
   /* Knight checks: 
    * the piece on the target square is a knight, see if it attacks the
    * enemy king. By explicitly using the target square piece, we
    * automatically consider under promotion to a knight.
    * NB: this check is slower than doing a super-piece check below when
    * doing a perft-test from the starting position, but it is slightly
    * faster in more open positions where other pieces may move to the
    * knight's square.
    */
   if (expect((get_move_piece(move)&PIECE)==KNIGHT, false) ||
       expect((get_promotion_piece(move)&PIECE)==KNIGHT, false)) {
      if (knight_attack[get_move_destination(move)] & king_bb)
         return true;
   }

   bool on_ray = squares_on_ray(king_square, get_move_origin(move)) ||
                 squares_on_ray(king_square, get_move_destination(move)) ||
                 squares_on_ray(king_square, get_capture_square(move));

   /* Catch vertical check by the rook after castling */
   if (!on_ray && is_castle_move(move)) {
      int square = (get_move_origin(move) + get_move_destination(move))/2;
      on_ray = squares_on_ray(king_square, square);
   }
   return on_ray && square_is_attacked(board, super_attack[king_square], king_square, side^BLACK);
}

static inline bool player_in_check(const board_t *board, int colour)
{
   /* This test is only needed if:
    *  - the player was in check (unless we only do legal move generation
    *    in that case)
    * - the move is a king move (non-castling)
    * - the piece is pinned (unless we do legal move generation for pinned
    *   pieces)
    * - the move is an en-passant capture (pins/xray attacks are
    *   complicated in this case, can omit if king is not on the same
    *   rank as the captured pawn)
    */
   uint8_t king_square = bitscan64(and_bitboards(board->bbc[colour>>7],
                                     board->bbp[KING]));
   if (square_is_attacked(board, super_attack[king_square], king_square, colour^BLACK))
      return true;
   return false;
}

/* Test whether a move leaves the player in check.
 * We can assume that the player was not in check prior to the move (since
 * we generate only legal evasions) so we only need to test whether the
 * piece was pinned, or is a king move into check.
 */
static inline bool move_leaves_player_in_check(const board_t *board, move_t move, int side)
{
   if ((get_move_piece(move)&PIECE) == KING)
      return player_in_check(board, side);

   bitboard_t king = board->bbp[KING] & board->bbc[side >> 7];
   bitboard_t bb;
   int square = get_move_origin(move);
   bb = orth[square];
   if (bb & king) {
      int king_square = bitscan64(king);
      bb &= orth[king_square];
      bitboard_t rooks = get_rook_movers(board) & board->bbc[1 - (side >> 7)];
      if (bb & rooks)
         return square_is_attacked(board, bb & rooks, king_square, side^BLACK);
   }

   bb = diag[square];
   if (bb & king) {
      int king_square = bitscan64(king);
      bb &= diag[king_square];
      bitboard_t bishops = get_bishop_movers(board) & board->bbc[1 - (side>>7)];
      if (bb & bishops)
         return square_is_attacked(board, bb & bishops, king_square, side^BLACK);
   }

   assert(!player_in_check(board, side));
   return false;
}


#endif
