/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef PSQ_H
#define PSQ_H

/* Dynamic piece square tables */
typedef struct {
   uint64_t lock;
   int8_t data[2][64];
} piece_square_table_t;

typedef struct {
   size_t number_of_elements;

   piece_square_table_t *data;
} piece_square_hash_table_t;

piece_square_hash_table_t *create_piece_hash_table(size_t nelem);
void destroy_piece_hash_table(piece_square_hash_table_t *table);
piece_square_table_t *query_piece_table_entry(piece_square_hash_table_t *table, uint64_t key);
piece_square_table_t *get_free_piece_table_entry(piece_square_hash_table_t *table, uint64_t key);
void store_piece_table_entry(piece_square_hash_table_t *table, uint64_t key, const piece_square_table_t *data);
void prefetch_piecetable(piece_square_hash_table_t *table, uint64_t key);

/* Map the pawn structure to a hash key for each particular piece type */

/* Bishop key:
 * Take into account both the pawn structure itself and the blocked pawns
 * FIXME: don't include pawns blocked by pawns, those are already in the
 * hash key.
 */
static inline uint64_t get_bishop_table_key(const board_t *board)
{
   bitboard_t pawns[2];
   uint64_t key;

   /* Pawn structure */
   key = board->pawn_hash;

   /* Key in the blocked pawns */
   pawns[0] = board->bbp[PAWN] & board->bbc[0];
   pawns[1] = board->bbp[PAWN] & board->bbc[1];
   pawns[0] &= board->bbc[1]>>8;
   pawns[1] &= board->bbc[0]<<8;

   for (int side = 0; side<2; side++) {
      while (pawns[side]) {
         int square = bitscan64(pawns[side]);
         pawns[side] ^= make_bitboard_square(square);
         key ^= piece_key[side][KING][square];
      }
   }

   return key;
}


/* Knight key:
 * Only take into account the pawn structure itself.
 */
static inline uint64_t get_knight_table_key(const board_t *board)
{
   uint64_t key;

   /* Pawn structure */
   key = board->pawn_hash;

   return key;
}


/* Queen key:
 * Only take into account the pawn structure itself.
 */
static inline uint64_t get_queen_table_key(const board_t *board)
{
   uint64_t key;

   /* Pawn structure */
   key = board->pawn_hash;

   return key;
}


/* Rook key:
 * Pawn structure and the row of the king.
 */
static inline int get_rook_table_key(const board_t *board)
{
   int king_square[2];
   uint64_t key;

   /* The position key depends on the pawn structure and on the rows that the
    * two kings are on.
    */
   king_square[0] = bitscan64(board->bbp[KING] & board->bbc[0]);
   king_square[1] = bitscan64(board->bbp[KING] & board->bbc[1]);
   key = board->pawn_hash;
   key ^= piece_key[0][KING][unpack_rank(king_square[0])];
   key ^= piece_key[1][KING][unpack_rank(king_square[1])];

   return key;
}


/* King key:
 * Take into account pawn structure as well as game phase.
 * FIXME: we can be a bit more coarse-grained than this; we don't need 16
 * different phases here, I think.
 */
static inline uint64_t get_king_table_key(const board_t *board)
{
   int majors[2];
   int minors[2];
   bitboard_t bb;
   uint64_t key;
   int p;

   /* Pawn structure */
   key = board->pawn_hash;

   /* Number of minors: knights + bishops */
   bb = board->bbp[KNIGHT] | board->bbp[BISHOP];
   minors[0] = sparse_popcount64(bb & board->bbc[0]);
   minors[1] = sparse_popcount64(bb & board->bbc[1]);

   /* Number of majors: rooks + 2*qeens (1 queen == 2 rooks approximately) */
   majors[0] = sparse_popcount64(board->bbp[ROOK] & board->bbc[0]) + 
             2*sparse_popcount64(board->bbp[QUEEN] & board->bbc[0]);
   majors[1] = sparse_popcount64(board->bbp[ROOK] & board->bbc[1]) + 
             2*sparse_popcount64(board->bbp[QUEEN] & board->bbc[1]);

   /* Game phase:
    * p = 16   - beginning of game
    * p = 14   - middle game (one piece of each exchanged)
    * p = 10   - late middle game (three pieces exchanged)
    * p = 8    - early end game: RR&2minor+RR&2minor, QRR+QRR
    * p = 6    - end game: RRminor+RRminor, Qminor+Qminor
    * p = 4    - normal end game: RR+RR, Q+Q, R&minor+R&minor
    * p = 2    - "simple" ending: R+R, minor-minor
    * p = 0    - pawn ending
    */
   p = majors[0]+minors[0] + majors[1]+minors[1];

   /* If we're in the end game, then the game phase isn't used to scale the
    * evaluation tables, so we might as well set it to 0 to avoid
    * unnecessarily recalculating the tables as pieces are exchanged.
    */
   if (p <= 6) p = 0;

   /* The number of pieces on the board is a proxi for the game-phase. If
    * majors+minors <=8, then this is an end game and we treat the score
    * differently.
    */
   key ^= piece_key[0][KING][p];

   return key;
}

#endif
