/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef SQUARE_H
#define SQUARE_H

/* Map squares from the rotated bitboard back to the normal board. */
extern int diagonal_a8h1_square[64];
extern int diagonal_a1h8_square[64];
extern int diagonal_a8h1_offset[64];
extern int diagonal_a1h8_offset[64];
extern int diagonal_a8h1_ord[64];
extern int diagonal_a1h8_ord[64];
extern int diagonal_a8h1_diagonal[64];
extern int diagonal_a1h8_diagonal[64];
extern int diagonal_a8h1_inverse_map[64];
extern int diagonal_a1h8_inverse_map[64];

void construct_inverse_diagonal_maps(void);

/********************************************************************
 * Functions for dealing with packing/unpacking square rows/columns *
 ********************************************************************/
static inline int pack_row_file(int row, int file)
{
   return file + (row<<3);
}

static inline int unpack_rank(int packed)
{
   return packed>>3;
}

static inline int unpack_file(int packed)
{
   return packed&7;
}

static inline int xchg_row_file(int packed)
{
   return (packed>>3) | ((packed & 7)<<3);
}

static inline int flip_horizontal(int packed)
{
   int file = packed & 7;
   return (packed & ~7) | (7 - file);
}

static inline int flip_vertical(int packed)
{
   int rank = packed >> 3;
   return (packed & 7) | ((7 - rank) << 3);
}

static inline unsigned int x88diff(int f, int t)
{
   return t - f + (t|7) - (f|7) + 120;
}


/********************************************************************
 * Functions for mapping squares onto a rotated bitboard (a8-h1)    * 
 ********************************************************************/
/* returns the number of the diagonal parallel to a8-h1 that the square is
 * on. Used for rotated bitboards.
 */
static inline int calculate_a8h1_diagonal(int packed)
{
   return unpack_rank(packed)+unpack_file(packed);
}

/* Get the number of the square along the length of the diagonal, starting
 * from the lower right hand side of the board.
 * Also used for rotated bitboards.
 */
static inline int calculate_a8h1_diagonal_ord(int packed)
{
   int row = unpack_rank(packed);
   int diag_nr = calculate_a8h1_diagonal(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;
   int n = k * (1 + (diag_nr&7) );
   return row - n;
}

/* Calculate the offset of the start of the diagonal in the rotated
 * bitboard. Again used for rotated bitboards.
 */
static inline int calculate_a8h1_diagonal_offset(int packed)
{
   int diag_nr = calculate_a8h1_diagonal(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;

   /* The total area under the diagonal, if we pretend that the board
    * continues indefinitely; really this is just Gauss' expression for the
    * sum of the integers from 1 to diag_nr, which is the number of squares
    * below this diagonal.
    * As long as we are below the main diagonal, we can use this number
    * directly, but above it we need to correct for the fact that the board
    * ends at the h-file.
    */
   int area = diag_nr*(diag_nr+1)/2;

   return area - k*((diag_nr&7) + 1)*(diag_nr&7);
}

/* Map a packed square to the equivalent square on the bitboard mapping the
 * main diagonal.
 * Again used for rotated bitboards.
 */
static inline int calculate_a8h1_diagonal_square(int packed)
{
   int diag_nr = calculate_a8h1_diagonal(packed);
   int row = unpack_rank(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;

   /* The total area under the diagonal, if we pretend that the board
    * continues indefinitely; really this is just Gauss' expression for the
    * sum of the integers from 1 to diag_nr, which is the number of squares
    * below this diagonal.
    * As long as we are below the main diagonal, we can use this number
    * directly, but above it we need to correct for the fact that the board
    * ends at the h-file.
    */
   int area = diag_nr*(diag_nr+1)/2;

   return row + area - k*((diag_nr&7) + 1)*((diag_nr&7) + 1);
}

/* Wrapper functions that can be used instead to query tables that contain
 * precomputed results of the above calculations. These gently speed
 * up the program in practice.
 */

/* returns the number of the diagonal parallel to a8-h1 that the square is
 * on. Used for rotated bitboards.
 */
static inline int get_a8h1_diagonal(int packed)
{
   return diagonal_a8h1_diagonal[packed];
}


/* Get the number of the square along the length of the diagonal, starting
 * from the lower right hand side of the board.
 * Also used for rotated bitboards.
 */
static inline int get_a8h1_diagonal_ord(int packed)
{
   return diagonal_a8h1_ord[packed];
}


/* Calculate the offset of the start of the diagonal in the rotated
 * bitboard. Again used for rotated bitboards.
 */
static inline int get_a8h1_diagonal_offset(int packed)
{
   return diagonal_a8h1_offset[packed];
}


/* Map a packed square to the equivalent square on the bitboard mapping the
 * main diagonal.
 * Again used for rotated bitboards.
 */
static inline int get_a8h1_diagonal_square(int packed)
{
   return diagonal_a8h1_square[packed];
}



/********************************************************************
 * Functions for mapping squares onto a rotated bitboard (a1-h8)    * 
 ********************************************************************/
/* returns the number of the diagonal parallel to a1-h8 that the square is
 * on. Used for rotated bitboards.
 */
static inline int calculate_a1h8_diagonal(int packed)
{
   return unpack_rank(packed)+7-unpack_file(packed);
}

/* Get the number of the square along the length of the diagonal, starting
 * from the lower right hand side of the board.
 * Also used for rotated bitboards.
 */
static inline int calculate_a1h8_diagonal_ord(int packed)
{
   int row = unpack_rank(packed);
   int diag_nr = calculate_a1h8_diagonal(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;
   int n = k * (1 + (diag_nr&7) );
   return row - n;
}

/* Calculate the offset of the start of the diagonal in the rotated
 * bitboard. Again used for rotated bitboards.
 */
static inline int calculate_a1h8_diagonal_offset(int packed)
{
   int diag_nr = calculate_a1h8_diagonal(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;

   /* The total area under the diagonal, if we pretend that the board
    * continues indefinitely; really this is just Gauss' expression for the
    * sum of the integers from 1 to diag_nr, which is the number of squares
    * below this diagonal.
    * As long as we are below the main diagonal, we can use this number
    * directly, but above it we need to correct for the fact that the board
    * ends at the h-file.
    */
   int area = diag_nr*(diag_nr+1)/2;

   return area - k*((diag_nr&7) + 1)*(diag_nr&7);
}

/* Map a packed square to the equivalent square on the bitboard mapping the
 * main diagonal.
 * Again used for rotated bitboards.
 */
static inline int calculate_a1h8_diagonal_square(int packed)
{
   int diag_nr = calculate_a1h8_diagonal(packed);
   int row = unpack_rank(packed);

   /* We need to switch the way we count above the main diagonal, k will be
    * the switch to decide this: it is 0 below the diagonal and 1 above it.
    */
   int k = (diag_nr&8) >> 3;

   /* The total area under the diagonal, if we pretend that the board
    * continues indefinitely; really this is just Gauss' expression for the
    * sum of the integers from 1 to diag_nr, which is the number of squares
    * below this diagonal.
    * As long as we are below the main diagonal, we can use this number
    * directly, but above it we need to correct for the fact that the board
    * ends at the h-file.
    */
   int area = diag_nr*(diag_nr+1)/2;

   return row + area - k*((diag_nr&7) + 1)*((diag_nr&7) + 1);
}


/* Wrapper functions that can be used instead to query tables that contain
 * precomputed results of the above calculations. These gently speed
 * up the program in practice.
 */

/* returns the number of the diagonal parallel to a1-h8 that the square is
 * on. Used for rotated bitboards.
 */
static inline int get_a1h8_diagonal(int packed)
{
   return diagonal_a1h8_diagonal[packed];
}

/* Get the number of the square along the length of the diagonal, starting
 * from the lower right hand side of the board.
 * Also used for rotated bitboards.
 */
static inline int get_a1h8_diagonal_ord(int packed)
{
   return diagonal_a1h8_ord[packed];
}

/* Calculate the offset of the start of the diagonal in the rotated
 * bitboard. Again used for rotated bitboards.
 */
static inline int get_a1h8_diagonal_offset(int packed)
{
   return diagonal_a1h8_offset[packed];
}

/* Map a packed square to the equivalent square on the bitboard mapping the
 * main diagonal.
 * Again used for rotated bitboards.
 */
static inline int get_a1h8_diagonal_square(int packed)
{
   return diagonal_a1h8_square[packed];
}

#endif
