#include "game.h"
#include "egtb.h"

#ifdef WANT_GTB
#include "gtb-probe.h"

static const char **tb_paths = NULL;
#endif

char *tb_initinfo = NULL;

int initialise_tablebase(const char *path, size_t cache_size, int wdl_fraction)
{
#ifdef WANT_GTB
   if (wdl_fraction > 128) wdl_fraction = 128;
   if (wdl_fraction < 0)   wdl_fraction = 0;
   if (cache_size == 0) cache_size = 32*1024*1024;  /* 32 MB cache by default */

   if (tb_is_initialized()) {
      /* cleanup old paths */
      tb_paths = tbpaths_done(tb_paths); 

      if (path == NULL || strlen(path) == 0) {
         tbcache_done();
         tb_done();
         tb_paths = NULL;
         return 0;
      }

      /* init new paths */
      tb_paths = tbpaths_init(); 
      tb_paths = tbpaths_add(tb_paths, path);

      /* restart */
      tb_initinfo = tb_restart(1, tb_CP4, tb_paths);
      tbcache_restart(cache_size, wdl_fraction); 

		return tb_availability();
   }

   if (path == NULL || strlen(path) == 0)
      return 0;

   if (tb_paths == NULL) tb_paths = tbpaths_init();
	tb_paths = tbpaths_add(tb_paths, path);

	/* init probing code, indexes, paths, etc. */
	tb_initinfo = tb_init(1, tb_CP4, tb_paths);

	/* init cache */
	tbcache_init(cache_size, wdl_fraction); 
	tbstats_reset();

   return tb_availability();
#else
   return 0;
#endif
}

void add_tablebase_path(const char *path)
{
#ifdef WANT_GTB
   if (path == NULL || strlen(path) == 0)
      return;

   if (tb_paths == NULL) tb_paths = tbpaths_init();
	tb_paths = tbpaths_add(tb_paths, path);

   if (tb_is_initialized())
      tb_initinfo = tb_restart(1, tb_CP4, tb_paths);
#else
   (void)path;
#endif
}

unsigned int tablebase_flags(void)
{
#ifdef WANT_GTB
   return tb_availability();
#else
   return 0;
#endif
}

#ifdef WANT_GTB
bool probe_hard(const board_t *board, int side_to_move, unsigned int *info, unsigned int *dtm)
{
   /* Probe tablebase */
   unsigned int  stm;		/* side to move */
   unsigned int  wsq[17];	/* list of squares for white */
   unsigned int  bsq[17];	/* list of squares for black */
   unsigned char wp[17];	/* what white pieces are on those squares */
   unsigned char bp[17];	/* what black pieces are on those squares */
   int n;

   if (side_to_move == 0)
      stm = tb_WHITE_TO_MOVE;
   else
      stm = tb_BLACK_TO_MOVE;

   bitboard_t bb = board->bbc[0];
   n = 0;
   while (bb) {
      int sq = bitscan64(bb);
      bb ^= make_bitboard_square(sq);
      wsq[n] = sq;
      wp[n]  = 1 + (get_piece(board, sq) & PIECE);
      n++;
   }
   wsq[n] = tb_NOSQUARE;
   wp[n] = tb_NOPIECE;

   bb = board->bbc[1];
   n = 0;
   while (bb) {
      int sq = bitscan64(bb);
      bb ^= make_bitboard_square(sq);
      bsq[n] = sq;
      bp[n]  = 1 + (get_piece(board, sq) & PIECE);
      n++;
   }
   bsq[n] = tb_NOSQUARE;
   bp[n] = tb_NOPIECE;

   bool ret = tb_probe_hard(stm, tb_NOSQUARE, tb_NOCASTLE, wsq, bsq, wp, bp, info, dtm);

   switch (*info) {
      case tb_DRAW:
         *info = heuristic_draw;
         break;

      case tb_WMATE:
         *info = (side_to_move == WHITE) ? heuristic_win  : heuristic_loss;
         break;

      case tb_BMATE:
         *info = (side_to_move == WHITE) ? heuristic_loss : heuristic_win;
         break;

      default:
         *info = heuristic_unknown;
   }

   return ret;
}

bool probe_soft(const board_t *board, int side_to_move, unsigned int *info, unsigned int *dtm)
{
   /* Probe tablebase */
   unsigned int  stm;		/* side to move */
   unsigned int  wsq[17];	/* list of squares for white */
   unsigned int  bsq[17];	/* list of squares for black */
   unsigned char wp[17];	/* what white pieces are on those squares */
   unsigned char bp[17];	/* what black pieces are on those squares */
   int n;

   if (side_to_move == 0)
      stm = tb_WHITE_TO_MOVE;
   else
      stm = tb_BLACK_TO_MOVE;

   bitboard_t bb = board->bbc[0];
   n = 0;
   while (bb) {
      int sq = bitscan64(bb);
      bb ^= make_bitboard_square(sq);
      wsq[n] = sq;
      wp[n]  = 1 + (get_piece(board, sq) & PIECE);
      n++;
   }
   wsq[n] = tb_NOSQUARE;
   wp[n] = tb_NOPIECE;

   bb = board->bbc[1];
   n = 0;
   while (bb) {
      int sq = bitscan64(bb);
      bb ^= make_bitboard_square(sq);
      bsq[n] = sq;
      bp[n]  = 1 + (get_piece(board, sq) & PIECE);
      n++;
   }
   bsq[n] = tb_NOSQUARE;
   bp[n] = tb_NOPIECE;

   bool ret = tb_probe_soft(stm, tb_NOSQUARE, tb_NOCASTLE, wsq, bsq, wp, bp, info, dtm);

   switch (*info) {
      case tb_DRAW:
         *info = heuristic_draw;
         break;

      case tb_WMATE:
         *info = (side_to_move == WHITE) ? heuristic_win  : heuristic_loss;
         break;

      case tb_BMATE:
         *info = (side_to_move == WHITE) ? heuristic_loss : heuristic_win;
         break;

      default:
         *info = heuristic_unknown;
   }


   return ret;
}

#endif
