/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <math.h>
#include "config.h"
#include "bitboard.h"
#include "movegen.h"
#include "board.h"
#include "game.h"
#include "evaluate.h"
#include "pawns.h"
#include "bits.h"
#include "piece_tables.h"
#include "evalhash.h"
#include "names.h"
#include "softexp.h"
#include "heuristic.h"

/* The evaluation function should consider the following:
 *
 * TODO: it is probably slightly better to give penalties for not being in
 * a favourable position than it is for giving bonuses for being there (eg,
 * penalise a rook on a closed file rather than award a rook on an open
 * file).
 *
 * Idea: Generate piece square tables based on pawn structure and the
 * location of the two kings. Before the search, initialise these tables
 * with values based on the current position. When doing move ordering,
 * use the difference between square values to sort regular moves
 * (reduction candidates). When it becomes time to do the evaluation,
 * we can recompute the tables (if needed, or look them up in a hash table)
 * and compute accurate scores for the pieces. For lazy evaluation, we
 * could use the score based on the old tables.
 *
 * Always:
 *  - material balance, this is a simple sum of piece values
 *  - slight (+5) bonus for side-to-move
 *  - piece-square tables
 *
 * Opening:
 *  - advance pawns
 *  - develop light pieces towards the centre
 *  - penalise moving a rook if not yet castled
 *  - penalise moving the king except when castling
 *  - penalise undeveloped light pieces
 *
 * Middle game:
 *  - penalise king-in-centre
 *  - penalise trapped pieces:
 *     * rook in the corner trapped by the king (encourage castling)
 *     * bishop trapped behind pawns of the same colour
 *     * bishop trapped in the corner behind enemy pawns
 *  - weakness of squares of a particular colour: majority of pawns on one
 *    colour and no bishop of that colour
 *  - award bonus for the bishop pair
 *  - evaluate king safety:
 *     * pawn structure
 *     * enemy pieces
 *  - pawn structure:
 *     * penalise doubled pawns
 *     * penalise backward pawns or isolated pawns
 *`      penalise more when on open (semi-open) files and when blocked
 *       penalty increases progressively with the number of weak pawns
 *     * penalise hanging pawns if they are blocked and minor pieces are
 *       exchanged
 *     * weak squares/bonus for knight/bishop on strong square
 *  - award rooks on open files/7th row
 *  - Pressure on weak pawns
 *  - if material is equal, give a bonus to the side with the most pieces
 *    (ie, non-pawns). Rationale: two light pieces are better than
 *    rook+pawn, etc.
 *
 * End game:
 *  - test for simple endings/exact positions
 *  - push king to centre
 *  - give bonus to side with more pawns if there are fewer pieces
 *    (encourage exchanging pieces if winning)
 *  - conversely, give bonus to fewer pawns if a pawn down
 *  - slightly decrease effective value of knight, which is weaker in the
 *    end game than sliding pieces.
 *  - bonus for pushing a candidate
 *  - bonus for free pawns/advanced pawns
 *     * defended free pawns
 *     * connected free pawns
 *     * defending king outside square
 *     * "2 pawns on 6th rank beat a rook"
 *     * key squares in front of free pawn should be occupied by king
 *  - rook should attack base of enemy pawn structure (generalise 7th row rule)
 *  - recognise draw-ish end games: rook vs rook+minor, opposing bishops,
 *    bishop vs knight with pawns on one wing, wrong pawn.
 *
 * Board control:
 *  award a bonus for the player who controls most squares in the extended
 *  centre and/or near the enemy king (although the latter should probably
 *  be done by king safety code).
 *
 * Additional pawn bits:
 *  - "When there is about fourteen points of material on both sides, the
 *  value of pawns on any file is about equal. After that, wing pawns
 *  become more valuable." (http://en.academic.ru/dic.nsf/enwiki/451243)
 * - Recognise and score minority/majority attack patterns
 * - According to Robert Hyatt: connected passers are better if there is
 *   more material on the board and should be scored lower as material
 *   disappears. Isolated pawns go the other way: they are bad when there
 *   is material on the board but become stronger when all pieces have been
 *   traded away (http://www.stmintz.com/ccc/index.php?id=341873) TODO
 *
 * FIXME: currently, strong squares are only squares in front of enemy
 * pawns. It's quite possible for a strong square to be behind enemy
 * lines...
 */

/* Debugging macros */
#undef DEBUG_PIECE_SQUARE_TABLES
#undef DEBUG_EVALUATION
#ifdef DEBUG_EVALUATION

#define WRITE_LOG(...) printf(__VA_ARGS__)

#else

#define WRITE_LOG(...) (void)0

#endif

/* Piece value related terms */
#ifdef PIECE_VALUE_VAR
int PAWN_BASE        = 100;
int MINOR_BASE       = 325;
int ROOK_BASE        = 550;
int QUEEN_BASE       = 975;
int EXCHANGE_BASE    = 50;
int MEXCHANGE_BASE   = 50;
int EVAL_BISHOP_PAIR = 50;
int ROOK_ADVANTAGE_BASE = 100;
int MINOR_ADVANTAGE_BASE = 90;
int KNIGHT_VALUE_SCALE = 4;
int ROOK_VALUE_SCALE = 0;
int RAZOR_MARGIN_UNIT = 320;
int FUTILITY1        = 0;
int FUTILITY2        = 300;
int FUTILITY3        = 500;
int FUTILITY4        = 900;
int FUTILITY5        = 1500;
#endif

int PAWN_COUNT_VALUE_ADJUST0  =  0;
int PAWN_COUNT_VALUE_ADJUST1  =  0;
int PAWN_COUNT_VALUE_ADJUST2  =  0;
int PAWN_COUNT_VALUE_ADJUST3  =  0;
int PAWN_COUNT_VALUE_ADJUST4  =  0;
int PAWN_COUNT_VALUE_ADJUST5  =  0;
int PAWN_COUNT_VALUE_ADJUST6  =  0;
int PAWN_COUNT_VALUE_ADJUST7  =  0;
int PAWN_COUNT_VALUE_ADJUST8  =  0;

int futility_margins[6];

/* Centre score table, used for various pieces (knight, king in end game) */
static const int8_t centre_table[64] = {
   0, 2,  4,  6,  6,  4, 2, 0,
   2, 4,  6,  8,  8,  6, 4, 2,
   4, 6,  8, 10, 10,  8, 6, 4,
   6, 8, 10, 12, 12, 10, 8, 6,
   6, 8, 10, 12, 12, 10, 8, 6,
   4, 6,  8, 10, 10,  8, 6, 4,
   2, 4,  6,  8,  8,  6, 4, 2,
   0, 2,  4,  6,  6,  4, 2, 0
};

/* Alternate centre table, used for bishops */
static const int8_t centre_table2[64] = {
   0, 2,  4,  6,  6,  4, 2, 0,
   2, 8,  6,  8,  8,  6, 8, 2,
   4, 6,  6,  6,  6,  6, 6, 4,
   6, 8, 10, 12, 12, 10, 8, 6,
   6, 8, 10, 12, 12, 10, 8, 6,
   4, 6,  8,  8,  8,  8, 6, 4,
   2, 6,  6,  8,  8,  6, 6, 2,
   0, 2,  4,  6,  6,  4, 2, 0
};

/* Evaluation table for KNBK ending, assuming a dark-squared bishop.
 * For a light-squared bishop, flip the board.
 * This table gives the score from the perspective of the attacker, so the
 * corner of the bishop colour gets a higher score.
 */
static const int8_t knbk_table[64] = {
   35, 30, 25, 20, 15, 10,  5,  0,
   30, 25, 20, 15, 10,  5,  0,  5,
   25, 20, 15, 10,  5,  0,  5, 10,
   20, 15, 10,  5,  0,  5, 10, 15,
   15, 10,  5,  0,  5, 10, 15, 20,
   10,  5,  0,  5, 10, 15, 20, 25,
    5,  0,  5, 10, 15, 20, 25, 30,
    0,  5, 10, 15, 20, 25, 30, 35
};

/* Centralisation table for the king.
 * Used in some end games.
 * TODO: experiment with using this over the generic table for all king
 * evaluation in the end game.
 */
static int8_t king_centre_table[64];

/* This table of code and the comment below are adopted from Crafty.
 *
 * This array is indexed by rook advantage and minor piece advantage.
 * rook advantage is 4 + white rook equivalents - black rook equivalents 
 * where a rook equivalent is number of rooks + 2 * number of queens.
 * minor piece advantage is 4 + white minors - black minors.  
 *
 * The classic bad trade case is two minors for a rook.  If white trades
 * two minors for a rook, rook advantage is +5 and minor advantage is +2.
 * imbalance[5][2] gives a penalty of -42 for this trade.
 *
 * TODO: these values are tuned for Crafty, but our piece evaluations are
 * (slightly) different, so these should probably be re-tuned.
 * Note that the elements orthogonl to the (0, 0) element really just
 * affect the base value of major and minor pieces, so really the base
 * value of a minor piece goes from 3.25 -> 4.13 and the base value of a
 * rook goes from 5.00 -> 6.04 with this table. 
 */
static int8_t imbalance[9][9] = {
/* n=-4  n=-3  n=-2  n=-1   n=0  n=+1  n=+2  n=+3 +n=+4 */
  {-126, -126, -126, -126, -126, -126, -126, -126,  -50 }, /* R=-4 */
  {-126, -126, -126, -126, -126, -126, -126,  -50,   50 }, /* R=-3 */
  {-126, -126, -126, -126, -126, -126,  -50,   50,  100 }, /* R=-2 */
  {-126, -126, -126, -126, -100,  -50,   50,  100,  126 }, /* R=-1 */
  {-126, -126, -126,  -90,    0,   90,  126,  126,  126 }, /*  R=0 */
  {-126, -100,  -50,   50,  100,  126,  126,  126,  126 }, /* R=+1 */
  {-100,  -50,   50,  126,  126,  126,  126,  126,  126 }, /* R=+2 */
  { -50,   50,  126,  126,  126,  126,  126,  126,  126 }, /* R=+3 */
  {  50,  126,  126,  126,  126,  126,  126,  126,  126 }  /* R=+4 */
};

#if 0
/* Original (from Crafty): */
/* n=-4  n=-3  n=-2  n=-1   n=0  n=+1  n=+2  n=+3 +n=+4 */
  {-..., -..., -..., -..., -..., -..., -..., -...,  -42 }, /* R=-4 */
  {-..., -..., -..., -..., -..., -..., -...,  -42,   42 }, /* R=-3 */
  {-..., -..., -..., -..., -..., -...,  -42,   42,   84 }, /* R=-2 */
  {-..., -..., -..., -..., -104,  -42,   42,   84,  ... }, /* R=-1 */
  {-..., -..., -...,  -88,    0,   88,  ...,  ...,  ... }, /*  R=0 */
  {-...,  -84,  -42,   42,  104,  ...,  ...,  ...,  ... }, /* R=+1 */
  { -84,  -42,   42,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+2 */
  { -42,   42,  ...,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+3 */
  {  42,  ...,  ...,  ...,  ...,  ...,  ...,  ...,  ... }  /* R=+4 */
/* n=-4  n=-3  n=-2  n=-1   n=0  n=+1  n=+2  n=+3 +n=+4 */
  {-..., -..., -..., -..., -..., -..., -..., -...,   MX }, /* R=-4 */
  {-..., -..., -..., -..., -..., -..., -...,   MX,  -MX }, /* R=-3 */
  {-..., -..., -..., -..., -..., -...,   MX,  -MX, -2MX }, /* R=-2 */
  {-..., -..., -..., -...,  -dR,  -EX,  -MX, -2MX,  ... }, /* R=-1 */
  {-..., -..., -...,  -dN,    0,   dN,  ...,  ...,  ... }, /*  R=0 */
  {-...,  2MX,   MX,   EX,   dR,  ...,  ...,  ...,  ... }, /* R=+1 */
  { 2MX,   MX,  -MX,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+2 */
  {  MX,  -MX,  ...,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+3 */
  { -MX,  ...,  ...,  ...,  ...,  ...,  ...,  ...,  ... }  /* R=+4 */

/* Absolute differences, standard piece values: */
/* n=-4  n=-3  n=-2  n=-1   n=0  n=+1  n=+2  n=+3 +n=+4 */
  {-..., -..., -..., -..., -..., -..., -..., -...,  -.. }, /* R=-4 */
  {-..., -..., -..., -..., -..., -..., -..., -500,   .. }, /* R=-3 */
  {-..., -..., -..., -..., -..., -..., -350,   25,   .. }, /* R=-2 */
  {-..., -..., -..., -..., -500, -175,  150,  475,  ... }, /* R=-1 */
  {-..., -..., -..., -325,    0,  325,  ...,  ...,  ... }, /*  R=0 */
  {-..., -475, -150,  175,  500,  ...,  ...,  ...,  ... }, /* R=+1 */
  { -..,  -25,  350,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+2 */
  { -..,  500,  ...,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+3 */
  {  ..,  ...,  ...,  ...,  ...,  ...,  ...,  ...,  ... }  /* R=+4 */

/* Including modification */
/* n=-4  n=-3  n=-2  n=-1   n=0  n=+1  n=+2  n=+3 +n=+4 */
  {-..., -..., -..., -..., -..., -..., -..., -...,  -.. }, /* R=-4 */
  {-..., -..., -..., -..., -..., -..., -..., -542,   .. }, /* R=-3 */
  {-..., -..., -..., -..., -..., -..., -392,   67,   .. }, /* R=-2 */
  {-..., -..., -..., -..., -604, -217,  192,  559,  ... }, /* R=-1 */
  {-..., -..., -..., -413,    0,  413,  ...,  ...,  ... }, /*  R=0 */
  {-..., -559, -192,  217,  604,  ...,  ...,  ...,  ... }, /* R=+1 */
  { -..,  -67,  392,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+2 */
  { -..,  542,  ...,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+3 */
  {  ..,  ...,  ...,  ...,  ...,  ...,  ...,  ...,  ... }  /* R=+4 */

/* Absolute differences, Revised piece values: */
/* n=-4  n=-3  n=-2  n=-1   n=0  n=+1  n=+2  n=+3 +n=+4 */
  {-..., -..., -..., -..., -..., -..., -..., -...,  -.. }, /* R=-4 */
  {-..., -..., -..., -..., -..., -..., -..., -537,   .. }, /* R=-3 */
  {-..., -..., -..., -..., -..., -..., -375,   33,   .. }, /* R=-2 */
  {-..., -..., -..., -..., -604, -191,  222,  635,  ... }, /* R=-1 */
  {-..., -..., -..., -413,    0,  413,  ...,  ...,  ... }, /*  R=0 */
  {-..., -635, -222,  191,  604,  ...,  ...,  ...,  ... }, /* R=+1 */
  { -..,  -33,  375,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+2 */
  { -..,  537,  ...,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+3 */
  {  ..,  ...,  ...,  ...,  ...,  ...,  ...,  ...,  ... }  /* R=+4 */

/* Difference: */
/* n=-4  n=-3  n=-2  n=-1   n=0  n=+1  n=+2  n=+3 +n=+4 */
  {-..., -..., -..., -..., -..., -..., -..., -...,  -.. }, /* R=-4 */
  {-..., -..., -..., -..., -..., -..., -...,   -5,   .. }, /* R=-3 */
  {-..., -..., -..., -..., -..., -...,  -17,   34,   .. }, /* R=-2 */
  {-..., -..., -..., -...,    0,  -26,   30,   76,  ... }, /* R=-1 */
  {-..., -..., -...,    0,    0,    0,  ...,  ...,  ... }, /*  R=0 */
  {-...,  -76,  -30,   26,    0,  ...,  ...,  ...,  ... }, /* R=+1 */
  { -..,  -34,   17,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+2 */
  { -..,    5,  ...,  ...,  ...,  ...,  ...,  ...,  ... }, /* R=+3 */
  {  ..,  ...,  ...,  ...,  ...,  ...,  ...,  ...,  ... }  /* R=+4 */
#endif

/* Penalty for doubled pawns, depending on the number of major pieces
 * (indexed by "rook equivalents" as above). According to Kaufman:
 *
 * 2 rooks each (R = 4): 1/16    1 queen each (R = 4):  1/16
 * 1 rook each  (R = 2): 4/16
 * 0 rooks each (R = 0): 6/16    0 queens each (R = 0): 4/16
 *
 * It probably matters if one has the queen or not in the case of material
 * imbalance (R = 1 or R = 3), but we'll ignore that here for now.
 * The case of rooks and queens present (R > 4) can probably be treated as
 * R = 4.
 *
 * NOTE: an alternative parameterisation of the same might be to index by
 * the number of remaining pawns, with the penalty becoming larger as pawns
 * are exchanged.
 * In that case, the run with pawn number (starting at 8 and counting
 * downward) is about 19, 11, 25, 20, 29, 33, 47
 */
static const int double_pawn_penalty[] = {
   38, 32, 25, 16,  6
};

static int pawn_count_value[9];

static int passed_pawn_bonus[17];

static int king_safety_penalty[65];

static int king_safety_weight[24];

/* Relative strength of different piece types, for evaluation of game phase
 * and pressure on the enemy king.
 * This is based on the attack pattern on a 3x3 and 5x5 board, measuring
 * "short-range" strength of different pieces.
 * The "army strength" of the initial array is 96.
 */
static int piece_attack_weight[] = {
   1, 8, 8, 12, 24, 8
};
static const int initial_army_strength = 96;

static int8_t piece_max_moves[NUM_PIECE_TYPES][64];

/* Penalty for hanging pieces, according to bitfield:
 * Bit 0: hanging pawns present
 * Bit 1: hanging minors present
 * Bit 2: hanging rooks present
 * Bit 3: hanging queens present
 * Bit 4: > 1 hanging piece, to distinguish cases where multiple pieces of
 *        the same type are hanging.
 *
 * This should be enough information to extract a penalty and requires
 * only 32 elements in a lookup table.
 * TODO: consider side-to-move information?
 */
static int hanging_piece_penalty[1<<5];

/* Data structure hodling attack bitboards, for either colour. used for
 * king safety evaluation and queen mobility.
 */
typedef struct {
   bitboard_t attacks[2][NUM_PIECE_TYPES];
   bitboard_t xattacks[2][NUM_PIECE_TYPES];
   bitboard_t all_attacks[2];
} attack_t;

/* Information/data used by various evaluation routines (for instance king
 * safety).
 * TODO: collect non-symmetric (side to move dependent) evaluation terms
 * here as well, or record if any are used. This can be used to improve the
 * caching at the end.
 */
typedef struct {
   attack_t attack;
   int piece_count[2][NUM_PIECE_TYPES];
} eval_info_t;

static inline int max(int x, int y)
{
   return (x>y)?x:y;
}

static inline int min(int x, int y)
{
   return (x<y)?x:y;
}

static int king_distance(int square1, int square2)
{
   int row1, file1;
   int row2, file2;

   row1 = unpack_rank(square1);
   file1 = unpack_file(square1);
   row2 = unpack_rank(square2);
   file2 = unpack_file(square2);

   return abs(row1 - row2) + abs(file1-file2);
}

static int rank_file_distance(int square1, int square2)
{
   int row1, file1;
   int row2, file2;

   row1 = unpack_rank(square1);
   file1 = unpack_file(square1);
   row2 = unpack_rank(square2);
   file2 = unpack_file(square2);

   return max(abs(row1 - row2), abs(file1-file2));
}

/* Initialise parameter-driven evaluation tables */
void initialise_evaluation(void)
{
   double x;
   int n;

   /* Passed pawn evaluation */
   passed_pawn_bonus[0] = 0;
   for (n=1; n<17; n++)
      passed_pawn_bonus[n] = PASSED_PAWN_CEIL / (1.0 + myexp(-PASSED_PAWN_SLOPE*(n - PASSED_PAWN_INFLECTION)));

   /* King safety */
   x = KING_SAFETY_CEIL / (1.0 + myexp(KING_SAFETY_SLOPE*KING_SAFETY_INFLECTION));
   king_safety_penalty[0] = 0;
   for (n=1; n<65; n++)
      king_safety_penalty[n] = KING_SAFETY_CEIL / (1.0 + myexp(-KING_SAFETY_SLOPE*(n - KING_SAFETY_INFLECTION))) - x;

   /* King safety weight, indexed by game phase
    * phase = 16   - beginning of game
    * phase = 14   - middle game (one piece of each exchanged)
    * phase = 10   - late middle game (three pieces exchanged)
    * phase = 8    - early end game: RR&2minor+RR&2minor, QRR+QRR
    * phase = 6    - end game: RRminor+RRminor, Qminor+Qminor
    * phase = 4    - normal end game: RR+RR, Q+Q, R&minor+R&minor
    * phase = 2    - "simple" ending: R+R, minor-minor
    * phase = 0    - pawn ending
    */
   king_safety_weight[0] = 0;
   for (n=1; n<24; n++) {
      king_safety_weight[n] = 64 / (1.0 + myexp(-KING_WEIGHT_SLOPE*(n - KING_WEIGHT_INFLECTION))) - x;
   }

   /* Old linear king safety weights, use until we can properly calibrate
    * the more advanced calibration.
    */
   for (n=0; n<24; n++) {
      king_safety_weight[n] = 0;
      if (n > 6) 
         king_safety_weight[n] = 8 * (n-6);
   }

   int avg = 0;
   for (n=0; n<64; n++) {
      king_centre_table[n] = popcount64(king_attack[n] | king_attack2[n]) + 3*popcount64(board_centre & make_bitboard_square(n));
      avg += king_centre_table[n];
   }
   for (n=0; n<64; n++)
      king_centre_table[n] -= avg/64;

   /* Piece values and material imbalance */
   chess_piece_values[PAWN]   = PAWN_BASE;
   chess_piece_values[KNIGHT] = MINOR_BASE;
   chess_piece_values[BISHOP] = MINOR_BASE;
   chess_piece_values[ROOK]   = ROOK_BASE;
   chess_piece_values[QUEEN]  = QUEEN_BASE;
   chess_piece_values[KING]   = 0;

   /* Exchange imbalances (rook vs minor)
    * imbalance [major][minor]
    */
   imbalance[4-4][4+4] = -EXCHANGE_BASE;
   imbalance[4-3][4+3] = -EXCHANGE_BASE;
   imbalance[4-2][4+2] = -EXCHANGE_BASE;
   imbalance[4-1][4+1] = -EXCHANGE_BASE;

   imbalance[4+1][4-1] =  EXCHANGE_BASE;
   imbalance[4+2][4-2] =  EXCHANGE_BASE;
   imbalance[4+3][4-3] =  EXCHANGE_BASE;
   imbalance[4+4][4-4] =  EXCHANGE_BASE;

   /* Minor exchange imbalances (rook vs 2 minors) */
   imbalance[4-1][4+2] =  MEXCHANGE_BASE;
   imbalance[4-2][4+3] =  MEXCHANGE_BASE;
   imbalance[4-3][4+4] =  MEXCHANGE_BASE;

   imbalance[4+1][4-2] = -MEXCHANGE_BASE;
   imbalance[4+2][4-3] = -MEXCHANGE_BASE;
   imbalance[4+3][4-4] = -MEXCHANGE_BASE;

   /* Three minors for a rook: a minor exchange+ */
   imbalance[4-1][4+3] =  2*MEXCHANGE_BASE;
   imbalance[4-2][4+4] =  2*MEXCHANGE_BASE;

   imbalance[4+1][4-3] = -2*MEXCHANGE_BASE;
   imbalance[4+2][4-4] = -2*MEXCHANGE_BASE;

   /* Rook and minor vs pawn advantage */
   imbalance[4+1][4] =  ROOK_ADVANTAGE_BASE;
   imbalance[4-1][4] = -ROOK_ADVANTAGE_BASE;

   imbalance[4][4+1] =  MINOR_ADVANTAGE_BASE;
   imbalance[4][4-1] = -MINOR_ADVANTAGE_BASE;

   /* Futility pruning margins */
   futility_margins[0] = 0;
   futility_margins[1] = FUTILITY1;
   futility_margins[2] = FUTILITY2;
   futility_margins[3] = FUTILITY3;
   futility_margins[4] = FUTILITY4;
   futility_margins[5] = FUTILITY5;

   /* Pawn value */
   for (n=0; n<9; n++) {
      static const int few_pawn_penalty[] = { 0, 25, 35, 40, 45, 50, 50, 50, 50 };
      pawn_count_value[n] = n*PAWN_BASE - few_pawn_penalty[n];
   }
   pawn_count_value[0] += PAWN_COUNT_VALUE_ADJUST0;
   pawn_count_value[1] += PAWN_COUNT_VALUE_ADJUST1;
   pawn_count_value[2] += PAWN_COUNT_VALUE_ADJUST2;
   pawn_count_value[3] += PAWN_COUNT_VALUE_ADJUST3;
   pawn_count_value[4] += PAWN_COUNT_VALUE_ADJUST4;
   pawn_count_value[5] += PAWN_COUNT_VALUE_ADJUST5;
   pawn_count_value[6] += PAWN_COUNT_VALUE_ADJUST6;
   pawn_count_value[7] += PAWN_COUNT_VALUE_ADJUST7;
   pawn_count_value[8] += PAWN_COUNT_VALUE_ADJUST8;

   /* Calculate maximum number of moves for each piece type as a function
    * of board location.
    * Useful as piece square table, but also for constructing a king piece
    * square table for king safety. The idea is that the king is safe on
    * squares where the attacking army has lower mobility, because those
    * squares are harder to attack.
    */
   int summ[NUM_PIECE_TYPES];
   memset(piece_max_moves, 0, sizeof piece_max_moves);
   memset(summ, 0, sizeof summ);
   for (int s=0; s<64; s++) {
      piece_max_moves[PAWN][s]   = 1;
      piece_max_moves[KING][s]   = popcount64(king_attack[s]) + popcount64(king_attack2[s]);
      piece_max_moves[KNIGHT][s] = popcount64(knight_attack[s]) + popcount64(knight_attack2[s]);
      piece_max_moves[BISHOP][s] = popcount64(diag[s]);
      piece_max_moves[ROOK][s]   = popcount64(orth[s]);
      piece_max_moves[QUEEN][s]  = popcount64(orth[s]|diag[s]);
      for (n=0; n<6; n++)
         summ[n] += piece_max_moves[n][s];
   }
   for (n=0; n<6; n++) {
      for (int s=0; s<64; s++) {
         piece_max_moves[n][s] -= summ[n] / 64;
      }
#if 0
      printf("%s %d\n", piece_str[n], summ[n]);
      for (int s=0; s<64; s++) {
         if (s && (s&7) == 0) printf("\n");
         printf("% 3d ", piece_max_moves[n][s]);
      }
      printf("\n");
#endif
   }

#if 0
   int kpsq[64];
   int pc[6] = { 6, 1, 1, 2, 0, 1 };
   memset(kpsq, 0, sizeof kpsq);
   int w = 8;
   int wf = 24;
   for (n=0; n<5; n++) {
      for (int s=0; s<64; s++) {
         kpsq[s] -= pc[n] * piece_attack_weight[n] * piece_max_moves[n][s];
      }
      w += pc[n] * piece_attack_weight[n];
   }
   if (w < wf) w = wf;
   for (int s=0; s<64; s++) {
      kpsq[s] *= (w - wf);
      kpsq[s] /= (initial_army_strength - wf);
      kpsq[s] /= 3;

      int ksqw = piece_attack_weight[KING]*(3*piece_max_moves[KING][s]/2 + centre_table[s]-6)/2;
      kpsq[s] += (initial_army_strength - w) * ksqw / (initial_army_strength-wf);
      //kpsq[s] += (initial_army_strength - w) * 10*(centre_table[s]-6)/(initial_army_strength-16);
   }
   kpsq[A1] = kpsq[H1] = kpsq[C1]-1;
   kpsq[A8] = kpsq[H8] = kpsq[C8]-1;
   printf("\n");
   printf("%d\n", w);
   for (int s=0; s<64; s++) {
      if (s && (s&7) == 0) printf("\n");
      printf("% 4d ", kpsq[s]);
   }
   printf("\n");

   exit(0);
#endif

   /* Hanging piece penalty table */
   int hp, hm, hr, hq, hh;
   for (hp=0; hp<2; hp++) {
      for (hm=0; hm<2; hm++) {
         for (hr=0; hr<2; hr++) {
            for (hq=0; hq<2; hq++) {
               for (hh=0; hh<2; hh++) {
                  int penalty = 0;
                  if (hq)             penalty = QUEEN_BASE;
                  if (hr && !penalty) penalty = ROOK_BASE;
                  if (hm && !penalty) penalty = MINOR_BASE;
                  if (hp && !penalty) penalty = PAWN_BASE;
                  if (hh == 0)        penalty /= 10;
                  int key = hp | (hm << 1) | (hr << 2) | (hq << 3) | (hh << 4);
                  hanging_piece_penalty[key] = penalty / 2;
               }
            }
         }
      }
   }
}

/************************************/
/* Specialised end game evaluations */
/************************************/

/* Lone bishop with or against pawns */
static bool evaluate_lone_bishop(const board_t *board, int side_to_move, int *ev)
{
   static const bitboard_t wrong_afile_bishop[2] = {
      0xAA55AA55AA55AA55ll, 0x55AA55AA55AA55AAll
   };
   static const bitboard_t wrong_hfile_bishop[2] = {
      0x55AA55AA55AA55AAll, 0xAA55AA55AA55AA55ll
   };
   int attacker = 0;
   int defender;

   /* Determine attacker/defender.
    * The attacker is the player with the bishop.
    */
   if (board->bbp[BISHOP] & board->bbc[1]) attacker = 1;
   defender = attacker^1;

   /* Test whether the defender has any pawns */
   if (board->bbc[defender] & board->bbp[PAWN]) {
      /* TODO */
      return false;
   }

   /* Check for rook pawn + wrong bishop.
    * If there are pawns on multiple files, just return.
    */
   uint8_t pawn_files = fill_north(board->bbp[PAWN]) >> 56;
   if (pawn_files & (pawn_files-1))
      return false;

   /* Only rook pawns? */
   if ( pawn_files & 0x7e )
      return false;

   /* Test whether the bishop is the wrong one */
   bitboard_t bb = 0;
   if (board->bbc[attacker]&board->bbp[PAWN]&board_afile)
      bb |= wrong_afile_bishop[attacker];
   if (board->bbc[attacker]&board->bbp[PAWN]&board_hfile)
      bb |= wrong_hfile_bishop[attacker];

   bool wrong_bishop = ((bb ^ board_all) && board->bbp[BISHOP] & bb);

   if (!wrong_bishop)
      return false;

   /* Rook pawn with bad bishop, this is certainly draw-ish.
    * Test whether the defending king is on or next to the promotion
    * square. If it is, then we simply return a draw score (0), otherwise
    * we just reduce the evaluation considerably.
    */
   bb = pawn_files;
   if (attacker == WHITE) bb <<= 56;

   /* Check whether the defending king is in the corner (can step onto the
    * promotion square).
    * If he can, it's a draw because he can never be driven away.
    */
   bitboard_t defend = board->bbp[KING] & board->bbc[defender];
   int defender_king_square = bitscan64(defend);
   defend |= king_attack[defender_king_square];

   if (bb & defend) {
      *ev = 0;
      return true;
   }

   /* Evaluate the position as drawish */
   *ev /= 4;
   return true;
}

/* Queen vs rook or knight pawn */
static bool evaluate_queen_pawn(const board_t *board, int player, int *ev, attack_t *attack)
{
   bitboard_t second_rank, promotion_square;
   bitboard_t king[2], pawn, defend, key, king_move;
   int attacker, defender;
   int attacker_king_square, defender_king_square;

   /* The attacking party is the party with the queen */
   attacker = WHITE;
   if (board->bbp[QUEEN] & board->bbc[1]) {
      attacker = 1;
   }
   defender = attacker^1;

   second_rank = board_rank7;
   if (defender != WHITE) {
      second_rank = board_rank2;
   }

   king[defender] = board->bbp[KING] & board->bbc[defender];
   defender_king_square = bitscan64(king[defender]);
   king[attacker] = board->bbp[KING] & board->bbc[attacker];
   attacker_king_square = bitscan64(king[attacker]);
   pawn = board->bbp[PAWN];


   /* If the pawn has not yet advanced to the second rank, then it's
    * usually a win and we don't worry about the exceptions here.
    */
   if (!(board->bbp[PAWN] & second_rank))
      return false;

   /* Squares where the attacking king defends the pawn (or the promotion
    * square).
    * Tempo: if the defender is to move, he can be one square away from the
    * pawn and still defend it.
    */
   if (player == defender)
      defend = king_attack2[defender_king_square];
   else
      defend = king_attack[defender_king_square];

   /* The pawn is on the second rank. If the queen or the attacking king
    * can move to the promotion square safely or take the pawn, then we're
    * fine.
    */
   if (defender == WHITE)
      promotion_square = pawn << 8;
   else
      promotion_square = pawn >> 8;

   if ( (attack->attacks[attacker][QUEEN] | king_attack[attacker_king_square]) &
        (promotion_square | pawn) & ~defend) {
      return false;
   }

   /* The pawn is on the second rank and it is protected. We now need to
    * test whether the attacking king is close enough. To do this, we need
    * to determine the "key squares" that the king has to reach in order to
    * deliver mate. If we can reach those, the position is still won.
    * Otherwise, it is drawn.
    * Key squares are the squares next to the pawn (so the king and the
    * queen both attack the promotion square) and the square diagonally
    * behind the pawn on the b/g file.
    * We also need to know whether the defending king is in position (near
    * the corner); if not we have extra time to bring the attacking king
    * closer.
    * There is a situation that we can detect where the defending king is
    * on the wrong side of the pawn, which gives the attacker more time to
    * bring his king closer.
    */
   defend = 0;
   defend |= (pawn << 1) & board_bfile;
   defend |= (pawn >> 1) & board_gfile;
   defend |= (promotion_square >> 1 | promotion_square << 1) & board_bfile;
   defend |= (promotion_square >> 1 | promotion_square << 1) & board_gfile;
   defend |= (promotion_square | promotion_square >> 2) & board_afile;
   defend |= (promotion_square | promotion_square << 2) & board_hfile;
   defend &= ~(promotion_square & (board_cfile | board_ffile));

   /* If the defending king has not yet reached the corner, then the
    * attacker has an extra tempo.
    * Note that we don't take into account "blocked" squares by the
    * defending king, since the defending king has zugzwang and will be
    * forced to move away, allowing our own king to pass.
    */
   king_move = king[attacker];
   if (!(king[defender] & defend)) {
      king_move |= king_attack[attacker_king_square];

      /* Special case: check whether the defending king can actually move
       * to the corner or not; if he can't, then we're done here too.
       */
      if (attack->attacks[attacker][QUEEN] & defend)
         return false;
   }

   key = 0;
   /* Rook pawn */
   if (pawn & (board_hfile | board_afile)) {
      bitboard_t bb = knight_attack[bitscan64(promotion_square)];
      if (player == attacker) {  /* It's good enough if we can move to a key square */
         while(bb) {
            int square = bitscan64(bb);
            unset_bitboard(&bb, square);
            key |= king_attack2[square];
         }
      } else {                   /* We need to be on the key square */
         key = bb;
      }
   } else {    /* Knight pawn */
      bitboard_t bb = 0;
      if (pawn & board_cfile) {
         bb = pawn >> 1;
         if (attacker)
            bb |= B6_MASK | D7_MASK;
         else
            bb |= B3_MASK | D2_MASK;
      } else {
         bb = pawn << 1;
         if (attacker)
            bb |= G6_MASK | E7_MASK;
         else
            bb |= G3_MASK | E2_MASK;
      }
      /* It's always good if we're already on the key square */
      key = bb;
      if (player == attacker) {  /* It's good enough if we can move to a key square */
         while(bb) {
            int square = bitscan64(bb);
            unset_bitboard(&bb, square);
            key |= king_attack[square];
         }
      }
   }

   /* The attacker wins if the king is on one of the key squares;
    * otherwise, the situation is that the defending king is in the corner
    * and the attacking king is too far away -> draw.
    * We could probably return a mate score through some static detection
    * here, but we just leave that to the search.
    */
   if (key & king_move)
      return false;

   /* Draw */
   *ev = 0;
   return true;
}

/* Queen vs rook
 * We simply adjust the score using a set of heuristics:
 *  - it's good for the attacker if the defending king is close to a corner
 *  - if the defender uses the "third rank defence", move the queen to the
 *    centre to counter it
 *  - it's good for the defender if the attacking king is cut off by the
 *    rook
 *  - recognise the Philidor position
 * TODO: improve, draws far too often vs Crafty
 */
static bool evaluate_queen_rook(const board_t *board, int side_to_move, const attack_t *attack, int *ev)
{
   bitboard_t king[2], rook, queen, kings, bb;
   int attacker, defender;
   int rs, ks, aks;

   /* The attacking party is the party with the queen */
   attacker = WHITE;
   if (board->bbp[QUEEN] & board->bbc[1]) {
      attacker = 1;
   }
   defender = attacker^1;

   kings = board->bbp[KING];
   king[attacker] = board->bbp[KING] & board->bbc[attacker];
   king[defender] = board->bbp[KING] & board->bbc[defender];
   queen = board->bbp[QUEEN];
   rook = board->bbp[ROOK];

   aks = bitscan64(king[attacker]);
   ks = bitscan64(king[defender]);
   rs = bitscan64(rook);

   /* Check if the rook prevents the attacking king from approaching the
    * defending king: this is good for the defender.
    */
   if (square_separates_pieces(rs, kings)) {
      if (defender == WHITE)
         *ev += 40;
      else
         *ev -= 40;
   }

#if 0
   if (square_separates_pieces(rs, board->bbc[attacker])) {
      if (defender == WHITE)
         *ev += 10;
      else
         *ev -= 10;
   }
#endif

   /* Defending king needs to be forced close to the edge */
   if (defender == WHITE)
      *ev += (2*king_centre_table[ks] - king_centre_table[aks]);
   else
      *ev -= (2*king_centre_table[ks] - king_centre_table[aks]);

   if (attacker == WHITE)
      *ev += 2*king_centre_table[bitscan64(queen)];
   else
      *ev -= 2*king_centre_table[bitscan64(queen)];
#if 0
   if ((board_edge & king[defender]) && ((board_offedge|board_xcentre) & rook)) {
      if (attacker == WHITE)
         *ev += 2*centre_table[bitscan64(queen)];
      else
         *ev -= 2*centre_table[bitscan64(queen)];
   }
#endif

   /* Bring the attacking king close to the defending king */
   if (attacker == WHITE)
      *ev -= 20*(king_distance(ks, aks) - 3);
   else
      *ev += 20*(king_distance(ks, aks) - 3);

#if 0
   /* The defending king wants to defend the rook */
   if (attacker == WHITE)
      *ev += (king_distance(ks, rs) - 1);
   else
      *ev -= (king_distance(ks, rs) - 1);
#endif

   /* Bonus for the attacker if the queen can fork the rook and king. */
   bb = (rays[ks] & rays[rs]) & ~diag[aks];
   if (bb & attack->attacks[attacker][QUEEN]) {
      if (attacker == WHITE)
         *ev += 50 + 50*(side_to_move == attacker);
      else
         *ev -= 50 + 50*(side_to_move == attacker);
   }

   bb = king_attack[ks] ^ (king_attack[ks] & king_attack[aks]);
   if (attack->attacks[attacker][QUEEN] & rook & ~bb) {
      if (attacker == WHITE)
         *ev += chess_piece_values[ROOK]/2;
      else
         *ev -= chess_piece_values[ROOK]/2;
   }

   /* Penalise if the defender can give check with the rook */
   bb = attack->attacks[defender][ROOK] & orth[aks] & ~attack->attacks[attacker][QUEEN];
   if (bb) {
      if (defender == WHITE)
         *ev += 10;
      else
         *ev -= 10;
   }

   if (king_attack[aks] & rook) {
      if (attacker == WHITE)
         *ev += 10;
      else
         *ev -= 10;
   }

#if 0
   if (king_attack2[aks] & rook) {
      if (attacker == WHITE)
         *ev += 5;
      else
         *ev -= 5;
   }
#endif

#if 0
   if (attack->attacks[defender][ROOK] & queen) {
      if (defender == WHITE)
         *ev += 10;
      else
         *ev -= 10;
   }
#endif

   /* Penalty if the attacking king is on a diagonal with the rook, which
    * means it gets in the way of the queen.
    */
#if 0
   if (diag[aks] & rook) {
      int bonus = 14;
      if (rook & (A1_MASK|A8_MASK|H1_MASK|H8_MASK)) bonus += 5;
      if (attacker == WHITE)
         *ev -= bonus;
      else
         *ev += bonus;
   }
#endif

   /* Test for Philidor position:
    * Defending king on the edge next to the corner, attacking king on the
    * c or f file a knight's jump away, rook next to the defending king
    * away from the edge and the queen on the edge. Eg:
    *  .k.
    *  .r.
    *  ..K
    *  Q..
    */
   if ((king[defender] & (B8_MASK|A7_MASK | G8_MASK|H7_MASK | B1_MASK|A2_MASK | G1_MASK|H2_MASK)) && 
       (king[attacker] & (C6_MASK|F6_MASK | C3_MASK|F3_MASK)) && (knight_attack[aks] & king[defender]) &&
       (king[defender] & (B7_MASK|G7_MASK | B2_MASK|G2_MASK) & rook)
      ) {
      if (attacker == WHITE)
         *ev += 20;
      else
         *ev -= 20;
   }

   return true;
}

/* King + rook + pawn vs king + rook
 * This is *very* tricky to do generically, and we'll need to leave a lot
 * of this up to the search.
 * The best we can do is score positions like the Philidor position so we
 * can benefit from knowing it's a draw when we find it.
 *
 * FIXME: give a minor bonus to the pawn side: it's still preferable to
 * have the pawn than to give it away, even if the resulting position is
 * still a dead draw, it leads to unnatural play...
 */
static bool evaluate_king_rook_pawn_rook_ending(const board_t *board, int side_to_move, attack_t *attack, int *ev)
{
   bitboard_t king[2], pawn, attacking_rook, defending_rook, promotion_square, pre_promotion_square;
   bitboard_t first_rank, fourth_rank, fifth_rank, sixth_rank, seventh_rank, eight_rank, king_draw_zone;
   int attacker, defender;
   int square, pre_square;

   /* The attacking party is the party with the pawn */
   attacker = WHITE;
   if (board->bbp[PAWN] & board->bbc[1]) {
      attacker = 1;
   }
   defender = attacker^1;

   king[attacker] = board->bbp[KING] & board->bbc[attacker];
   king[defender] = board->bbp[KING] & board->bbc[defender];
   attacking_rook = board->bbp[ROOK] & board->bbc[attacker];
   defending_rook = board->bbp[ROOK] & board->bbc[defender];
   pawn = board->bbp[PAWN];

   /* Find promotion square */
   square = bitscan64(pawn);

   /* We check for Philidor's position and variations thereof; these are
    * known draws.
    */
   if (attacker == WHITE) {
      square |= 7 << 3;
      pre_square = square - 8;
      first_rank = board_rank1;
      fourth_rank = board_rank4;
      fifth_rank = board_rank5;
      sixth_rank = board_rank6;
      seventh_rank = board_rank7;
      eight_rank = board_rank8;
      /* The zone where the king can be cut off */
      king_draw_zone = board_rank5 | board_rank4 | board_rank3 | board_rank2 | board_rank1;
   } else {
      square &= 0x07;
      pre_square = square + 8;
      first_rank = board_rank8;
      fourth_rank = board_rank5;
      fifth_rank = board_rank4;
      sixth_rank = board_rank3;
      seventh_rank = board_rank2;
      eight_rank = board_rank1;
      /* The zone where the king can be cut off */
      king_draw_zone = board_rank4 | board_rank5 | board_rank6 | board_rank7 | board_rank8;
   }
   promotion_square = make_bitboard_square(square);
   promotion_square |= king_attack[square];
   pre_promotion_square = make_bitboard_square(pre_square);
   pre_promotion_square |= king_attack[pre_square];

   /* Sanity check: make sure nothing is under attack, we don't handle that here. */
   if (attack->attacks[attacker][ROOK] & defending_rook ||
       attack->attacks[defender][ROOK] & attacking_rook ||
       attack->attacks[defender][ROOK] & king[attacker] ||
       attack->attacks[attacker][ROOK] & king[defender] ||
       attack->attacks[defender][ROOK] & pawn)
      return false;

   /* Philidor: defending king touches promotion square, pawn on the sixth
    * rank, defending king beyond the sixth rank and cut off on the fourth.
    */
   if (king[defender] & promotion_square &&
       pawn & fifth_rank &&
       king[attacker] & king_draw_zone &&
       defending_rook & sixth_rank) {
      *ev = 0;
      return true;
   }

   /* Philidor, shifted down one rank: defending king touches promotion
    * square, pawn on the fifth
    * rank, defending king beyond the sixth rank and cut off on the fourth.
    */
   if (king[defender] & pre_promotion_square &&
       pawn & fourth_rank &&
       king[attacker] & king_draw_zone &&
       defending_rook & fifth_rank) {
      *ev = 0;
      return true;
   }

   /* An outcome of the Philidor: the pawn has advanced to the sixth rank,
    * then the defending rook needs to be on the first rank.
    */
   if (king[defender] & promotion_square &&
       pawn & sixth_rank &&
       king[attacker] & (sixth_rank | king_draw_zone) &&
       defending_rook & first_rank &&
       side_to_move == defender) {
      *ev = 0;
      return true;
   }

   /* An outcome of the shifted Philidor: the pawn has advanced to the sixth rank,
    * then the defending rook needs to be on the first rank.
    */
   if (king[defender] & pre_promotion_square &&
       pawn & fifth_rank &&
       king[attacker] & king_draw_zone &&
       defending_rook & first_rank &&
       side_to_move == defender) {
      *ev = 0;
      return true;
   }

   /* Back rank defence, when the pawn is too far advanced for the
    * Philidor.
    * Knight and rook pawn always draw, but the defending king needs to
    * defend the corner.
    */
   if (king[defender] & promotion_square &&
       king[defender] & (board_afile | board_bfile | board_gfile | board_hfile) &&
       pawn & (sixth_rank | seventh_rank) &&
       pawn & (board_afile | board_bfile | board_gfile | board_hfile) &&
       king[attacker] & (sixth_rank | king_draw_zone) &&
       defending_rook & (eight_rank & ~(board_hfile|board_afile))) {
      *ev = 0;
      return true;
   }

   /* Back rank defence, when the pawn is too far advanced for the
    * Philidor.
    * Centre and bishops pawns draw if the attacking king has not advanced
    * to the sixth rank.
    */
   if (king[defender] & promotion_square &&
       pawn & (board_cfile | board_dfile | board_efile | board_ffile) &&
       pawn & sixth_rank &&
       king[attacker] & king_draw_zone &&
       ( (defending_rook & (eight_rank & ~(board_hfile|board_afile)) && side_to_move == defender) ||
          defending_rook & first_rank) ) {
      *ev = 0;
      return true;
   }


   return false;
}

/* King + rook + pawn vs king + queen
 * There are some fortress draws.
 */
static bool evaluate_king_rook_pawn_queen_ending(const board_t *board, int side_to_move, attack_t *attack, int *ev)
{
   bitboard_t king[2], pawn, queen, rook;
   bitboard_t pawn_draw_zone, bb, ka;
   int attacker, defender;
   int square;

   /* The attacking party is the party with the queen */
   attacker = WHITE;
   if (board->bbp[QUEEN] & board->bbc[1]) {
      attacker = 1;
   }
   defender = attacker^1;

   /* If it's queen+pawn vs rook, return immediately; we're not dealing
    * with that here.
    */
   if (board->bbc[attacker] & board->bbp[PAWN])
      return false;

   king[attacker] = board->bbp[KING] & board->bbc[attacker];
   king[defender] = board->bbp[KING] & board->bbc[defender];
   rook           = board->bbp[ROOK];
   queen          = board->bbp[QUEEN];
   pawn           = board->bbp[PAWN];

   pawn_draw_zone = board_rank7 | board_bfile | board_gfile | board_rank2;
   if (attacker == WHITE) {
      pawn_draw_zone |= board_rank3;
      pawn_draw_zone ^= A7_MASK | A6_MASK | A3_MASK | H7_MASK | H6_MASK | H3_MASK;
   } else {
      pawn_draw_zone |= board_rank6;
      pawn_draw_zone ^= A2_MASK | A3_MASK | A6_MASK | H2_MASK | H3_MASK | H6_MASK;
   }

   if (!(pawn & pawn_draw_zone))
      return false;

   /* Check if the king can defend the pawn */
   square = bitscan64(king[defender]);
   if (attack->attacks[attacker][QUEEN] & pawn) {
      ka = king_attack[square] & pawn;
      if (side_to_move == defender)
         ka = king_attack2[square] & pawn;
      if (!ka)
         return false;
   } else {
      ka = king_attack2[square] & pawn;
      if (!ka)
         return false;
   }

   /* Check if the pawn can defend the rook. */
   bb = rook | pawn;
   if (!(attack->attacks[defender][PAWN] & (rook|attack->attacks[defender][ROOK])))
      return false;

   /* Check if the attacking king is cut-off from the pawn by the rook. */
   square = bitscan64(rook);
   int file = unpack_file(square);
   int rank = unpack_rank(square);
   bb = king[attacker] | pawn;

   if ((board_north_of_rank[rank] & bb) && (board_south_of_rank[rank] & bb)) {
      *ev = 0;
      return true;
   }

   if ((board_east_of_file[file] & bb) && (board_west_of_file[file] & bb)) {
      *ev = 0;
      return true;
   }

   return false;
}

static bool evaluate_king_rook_minor_ending(const board_t *board, int side_to_move, const attack_t *attack, int *ev)
{
   bitboard_t king[2], rook, minor;
   int attacker, defender;
   int square, asquare;

   /* The attacking party is the party with the rook */
   attacker = WHITE;
   if (board->bbp[ROOK] & board->bbc[1]) {
      attacker = 1;
   }
   defender = attacker^1;

   king[0] = board->bbp[KING] & board->bbc[0];
   king[1] = board->bbp[KING] & board->bbc[1];
   rook = board->bbp[ROOK];
   minor = board->bbp[KNIGHT] + board->bbp[BISHOP];

   if (board->bbp[BISHOP]) {
#if 0
      int heuristic_score = heuritsic_krkb(board, side_to_move);
      if (heuristic_score == heuristic_win || heuristic_score == heuristic_loss) {
         return true;
      }
      if (heuristic_score == heuristic_draw) {
         *ev = 0;
         return true;
      }
#endif

#if 1
      *ev /= 2;
      const bitboard_t centre = board_centre | board_xcentre | board_offedge;
      bitboard_t bb = (king[defender] | minor);
      if ((centre & bb) == bb) {
         *ev /= 8;
         return true;
      }

      /* Bring the kings close together */
      square = bitscan64(king[defender]);
      asquare = bitscan64(king[attacker]);
      if (attacker == WHITE) {
         *ev -= 4*king_distance(square, asquare);
         *ev -= king_centre_table[square];
      } else {
         *ev += 4*king_distance(square, asquare);
         *ev += king_centre_table[square];
      }

      /* Defending king on the edge of the board, cut-off by the attacking
       * king.
       */
      bb = orth[square] & board_edge;
      if (bb &&
          king[attacker] & (~bb | F1_MASK | C1_MASK | F8_MASK | C8_MASK | A3_MASK | A6_MASK | H3_MASK | H6_MASK) &&
          king_attack[asquare] & king_attack[square]) {
         if (attacker == WHITE)
            *ev += 50;
         else
            *ev -= 50;
      }

      /* King near the corner of the bishop colour - bad! */
      bitboard_t corner = A1_MASK | H1_MASK | A8_MASK | H8_MASK;
      bitboard_t kk = king_attack[square] | king[defender];

      corner &= kk;
      if (corner) {
         if (attacker == WHITE)
            *ev -= 4*king_distance(asquare, bitscan64(corner));
         else
            *ev += 4*king_distance(asquare, bitscan64(corner));

         bitboard_t bb = corner | minor;
         if ( !(bb & board_dark) || !(bb & board_light) ){
            return true;
         } else {
            square = bitscan64(corner);
            /* Detect fortress positions */
            *ev /= 4;
            if (attack->attacks[defender][BISHOP] & king_attack[square]) {
               *ev = 0;
               return true;
            }
         }
      }

      return true;
#endif
   }

   return false;
}

/* King + rook vs. king + pawn.
 * Wins in general, but there are a few things that can make this position
 * draw-ish. We drag the score closer to 0 in those cases so the search
 * knows to stay away from them.
 */
static bool evaluate_king_rook_king_pawn_ending(const board_t *board, int side_to_move, attack_t *attacki, int *ev)
{
   bitboard_t king[2], pawn, rook, pre_promotion_square, promotion_square, pawn_square;
   bitboard_t defend, attack, span, back_rank, back_ranks, stop_square, behind;
   int attacker, defender;
   int ks, square, ps, pre_square, pawn_file;

   /* The attacking party is the party with the rook */
   attacker = WHITE;
   if (board->bbp[ROOK] & board->bbc[1]) {
      attacker = 1;
   }
   defender = attacker^1;

   king[attacker] = board->bbp[KING] & board->bbc[attacker];
   king[defender] = board->bbp[KING] & board->bbc[defender];
   rook           = board->bbp[ROOK] & board->bbc[attacker];
   pawn           = board->bbp[PAWN];

   /* Find promotion square */
   ps = square = bitscan64(pawn);
   pawn_file = unpack_file(square);

   if (defender == WHITE) {
      stop_square = pawn << 8;
      pawn_square = white_pawn_square[square];
      square |= 7 << 3;
      pre_square = square - 8;
      back_rank = board_rank8;
      back_ranks = board_rank8 | board_rank7 | board_rank6;
      behind = board_south_of_rank[unpack_rank(ps)];
   } else {
      stop_square = pawn >> 8;
      pawn_square = black_pawn_square[square];
      square &= 0x07;
      pre_square = square + 8;
      back_rank = board_rank1;
      back_ranks = board_rank1 | board_rank2 | board_rank3;
      behind = board_north_of_rank[unpack_rank(ps)];
   }
   promotion_square = make_bitboard_square(square);
   pre_promotion_square = make_bitboard_square(pre_square);
   span = connecting_ray[ps][square];

   bitboard_t bb = board_file[pawn_file];
   bitboard_t east = board_east_of_file[pawn_file];
   bitboard_t west = board_west_of_file[pawn_file];

   /* Get squares controlled by the attacker */
   ks = bitscan64(king[attacker]);
   attack = attacki->attacks[attacker][ROOK] | king_attack[ks];

   /* If the defending king is too far away from the pawn to defend it,
    * abort: the pawn will most likely be lost.
    */
   defend = board_empty;
   ks = bitscan64(king[defender]);
   defend = king_attack2[ks];
   if (side_to_move == attacker && (attack & pawn))
      defend = king_attack[ks];

   /* If the king is cut-off from the pawn by the rook, it cannot defend
    * the pawn and it is likely that the pawn will be lost unless it can
    * promote immediately.
    */
   if ((west & king[defender]) && (west & rook)) {
      bitboard_t ke = board_east_of_file[unpack_file(ks)];
      if (ke & rook) return false;
   }

   if ((east & king[defender]) && (east & rook)) {
      bitboard_t kw = board_west_of_file[unpack_file(ks)];
      if (kw & rook) return false;
   }

   if ((defend & pawn) == board_empty)
      return false;

   /* The king can defend the pawn, where is the attacking king ? */
   attack = board_empty;
   if (side_to_move == attacker)
      attack = king_attack[bitscan64(king[attacker])];
   else
      attack = king[attacker];

   /* Trivial case: if the defender is to move and can savely promote the pawn,
    * the position is at least a draw.
    * The search will find out if it's more than that.
    */
   if (side_to_move == defender && (pawn & pre_promotion_square) && (king_attack[ks] & promotion_square) && !(rook & (bb|back_rank))) {
      *ev = 0;
      return true;
   }

   /* Attacking king in front of the pawn? */
   if (attack & pawn_square) {
      /* Yes - probably a win, leave it to the search */
      return false;
   }

   /* Defending king in front of an edge pawn and cut-off by the rook while
    * the attacking king is far away? Draw.
    */
   if (span & king[defender] & (board_afile|board_hfile)) {
      if ((defend & promotion_square) || (defend & (~attack|span)) == board_empty) {
         *ev /= 16;
         return true;
      }

      /* Draw-ish, unless the king is actually blocking the advance of the
       * pawn, in which case the defender must lose a tempo to advance the
       * pawn.
       */
      if (!(stop_square & king[defender])) {
         *ev /= 8;
         return true;
      }
   }


   /* The defending king is close to the pawn and the attacking king is not
    * in front of the pawn, or at least close to the promotion square.
    * Draw-ish, even more so if the two kings are on the same side of the
    * pawn.
    */
   if ((board->bbp[KING] & east) == board_empty || (board->bbp[KING] & west) == board_empty) {
      *ev /= 4;
      return true;
   }

   /* Draw-ish if the attacking king is too far away and the pawn is close
    * to promotion - but maybe much worse if the rook is not defending the
    * promotion square!.
    */
   if (back_ranks & pawn) {
      if (bb & behind & rook)
         *ev /= 2;
      *ev /= 2;
      return true;
   }
   

   return false;
}

/* King + pawn vs bare king.
 * The search will find this, but it won't judge it properly until ~25
 * plies have been burned. Unfortunately this is not quite at-a-glance,
 * only for the clearest of positions. Still, that should help.
 */
static bool evaluate_king_pawn_ending(int side_to_move, int attacker, bitboard_t *king, bitboard_t pawn, int *ev)
{
   bitboard_t key, defence_key, king_move, square;
   int pawn_square, king_square;
   int defender = attacker^1;

   /* Rule of the square */
   pawn_square = bitscan64(pawn);
   king_square = bitscan64(king[defender]);

   if (attacker == WHITE)  /* White */
      square  = white_pawn_square[pawn_square];
   else                    /* Black */
      square  = black_pawn_square[pawn_square];
   
   if (side_to_move == defender)
      king_move = king_attack[king_square];
   else
      king_move = king[defender];

   /* The defending king is outside the square, hand this back to the main
    * search which should take care of it.
    */
   if (!(square & king_move))
      return false;

   /* The defending king is inside the square. It now matters whether the
    * attacking king can reach one of the key squares.
    */

   /* Determine key squares */
   if (attacker == WHITE) {
      defence_key = key = pawn << 16;
      defence_key |= pawn << 8;
      key |= shift_bitboard_file_left(key);
      key |= shift_bitboard_file_right(key);
      if (pawn & board_black)
         key |= key >> 8;
      if (pawn & board_rank8)
         key |= key >> 8;
      if (pawn & board_rank6)
         defence_key &= board_rank7;
   } else {
      defence_key = key = pawn >> 16;
      defence_key |= pawn >> 8;
      key |= shift_bitboard_file_left(key);
      key |= shift_bitboard_file_right(key);
      if (pawn & board_white)
         key |= key << 8;
      if (pawn & board_rank1)
         key |= key << 8;
      if (pawn & board_rank3)
         defence_key &= board_rank2;
   }

   /* If the defender is to move and can occupy a key square, he draws */
   king_move = king[attacker];
   if (side_to_move == attacker)
      king_move |= king_attack[bitscan64(king[attacker])] & ~king_attack[king_square];

   /* If attacker occupies a key square, then he wins (leave it to the
    * search to find the win)
    */
   if (king_move & key)
      return false;

   /* If the defender can hold the attacking king off the key squares, then
    * it is a draw - provided the pawn cannot advance to the seventh rank.
    */
   king_move = king[defender];
   if (side_to_move == defender)
      king_move |= ~king_attack[bitscan64(king[attacker])] & king_attack[king_square];

   if (king_move & defence_key) {
      *ev = 0;
      return true;
   }

   return false;
}

/* Asses the strength of a pawn duo in the end game.
 * The search will find this eventually.
 * The attacker is the player with the duo.
 */
static bool evaluate_pawn_duo(const board_t *board, const pawn_structure_t *pawn, int side_to_move, int attacker, int *ev)
{
   static const bitboard_t back_rank[2] = {
      board_rank6|board_rank7|board_rank8, 
      board_rank1|board_rank2|board_rank3
   };
   bitboard_t king, king_move, square, duo, pawn_attack[2];
   int pawn_square, king_square;
   int defender = attacker^1;

   /* Don't award a big bonus if the defender still has pieces besides
    * rooks, pawns and a king.
    */
   if ( (board->bbp[KNIGHT]|board->bbp[BISHOP] | board->bbp[QUEEN]) & board->bbc[defender] )
      return false;

   /* More than one rook -> no bonus */
   if (!onebit64(board->bbp[ROOK] & board->bbc[defender]))
      return false;

   /* Make sure the defending king can't step into the square */
   duo = pawn->connected_free & board->bbc[defender] & back_rank[defender];
   square = 0;
   while (duo) {
      pawn_square = bitscan64(duo);
      duo ^= make_bitboard_square(pawn_square);

      if (attacker == WHITE)  /* White */
         square  |= white_pawn_square[pawn_square];
      else                    /* Black */
         square  |= black_pawn_square[pawn_square];
   }

   /* Tempo */
   king = board->bbp[KING] & board->bbc[defender];
   king_square = bitscan64(king);
   if (side_to_move == defender)
      king_move = king_attack[king_square] | king;
   else
      king_move = king;

   /* Remove illegal squares from the king's destination */
   get_pawn_attacks(board, pawn_attack, pawn_attack+1);
   king_move &= ~board->bbc[defender];
   king_move &= ~pawn_attack[attacker];

   /* The king can't stop the pawns */
   if (!(king_move & square)) {
      if (attacker == WHITE)
         *ev += EVAL_ADVANCED_DUO;
      else
         *ev -= EVAL_ADVANCED_DUO;
      return true;
   }

   return false;
}


/* Helper function for the pawn endgame evaluation:
 * Calculate the corresponding squares for either side (0 or 1 for white or
 * black).
 * Based on code from
 * http://chessprogramming.wikispaces.com/Corresponding+Squares
 *
 * Input:
 *    side              - the side for which we want to know the corresponding squares
 *    kings[]           - bitboards for the two kings
 *    free_squares[]    - squares the king may occupy
 *    key               - the key squares the attacker wants to reach
 * Output:
 *    cosq[]            - corresponding squares for each square the attacking king occupies
 */
static void calculate_corresponding_squares(int side, const bitboard_t *kings, const bitboard_t *free_squares, const bitboard_t key, bitboard_t *cosq)
{
   bitboard_t changes = board_all;
   bitboard_t move_attacking_king;
   bitboard_t move_defending_king;
   bitboard_t bb;
   int square;

   /* Initialise list of potential corresponding squares squares; none in
    * case the attacking king is standing on a key square. Otherwise, any
    * square that is free and not attacked by the attacking king is a
    * possibility.
    */
   for (square=A1; square<=H8; square++) {
      if (key & make_bitboard_square(square))
         cosq[square] = board_empty;
      else
         cosq[square] = ~(king_attack[square] | make_bitboard_square(square)) & free_squares[1-side];
   }

   /* Loop through all squares on the board and mark corresponding squares */
   while (changes) {
      for (square=A1; square<=H8; square++) {
         bb = make_bitboard_square(square);
         if (changes & bb) {
            changes ^= bb;    /* Mark this square as done */

            /* Locations where the attacking king can reach the current square */
            move_attacking_king = king_attack[square] & free_squares[side];

            /* Locations where the defending king can defend the current
             * square: squares one king move away from this square or one
             * king move away from the corresponding squares.
             */
            move_defending_king = king_attack[square] | bb                 |
                                  shift_bitboard_file_left(cosq[square])   |
                                  shift_bitboard_file_right(cosq[square])  |
                                  shift_bitboard_row_up(cosq[square])      |
                                  shift_bitboard_row_down(cosq[square])    |
                                  shift_bitboard_left_up(cosq[square])     |
                                  shift_bitboard_left_down(cosq[square])   |
                                  shift_bitboard_right_up(cosq[square])    |
                                  shift_bitboard_right_down(cosq[square]);

            /* Now consider all moves by the attacking king. */
            bb = move_attacking_king;
            while (bb) {
               int king_square = bitscan64(bb);
               bitboard_t bp = make_bitboard_square(king_square);
               bb ^= bp;

               /* Need to revisit this square on next iteration if we
                * find it's a corresponding square.
                */
               changes |= bp * ((cosq[king_square] & ~move_defending_king) != 0);
               cosq[king_square] &= move_defending_king;
            }
         }
      }
   }
}

/* Evaluate pawn end game, with pawns on both sides.
 * Uses the principle of corresponding squares to asses whether either
 * player can force zugzwang and reach one of its goals. Goals can be
 * capturing an enemy pawn, or helping a free pawn advance.
 * A pre condition of this function for this to work with static evaluation
 * is that pawns are blocked: either rammed, backward, or the enemy king is
 * blocking their advance. We restrict the number of free pawns to 1,
 * otherwise the position becomes too dynamic.
 *
 * TODO: recognise the following drawing patterns (fortresses)
 * 6k1/4K2p/7P/8/8/8/8/8 w - -     (also if pawns are base of a chain)
 * 8/5kPp/7P6K1/8/8/8/8 w - -
 * 6k1/4K2p/7P/8/8/8/6p1/8 w - -   (also if pawn is advanced, obviously)
 * 5k2/3K2p1/6P1/8/8/8/8/8 w - -   (also if pawns are base of a chain)
 * 5k2/3K2p1/6P1/8/8/8/7P/8 w - -   (also if pawn is advanced, obviously; also works for base of chain)
 * 4kPp1/6P1/4K3/8/8/8/8 w - -
 * (rook or knight pawn on 7th, king in corner, all white pawns on same file)
 * (rook pawn with defnding king on bishop square)
 * (base of pawn structure rammed on a7/h7 and one open pawn on the edge of the chain)
 */
static bool evaluate_pawn_ending(gamestate_t *game, int side_to_move, int *ev)
{
   pawn_structure_t *pawn_structure;
   bitboard_t corresponding[2];
   bitboard_t all_corresponding[64];
   bitboard_t passed_pawn;
   bitboard_t legal_squares[2];
   bitboard_t free_squares[2];
   bitboard_t move_king[2];
   bitboard_t king[2];
   bitboard_t pawns[2];
   bitboard_t mobile, square;
   bitboard_t goal;
   board_t *board;

   return false;

   board = game->board;
   pawn_structure = evaluate_pawn_structure(game);

   /* Check number of passed pawns; we might evaluate one here, but not more
    * than one. Certainly not free pawns on both sides.
    */
   passed_pawn = pawn_structure->free;
   if (passed_pawn && !onebit64(passed_pawn))
      return false;

   king[0] = board->bbp[KING] & board->bbc[0];
   king[1] = board->bbp[KING] & board->bbc[1];

   pawns[0] = board->bbp[PAWN] & board->bbc[0];
   pawns[1] = board->bbp[PAWN] & board->bbc[1];

   /* Confirm that the position is not dynamic: all pawns are statically or
    * dynamically blocked.
    */
   mobile = 0;
   mobile |= pawns[0] & ~((pawns[0]|pawns[1]) >> 8 | shift_bitboard_right_down(pawns[1]) | shift_bitboard_left_down(pawns[1]));
   mobile |= pawns[1] & ~((pawns[0]|pawns[1]) << 8 | shift_bitboard_right_up(pawns[0]) | shift_bitboard_left_up(pawns[0]));

   /* FIXME: we can actually handle situations where some pawns (other than
    * the free pawn) can be pushed forward and exchanged with support from
    * the king, although we do need extra code to evaluate the situation
    * after the pawn has been pushed. For now, KISS.
    */
   if (mobile != passed_pawn)
      return false;

   legal_squares[0] = ~(pawns[0] | shift_bitboard_right_down(pawns[1]) | shift_bitboard_left_down(pawns[1]));
   legal_squares[1] = ~(pawns[1] | shift_bitboard_right_up(pawns[0]) | shift_bitboard_left_up(pawns[0]));

   /* Tempo */
   move_king[side_to_move] = king_attack[bitscan64(king[side_to_move])];
   move_king[1-side_to_move] = king[1-side_to_move];

   /* Constraint: defending king needs to stay inside the square of any
    * free pawn.
    */
   if (passed_pawn) {
      if (passed_pawn & pawns[0])
         square = white_pawn_square[bitscan64(passed_pawn)];
      else
         square = black_pawn_square[bitscan64(passed_pawn)];
   } else {
      square = board_all;
   }

   /* Evaluate goals ("threats") for black and white.
    * Possible threats:
    *  - the opponent takes a pawn
    *  - the opponent can support his mobile (free?) pawn with the king
    */

   /* White */
   goal = pawns[1];
   goal |= shift_bitboard_file_left(pawn_structure->free & board->bbc[0]);
   goal |= shift_bitboard_file_right(pawn_structure->free & board->bbc[0]);
   goal &= legal_squares[0];

   /* Constraint: the attacking king (white) needs to stay inside the
    * square of the black free pawn.
    */
   free_squares[0] = legal_squares[0];
   free_squares[1] = legal_squares[1];
   if (passed_pawn & pawns[1])
      free_squares[0] &= square;
   calculate_corresponding_squares(0, king, free_squares, goal, all_corresponding);
   corresponding[1] = all_corresponding[bitscan64(king[0])];

   /* Apply constraint: the defending king (black) needs to stay inside the
    * square of the white free pawn.
    */
   if (passed_pawn & pawns[0]) {
      corresponding[1] &= square;
   }

   /* Ditto black */
   goal = pawns[0];
   goal |= shift_bitboard_file_left(pawn_structure->free & board->bbc[1]);
   goal |= shift_bitboard_file_right(pawn_structure->free & board->bbc[1]);
   goal &= legal_squares[1];

   /* Constraint: the attacking king (black) needs to stay inside the
    * square of the white free pawn.
    */
   free_squares[0] = legal_squares[0];
   free_squares[1] = legal_squares[1];
   if (passed_pawn & pawns[0])
      free_squares[1] &= square;
   calculate_corresponding_squares(1, king, free_squares, goal, all_corresponding);
   corresponding[0] = all_corresponding[bitscan64(king[1])];

   /* Apply constraint: the defending king (white) needs to stay inside the
    * square of a black free pawn.
    */
   if (passed_pawn & pawns[1]) {
      corresponding[0] &= square;
   }

   /* Now check if black and white can both reach a corresponding square;
    * if so, neither of them can reach their objective and the position is
    * a draw.
    */
   if ( (move_king[0] & corresponding[0]) &&
        (move_king[1] & corresponding[1]) ) {
      *ev = 0;
      return true;
   }

   /* One of the two sides needs to forfeit, penalise them */
   int sign = 1;
   if (side_to_move == 1) sign = -1;
   if (move_king[side_to_move] & corresponding[side_to_move])
      *ev += sign*50;
   else
      *ev -= sign*50;

   return false;
}



/* Lone king */
static bool evaluate_lone_king(gamestate_t *game, int side_to_move, int *minors, int *majors, int *pawns, int16_t *material, int *ev)
{
   board_t *board = game->board;
   /* Bitmasks for evaluating rook pawns and bad bishops */
   static const bitboard_t good_bishop[] = { board_light|0, board_dark|0 };
   static const bitboard_t back_rows[] = {
      board_rank8 | board_rank7, board_rank1 | board_rank2
   };
   static const bitboard_t corner[] = {
      A1_MASK|B1_MASK|A2_MASK|B2_MASK | A8_MASK|B8_MASK|A7_MASK|B7_MASK,
      H1_MASK|G1_MASK|H2_MASK|G2_MASK | H8_MASK|G8_MASK|H7_MASK|G7_MASK,
   };
   bitboard_t king[2];
   int attacker, defender;
   int attacker_king_square, defender_king_square;

   attacker = 0;
   if (onebit64(board->bbc[0])) {
      attacker = 1;
   }
   defender = attacker^1;

   /* Bare kings */
   if (onebit64(board->bbc[attacker])) {
      *ev = 0;
      return true;
   }

   king[defender] = board->bbc[defender];
   king[attacker] = board->bbp[KING] ^ board->bbc[defender];

   if (!pawns[attacker]) {    /* No pawns */
      /* A single minor is not enough to force mate */
      if (minors[attacker] == 1 && majors[attacker] == 0) {
         *ev = 0;
         return true;
      }

      /* Two knights is likewise not enough */
      if (majors[attacker] == 0 && board->bbp[BISHOP] == 0) {
         *ev = 0;
         return true;
      }
   } else {                   /* Still pawns left */
      /* Almost done; check whether we still have major pieces or knights.
       * The knight might actually draw with a rook pawn if the defending
       * king traps the attacking king in the corner, but we don't detect
       * that here.
       * Drop back to the normal evaluation.
       */
      if (majors[attacker] || board->bbp[KNIGHT]) {
         return false;
      }
      
      /* If all pawns are on the A or H file and the defending
       * king has reached the corner, it is a draw. This is even true for
       * endings with the "wrong bishop".
       */
      if (!(board->bbp[PAWN] & ~board_afile)) {          /* All pawns on A-file */
         /* For white, the light bishop is good, for black the dark bishop */
         if (board->bbp[BISHOP] & good_bishop[attacker]) {
            return false;
         }

         /* If the defending king has reached the promotion square, then it
          * is a draw.
          */
         if (king[defender] & back_rows[defender] & corner[0]) {
            *ev = 0;
            return true;
         }
      } else if (!(board->bbp[PAWN] & ~board_hfile)) {   /* All pawns on H-file */
         /* For white, the dark bishop is good, for black the light bishop */
         if (board->bbp[BISHOP] & good_bishop[1-attacker]) {
            return false;
         }

         /* If the defending king has reached the promotion square, then it
          * is a draw.
          */
         if (king[defender] & back_rows[defender] & corner[1]) {
            *ev = 0;
            return true;
         }
      }

      /* Check for KPK */
      if (pawns[attacker] == 1 && (minors[attacker]+majors[attacker] == 0))  {
         return evaluate_king_pawn_ending(side_to_move, attacker, king, board->bbp[PAWN], ev);
      }

      /* Pawns on multiple files, leave it to the main search.
       * FIXME: should deal with plain KPK and rules of the square...
       * However, the search will catch this quickly enough.
       */
      return false;
   }

   /* Base score: material evaluation.
    * Give a (big) bonus to the side with pieces, mainly to make sure the
    * lone-king ending is preferred to a position where the defender still
    * has a pawn.
    */
   *ev = 100 + material[attacker];

   /* The attacker should try to force the defending king to the edge of
    * the board and bring his own king close.
    */
   defender_king_square = bitscan64(king[defender]);
   attacker_king_square = bitscan64(king[attacker]);

   /* KNBK: the lone king can only be mated in the corner of the bishop
    * colour.
    */
   //int factor = (101 - game->fifty_counter[game->moves_played-1]);
   if (board->bbp[KNIGHT] && board->bbp[BISHOP]) {
      if (board->bbp[BISHOP] & board_dark) {
         *ev += 10 * knbk_table[defender_king_square];
      } else {
         *ev += 10 * knbk_table[flip_horizontal(defender_king_square)];
      }
      /* Prefer to centralise the knight */
      int square = bitscan64(board->bbp[KNIGHT]);
      *ev += king_centre_table[square]/2;
   } else {
      /* How close is the defender to the edge? */
      //row = unpack_rank(defender_king_square);
      //column = unpack_rank(defender_king_square);
      //*ev += 6*(abs(2*row - 7) + abs(2*column - 7));
      *ev -= king_centre_table[defender_king_square];
   }

   /* Manhatten distance between the two kings */
   *ev -= 8*king_distance(attacker_king_square, defender_king_square);

   /* TODO: piece placement for attacking pieces */
   bitboard_t bb = board->bbp[QUEEN];
   while (bb) {
      int square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      *ev += king_centre_table[square];
   }

   if (side_to_move)
      *ev = -*ev;

   return true;
}


/***************************************/
/* Piece-specific evaluation functions */
/***************************************/

/* Evaluate the king safety, from white's point of view. */
static int evaluate_king(gamestate_t *game, const pawn_structure_t *pawn_structure, const int *king_square, const int *majors, const int *minors, eval_info_t *ei)
{
   const board_t *board = game->board;
   attack_t *attack = &ei->attack;
   piece_square_table_t *psq;
   int square;
   int ev = 0;
   uint64_t key;
   int phase;

   prefetch_piecetable(game->king_psq, board->pawn_hash);

   /* The position key depends only on the pawn structure, the castling
    * rights, and the game phase.
    */
   key = board->pawn_hash;

   /* castling rights - approximately: only whether the king may castle or
    * not, not to which side.
    */
   //if (may_castle(board, 0)) key ^= castle_rook_key[0][0];
   //if (may_castle(board, 1)) key ^= castle_rook_key[1][0];

   /* Game phase:
    * phase = 16   - beginning of game
    * phase = 14   - middle game (one piece of each exchanged)
    * phase = 10   - late middle game (three pieces exchanged)
    * phase = 8    - early end game: RR&2minor+RR&2minor, QRR+QRR
    * phase = 6    - end game: RRminor+RRminor, Qminor+Qminor
    * phase = 4    - normal end game: RR+RR, Q+Q, R&minor+R&minor
    * phase = 2    - "simple" ending: R+R, minor-minor
    * phase = 0    - pawn ending
    */
   phase = majors[0]+minors[0] + majors[1]+minors[1];

   /* If we're in the end game, then the game phase isn't used to scale the
    * evaluation tables, so we might as well set it to 0 to avoid
    * unnecessarily recalculating the tables as pieces are exchanged.
    */
   if (phase <= 6) phase = 0;

   /* Factor in the number of pieces of each type, which is used in the
    * construction of the piece square table as well as the scaling of the
    * king safety.
    * Don't bother with pawns (which are already taken into account using
    * the pawn hash) or kings (of which there is always one per side).
    */
   for (int side=0; side<2; side++) {
      for (int piece=KNIGHT; piece<KING; piece++) {
         key ^= piece_key[side][piece][ei->piece_count[side][piece]];
      }
   }

   /* See whether we calculated PSQs for this position before; if not,
    * generate them.
    */
   psq = query_piece_table_entry(game->king_psq, key);
   if (!psq) {
      psq = get_free_piece_table_entry(game->king_psq, key);
      memset(&psq->data, 0, sizeof psq->data);

      /* Construct king PSQ
       *
       * Good squares for a king:
       * - End game: near (free) pawns
       * - End game: in front of enemy free pawns on a key square
       * - End game: on the promotion square of enemy free pawns
       * - Middle game: behind a pawn shield in a corner
       * Avoid:
       * - End game: the edge of the board or corners (except if there are
       *   pawns that draw the king there)
       * - Middle game: standing next to (semi-)open files
       * - The centre
       */
      if (phase <= 6) {            /* End game */
         /* The ideal location for the king depends on the location of
          * pawns on the board.
          * TODO: this can be done better. The rule-of-thumb is that the
          * king should be close to the centre if there are pawns on both
          * wings. However, if there is an outside passed pawn then the
          * king should attack the enemy pawns on the other wing.
          */
         bool both_wings = false;
         both_wings = (board->bbp[PAWN] & board_kwing) ||
                      (board->bbp[PAWN] & board_qwing);
         if (!board->bbp[PAWN] || both_wings) { /* Centralise */
            for (square=0; square<64; square++) {
               psq->data[0][square] = psq->data[1][square] = 
                  centre_table[square];
            }
         } else {                            /* Move to pawns */
            int row, file;
            /* Right side of the board */
            if ( board->bbp[PAWN] & board_kwing ) {
               for (row = 0; row<8; row++) {
                  square = pack_row_file(row, 7);
                  psq->data[0][square] = psq->data[1][square] = centre_table[square-3];
               }
               for (file = 6; file>=0; file--) {
                  for (row = 0; row<8; row++) {
                     square = pack_row_file(row, file);
                     psq->data[0][square] = psq->data[1][square] = psq->data[0][square+1]-2;
                  }
               }
            }

            /* Left side of the board */
            if ( board->bbp[PAWN] & board_qwing ) {
               for (row = 0; row<8; row++) {
                  square = pack_row_file(row, 0);
                  psq->data[0][square] = psq->data[1][square] = centre_table[square+3];
               }
               for (file = 1; file<8; file++) {
                  for (row = 0; row<8; row++) {
                     square = pack_row_file(row, file);
                     psq->data[0][square] = psq->data[1][square] = psq->data[0][square-1]-2;
                  }
               }
            }

         }
      }

      int kpsq[2][64];
      memset(kpsq, 0, sizeof kpsq);
      for (int side = 0; side<2; side++) {
         int w = 8;
         int wf = 24;
         for (int n=0; n<5; n++) {
            int ww = ei->piece_count[side][n] * piece_attack_weight[n];
            for (int s=0; s<64; s++) {
               kpsq[side][s] -= ww * piece_max_moves[n][s];
            }
            w += ww;
         }
         if (w < wf) w = wf;
         for (int s=0; s<64; s++) {
            kpsq[side][s] *= (w - wf);
            kpsq[side][s] /= (initial_army_strength - wf);
            kpsq[side][s] /= 3;

            int ksqw = piece_attack_weight[KING]*(3*piece_max_moves[KING][s]/2 + centre_table[s]-6)/4;
            kpsq[side][s] += (initial_army_strength - w) * ksqw / (initial_army_strength-wf);
         }
      }
      kpsq[0][A1] = kpsq[0][B1]-1;
      kpsq[0][H1] = kpsq[0][G1]-1;
      kpsq[1][A8] = kpsq[1][B8]-1;
      kpsq[1][H8] = kpsq[1][G8]-1;

      for (square=0; square<64; square++) {
         psq->data[0][square] += kpsq[0][square];
         psq->data[1][square] += kpsq[1][square];
      }


      /* Store entry for later retrieval */
#ifdef DEBUG_PIECE_SQUARE_TABLES
      int sum[2];
      sum[0] = sum[1] = 0;
      for (square = A1; square <= H8; square++) {
         int n;
         for (n=0; n<2; n++) {
            sum[n] += psq->data[n][square];
         }
      }
      sum[0] /= 64;
      sum[1] /= 64;
      printf("King <PSQ>:   % 3d % 3d\n", sum[0], sum[1]);
#endif
   }

   /* We now have the dynamic king PSQ, adjust the score */
   ev += psq->data[0][king_square[0]];
   ev -= psq->data[1][king_square[1]];

   attack->attacks[0][KING] = king_attack[king_square[0]];
   attack->attacks[1][KING] = king_attack[king_square[1]];

   /* Pawn shelter */
   if (phase > 6) {
      for (int side=0; side<2; side++) {
         int rank = unpack_rank(king_square[side]);
         int file = unpack_file(king_square[side]);
         int prank = rank + 1 - 2*side;
         int score = 0;
         if (prank < 0) prank = 0;
         if (prank > 7) prank = 7;
         bitboard_t bb = king_attack[king_square[side]] & board_rank[prank];

         if ((board->bbp[PAWN] & board->bbc[side] & bb) == bb)
            continue;

         /* Penalise the king for being on or next to semi-open files, open
          * files, and open diagonals.
          */
         if (pawn_structure->open_files & (1 << file))
            score += KING_SHELTER_OPEN;

         if (pawn_structure->half_open_files[side] & (1 << file))
            score += KING_SHELTER_OWN_SEMI;

         if (pawn_structure->half_open_files[1-side] & (1 << file))
            score += KING_SHELTER_OWN_SEMI;

         /* Next to open file/semi open file? */
         if (pawn_structure->open_files & (1 << (file-1)))
            score += KING_SHELTER_OPEN/2;

         if (pawn_structure->half_open_files[side] & (1 << (file-1)))
            score += KING_SHELTER_OWN_SEMI/2;

         if (pawn_structure->half_open_files[1-side] & (1 << (file-1)))
            score += KING_SHELTER_OWN_SEMI/2;

         /* Next to open file/semi open file? */
         if (pawn_structure->open_files & (1 << (file+1)))
            score += KING_SHELTER_OPEN/2;

         if (pawn_structure->half_open_files[side] & (1 << (file+1)))
            score += KING_SHELTER_OWN_SEMI/2;

         if (pawn_structure->half_open_files[1-side] & (1 << (file+1)))
            score += KING_SHELTER_OWN_SEMI/2;

         /* Open diagonal? */
         bb = side ? board_black & board_south_of_rank[rank]
                   : board_white & board_north_of_rank[rank];
         bb &= diag[king_square[side]];

         if ((bb & board->bbp[PAWN] & board->bbc[side]) == board_empty) {
            int sqcol = (file^rank)&1;
            bitboard_t mask = sqcol ? board_light : board_dark;
            int factor = 0;

            if (mask & board->bbp[BISHOP] & board->bbc[1-side])
               factor = 1;

            if (board->bbp[QUEEN] & board->bbc[1 - side])
               factor = 2 * (factor+1);

            if (mask & board->bbp[BISHOP] & board->bbc[side] && factor)
               factor--;

            score += factor * KING_SHELTER_DIAG;
         }

         /* Enemy pawns storming the pawn shelter */
#if 0
         for (int f=file-1; f<file+2; f++) {
            if (file < 0 || file > 7) continue;
            bb = board->bbp[PAWN] & board->bbc[1 - side] & board_file[f];

            if (bb) {
               int pawn_storm[8] = { 0, 0, 30, 15, 5, 0, 0, 0, };
               int sq = side ? forward_bitscan64(bb) : bitscan64(bb);
               int rank = unpack_rank(sq);
               if (side == 1) rank = 7 - rank;
               score -= pawn_storm[rank];
            }
         }
#endif

         score = (score * (phase-6)) / 10;

         ev -= side ? -score : score;
      }
   }

   /* Adjust score for attacks on squares near the king, but only in the
    * middle game
    * TODO: pawn shelter, pawn storm
    */
   if (king_safety_weight[phase] > 1) {
      /* King safety evaluation, based on the description of Rebel's king
       * safety evaluation by Ed Schroeder.
       * Apart from the implementation, differences include scaling with
       * game phase, rather than switching king safety on or off depending
       * on the presence of an enemy queen.
       * There are three types of square around the king: squares behind
       * and next to the king (A), squares on the row directly in front of
       * the king (B) and squares two rows in front of the king (C).
       */
      bitboard_t sq, sqa, sqb, sqc, sqabc, weak;
      bitboard_t attackers[2];
      int index, counter, scale;
      /* Tables taken from Ed Schroeder's writeup for Rebel */
      static const int pattern[] = {
      //      . P N N R R R R Q Q Q Q Q Q Q Q K K K K K K K K K K K K K K K K
      //            P   P N N   P N N R R R R   P N N R R R R Q Q Q Q Q Q Q Q
      //                    P       P   N N N       P   P N N   P N N R R R R
              0,0,0,0,0,0,1,1,0,1,2,2,2,3,3,3,0,0,0,0,1,1,2,2,2,3,3,3,3,3,3,3 };

      /* All attacked squares, for either player */
      attackers[0] = attack->attacks[0][PAWN] | attack->attacks[0][KNIGHT] |
                     attack->attacks[0][BISHOP] | attack->attacks[0][ROOK] |
                     attack->attacks[0][QUEEN];
      attackers[1] = attack->attacks[1][PAWN] | attack->attacks[1][KNIGHT] |
                     attack->attacks[1][BISHOP] | attack->attacks[1][ROOK] |
                     attack->attacks[1][QUEEN];

      /* White king */
      sq = make_bitboard_square(king_square[0]);
      sq |= shift_bitboard_file_left(sq);
      sq |= shift_bitboard_file_right(sq);
      sqa = (sq >> 8 | sq)^make_bitboard_square(king_square[0]);
      sqb = sq << 8;
      sqc = sq << 16;
      sqabc = sqa | sqb | sqc;

      /* Penalty if the king is further away from the back rank */
      counter = 2*unpack_rank(king_square[0]);

      /* Attack pattern on "A", "B" and "C" squares */
      /* TODO: replace with "strength of attacking force", as in Sjaak */
      index = 0;
      index |= 0x01 * ((sqabc & attack->attacks[1][PAWN]) != 0);
      index |= 0x02 * ((sqabc & (attack->attacks[1][KNIGHT]|attack->attacks[1][BISHOP])) != 0);
      index |= 0x04 * ((sqabc & attack->attacks[1][ROOK]) != 0);
      index |= 0x08 * ((sqabc & attack->attacks[1][QUEEN]) != 0);
      index |= 0x10 * ((sqabc & king_attack[king_square[1]]) != 0);
      counter += pattern[index];

      /* Number of attackers on "A" and "B" squares; includes king */
      counter += popcount64((attackers[1] | king_attack[king_square[1]]) & (sqa | sqb));

      /* Penalise more if there are no defenders for "A" and "B" type squares
       * (other than the king itself)
       */
      counter += popcount64(attackers[1] & ~attackers[0] & (sqa | sqb));

      /* Penalise missing pieces on "B" type squares */
      counter += popcount64(sqb) - popcount64(sqb & board->bbc[0]);

      /* Penalise weak pawns on "B" and "C" type squares, more so if
       * there's pressure from queen/rook
       */
      weak = (sqb | sqc) & pawn_structure->weak & board->bbc[0];
      counter += 2*popcount64(weak);
      counter += popcount64(weak & (attack->attacks[1][ROOK]|attack->attacks[1][QUEEN]));

      /* Adjust score */
      scale = 64;
      if ((board->bbp[QUEEN] & board->bbc[1]) == 0) scale *= 2;
      ev -= king_safety_penalty[counter] * king_safety_weight[phase] / scale;

      /* Black king */
      sq = make_bitboard_square(king_square[1]);
      sq |= shift_bitboard_file_left(sq);
      sq |= shift_bitboard_file_right(sq);
      sqa = (sq << 8 | sq)^make_bitboard_square(king_square[1]);
      sqb = sq >> 8;
      sqc = sq >> 16;
      sqabc = sqa | sqb | sqc;

      /* Penalty if the king is further away from the back rank */
      counter = 2*(7 - unpack_rank(king_square[1]));

      /* Attack pattern on "A", "B" and "C" squares */
      index = 0;
      index |= 0x01 * ((sqabc & attack->attacks[0][PAWN]) != 0);
      index |= 0x02 * ((sqabc & (attack->attacks[0][KNIGHT]|attack->attacks[0][BISHOP])) != 0);
      index |= 0x04 * ((sqabc & attack->attacks[0][ROOK]) != 0);
      index |= 0x08 * ((sqabc & attack->attacks[0][QUEEN]) != 0);
      index |= 0x10 * ((sqabc & king_attack[king_square[0]]) != 0);
      counter += pattern[index];

      /* Number of attackers on "A" and "B" squares; includes king */
      counter += popcount64((attackers[0] | king_attack[king_square[0]]) & (sqa | sqb));

      /* Penalise more if there are no defenders for "A" and "B" type squares
       * (other than the king itself)
       */
      counter += popcount64(attackers[0] & ~attackers[1] & (sqa | sqb));

      /* Penalise missing pieces on "B" type squares */
      counter += popcount64(sqb) - popcount64(sqb & board->bbc[1]);

      /* Penalise weak pawns on "B" and "C" type squares, more so if
       * there's pressure from queen/rook
       */
      weak = (sqb | sqc) & pawn_structure->weak & board->bbc[1];
      counter += 2*popcount64(weak);
      counter += popcount64(weak & (attack->attacks[0][ROOK]|attack->attacks[0][QUEEN]));

      /* Adjust score */
      scale = 64;
      if ((board->bbp[QUEEN] & board->bbc[0]) == 0) scale *= 2;
      ev += king_safety_penalty[counter] * king_safety_weight[phase] / scale;
   }

   return ev;
}

/* Asses the positions of the knights by examining the pawn structure and the
 * presence of other pieces.
 * Computes new piece-square tables and then applies these to the current
 * position.
 */
static int evaluate_knights(gamestate_t *game, const pawn_structure_t *pawn_structure, const int *king_square, attack_t *attack)
{
   const board_t *board = game->board;
   piece_square_table_t *psq;
   bitboard_t bb;
   int square;
   int ev = 0;
   uint64_t key;

   /* Nothing to do if there are no knights left */
   if (!board->bbp[KNIGHT])
      return 0;

   /* The position key depends only on the pawn structure */
   key = get_knight_table_key(board);

   /* See whether we calculated PSQs for this position before; if not,
    * generate them.
    */
   psq = query_piece_table_entry(game->knight_psq, key);
   if (!psq) {
      bitboard_t pawns[2];
      bitboard_t stop_squares[2];
      int first = 0;
      int last = 2;
      int n;

      psq = get_free_piece_table_entry(game->knight_psq, key);
      //psq = &knight_psq;

      /* If white has no knights, don't update the table for white.
       * Ditto black.
       */
      /* FIXME: this leads to an inconsistency in that it's possible for
       * the pawn structure to first appear in a position without knights
       * on one side, and then later it emerges with knights on that side
       * in a different branch. In this situation we do need both tables...
       * A reasonable alternative might be to look at the root board: if
       * the player has no knights there he probably won't have any later.
       * Underpromotions should be rare enough that it doesn't matter
       * much...
       * Ditto bishops.
       */
      if (!(game->root_board->bbp[KNIGHT] & game->root_board->bbc[0]))
         first = 1;

      if (!(game->root_board->bbp[KNIGHT] & game->root_board->bbc[1]))
         last = 1;

      pawns[0] = board->bbp[PAWN] & board->bbc[0];
      pawns[1] = board->bbp[PAWN] & board->bbc[1];
      /* Construct knight PSQ
       *
       * Good squares for a knight:
       * - central squares
       * - outposts (closer to centre and on enemy side of board is better)
       * - stop squares
       * - central squares
       * Avoid:
       * - squares where most possible destinations are blocked
       * - squares where we block our own pawn
       * - corners
       */

      /* Basic (static) centralisation score */
      for (n=first; n<last; n++) {
         int square;
         for (square=A1; square<=H8; square++)
            psq->data[n][square] = centre_table[square];
      }

      /* Extra bonus: knight on the 5th or 6th rank. */
      for (square=A5; square<=H6; square++) psq->data[0][square] += 2;
      for (square=A3; square<=H4; square++) psq->data[1][square] += 2;

      /* Calculate stop squares */
      stop_squares[0] = (pawn_structure->free & board->bbc[1]) >> 8;
      stop_squares[1] = (pawn_structure->free & board->bbc[0]) << 8;

      /* Outposts */
      for (n=first; n<last; n++) {
         bitboard_t bp = 0;

         bb = pawn_structure->outposts[n];
         while(bb) {
            int8_t score = KNIGHT_OUTPOST_BASE;
            square = bitscan64(bb);
            bb ^= make_bitboard_square(square);

            /* The outpost is more important if it is on the opponent's side
             * of the board and if it is closer to the centre of the board.
             */
            score += KNIGHT_OUTPOST_CENTRE * centre_table[square]/16;
            if (n==0)
               score += KNIGHT_OUTPOST_RANK * (unpack_rank(square))/16;
            else
               score += KNIGHT_OUTPOST_RANK * (7 - unpack_rank(square))/16;

            psq->data[n][square] += score;

            /* Squares from which the knight can reach the strong square
             * are also good (because it can reach the strong square from
             * there).
             * This makes the evaluation a bit "smoother" and makes it
             * easier for the engine to find positional enhancements.
             */
            bp |= knight_attack[square];
            //bp = knight_attack2[square] ^ make_bitboard_square(square);
            //while (bp) {
            //   square = bitscan64(bp);
            //   bp ^= make_bitboard_square(square);
            //   psq->data[n][square] += score/4;
            //}
         }
         static const int outpost_step_bonus[] = { 0, 7, 8, 9, 9, 9, 9, 9, 9};
         while (bp) {
            square = bitscan64(bp);
            bp ^= make_bitboard_square(square);
            psq->data[n][square] += outpost_step_bonus[popcount64(knight_attack[square] & pawn_structure->outposts[n])];
         }
      }

      /* Stop squares
       * These are already evaluated (for all pieces) in the main
       * evaluation, so only award squares from which we can jump to the
       * stop square.
       */
      for (n=first; n<last; n++) {
         bitboard_t bp = 0;
         bb = stop_squares[n];
         while(bb) {
            square = bitscan64(bb);
            bb ^= make_bitboard_square(square);
            bp |= knight_attack[square];
         }
         while (bp) {
            square = bitscan64(bp);
            bp ^= make_bitboard_square(square);
            psq->data[n][square] += KNIGHT_STOPSQ_ATTACK;
         }
      }

      /* Attacking weak pawns */
      for (n=first; n<last; n++) {
         bitboard_t bp = 0;
         bb = pawn_structure->weak & pawns[1-n];
         while(bb) {
            square = bitscan64(bb);
            bb ^= make_bitboard_square(square);
            bp |= knight_attack[square];
         }
         while (bp) {
            square = bitscan64(bp);
            bp ^= make_bitboard_square(square);
            psq->data[n][square] += KNIGHT_WEAK_PAWN_ATTACK;
         }
      }

      /* Evaluate "blocked" squares: squares that either contain a friendly
       * pawn, or are attacked by an enemy pawn.
       * This is quite slow in practice... even if we cache the results.
       */
      for (n = first; n<last; n++) {
         bitboard_t bp, a;

         bb = pawns[n];
         if (n == 0) {
            bb |= shift_bitboard_right_down(pawns[1-n]);
            bb |= shift_bitboard_left_down(pawns[1-n]);
         } else {
            bb |= shift_bitboard_right_up(pawns[1-n]);
            bb |= shift_bitboard_left_up(pawns[1-n]);
         }

         /* Now mark all squares for which knight movement is impaired */
         bp = bb;
         a = 0;
         while (bp) {
            square = bitscan64(bp);
            bp ^= make_bitboard_square(square);

            a |= knight_attack[square];
         }

         /* Board scan: find bad squares for the knights */
         while (a) {
            int bits;
            square = bitscan64(a);
            a ^= make_bitboard_square(square);

            bp = knight_attack[square] & ~bb;

            if (!bp) {  /* Knight has no moves at all here, very bad! */
               psq->data[n][square] = KNIGHT_NOMOVES_PENALTY;
               continue;
            }
            bits = sparse_popcount64(bp);

            /* The penalty is *worse* if few bits are set on squares near
             * the centre; the thought behind this is that the knight has
             * reduced mobility on the rim anyway, and we don't want to
             * double-count.
             */
            switch (bits) {
               case 1:
                  psq->data[n][square] += KNIGHT_FEWMOVES_BASE1 - KNIGHT_FEWMOVES_CENTRE*centre_table[square]/16;
                  break;

               case 2:
                  psq->data[n][square] += KNIGHT_FEWMOVES_BASE2 - KNIGHT_FEWMOVES_CENTRE*centre_table[square]/16;
                  break;

               case 3:
                  psq->data[n][square] += KNIGHT_FEWMOVES_BASE3 - KNIGHT_FEWMOVES_CENTRE*centre_table[square]/16;
                  break;
            }
         }
      }

      /* Store entry for later retrieval */
#ifdef DEBUG_PIECE_SQUARE_TABLES
      int sum[2];
      sum[0] = sum[1] = 0;
      for (square = A1; square <= H8; square++) {
         for (n=0; n<2; n++) {
            sum[n] += psq->data[n][square];
         }
      }
      sum[0] /= 64;
      sum[1] /= 64;
      printf("Knight <PSQ>: % 3d % 3d\n", sum[0], sum[1]);
#endif
   }

   /* We now have the dynamic knight PSQ, adjust the score */
   bb = board->bbp[KNIGHT] & board->bbc[0];
   while (bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      ev += psq->data[0][square];
      attack->attacks[0][KNIGHT] |= knight_attack[square];
   }

   bb = board->bbp[KNIGHT] & board->bbc[1];
   while (bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      ev -= psq->data[1][square];
      attack->attacks[1][KNIGHT] |= knight_attack[square];
   }

   return ev;
}

/* Asses the positions of the bishops by examining the pawn structure and the
 * presence of other pieces.
 * Computes new piece-square tables and then applies these to the current
 * position.
 */
static int evaluate_bishops(gamestate_t *game, const pawn_structure_t *pawn_structure, const int *king_square, attack_t *attack)
{
   const board_t *board = game->board;
   piece_square_table_t *psq;
   bitboard_t bb, a, xa;
   int square;
   int ev = 0;
   uint64_t key;

   /* Nothing to do if there are no bishops left */
   if (!board->bbp[BISHOP])
      return 0;

   /* The position key depends only on the pawn structure */
   key = get_bishop_table_key(board);

   /* See whether we calculated PSQs for this position before; if not,
    * generate them.
    */
   psq = query_piece_table_entry(game->bishop_psq, key);
   if (!psq) {
      bitboard_t pawns[2];
      bitboard_t blocked_pawns;
      int first = 0;
      int last = 2;
      int n;

      psq = get_free_piece_table_entry(game->bishop_psq, key);
      memset(psq->data, 0, sizeof psq->data);

      /* If white has no bishops, don't update the table for white.
       * Ditto black.
       */
      if (!(game->root_board->bbp[BISHOP] & game->root_board->bbc[0]))
         first = 1;

      if (!(game->root_board->bbp[BISHOP] & game->root_board->bbc[1]))
         last = 1;

      pawns[0] = board->bbp[PAWN] & board->bbc[0];
      pawns[1] = board->bbp[PAWN] & board->bbc[1];

      /* Construct bishop PSQ
       *
       * Good squares for a bishop:
       * - outposts (closer to centre and on enemy side of board is better)
       * - stop squares
       * - central squares
       * - fiancetto bishops
       * - TODO: find out whether it's good to place a bishop on the row below a pawn chain
       * Avoid:
       * - squares where we block our own pawn
       * - forward diagonals that are blocked by blocked pawns (bad bishop)
       * - forward diagonals with more than one own pawn (bad bishop)
       * - trapped bishops on a7/h7 (classical trap)
       * - start square if both pawns are still blocking
       */

      /* Basic (static) centralisation score */
      for (square=A1; square<=H8; square++) {
         psq->data[0][square] += centre_table2[square];
         psq->data[1][square] += centre_table2[H8-square];
      }

      /* Outposts */
      for (n=first; n<last; n++) {
         bb = pawn_structure->outposts[n] & ~board_edge;
         while(bb) {
            int8_t score = BISHOP_OUTPOST_BASE;
            square = bitscan64(bb);
            bb ^= make_bitboard_square(square);

            /* The outpost is more important if it is on the opponent's side
             * of the board and if it is closer to the centre of the board.
             */
            score += centre_table[square]/2;
            if (n==0)
               score += (unpack_rank(square))/2;
            else
               score += (7 - unpack_rank(square))/2;

            psq->data[n][square] += score;

            /* Squares from which the bishop can reach the strong square
             * are also good (because it can reach the strong square from
             * there).
             * This makes the evaluation a bit "smoother" and makes it
             * easier for the program to find positional enhancements.
             * *TODO*
             */
         }
      }

      /*
       * Find blocked diagonals
       *
       * A diagonal is considered blocked in the forward direction only,
       * being outside the pawn chain is not such a bad thing.
       *
       * A diagonal is considered blocked by
       *  1. blocked pawns
       *  2. pawn chains
       * 
       * However, we only go a limited distance bahind the pawn chain. The
       * idea is that if we are far enough behind the pawn, then we still
       * have good mobility.
       * A smaller penalty is awarded for diagonals behind non-blocked
       * pawns.
       *
       * FIXME: should the penalty be larger for a central pawn?
       */
      /* White */
      if (first == 0) {
      blocked_pawns = pawns[0] & (board->bbp[PAWN]>>8);
      blocked_pawns &= (board->bbc[1]>>8);

      /* Add the back pawn in a chain as a "blocked pawn". It's not really
       * blocked, but the diagonal is blocked, and we should not put a
       * bishop behind it
       */
      blocked_pawns |= pawns[0] & (shift_bitboard_left_down(pawns[0]));
      blocked_pawns |= pawns[0] & (shift_bitboard_right_down(pawns[0]));

      /* Being on the same diagonal as a blocked pawn is a bad idea. Find
       * the back diagonal spans of all blocked pawns and mark those
       * squares as bad. Squares on the intersection are marked twice, so
       * that bishops that are trapped from multiple sides get a double
       * penalty.
       */
      bb = blocked_pawns;
      bb |= shift_bitboard_left_down(bb);
      bb |= shift_bitboard_left_down(bb);
      bb |= shift_bitboard_left_down(bb);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[0][square] -= BISHOP_BLOCKED_PENALTY;
      }
      bb = blocked_pawns;
      bb |= shift_bitboard_right_down(bb);
      bb |= shift_bitboard_right_down(bb);
      bb |= shift_bitboard_right_down(bb);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[0][square] -= BISHOP_BLOCKED_PENALTY;
      }

      /* We give a smaller penalty for non-blocked pawns */
      bb = pawns[0] & ~blocked_pawns;
      bb |= shift_bitboard_left_down(bb);
      bb |= shift_bitboard_left_down(bb);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[0][square] -= BISHOP_SEMI_BLOCKED_PENALTY;
      }
      bb = pawns[0] & ~blocked_pawns;
      bb |= shift_bitboard_right_down(bb);
      bb |= shift_bitboard_right_down(bb);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[0][square] -= BISHOP_SEMI_BLOCKED_PENALTY;
      }
      }

      /* Black */
      if (last == 2) {
      blocked_pawns = pawns[1] & (board->bbp[PAWN]<<8);
      blocked_pawns &= (board->bbc[0]<<8);

      /* Add the back pawn in a chain as a "blocked pawn". It's not really
       * blocked, but the diagonal is blocked, and we should not put a
       * bishop behind it
       */
      blocked_pawns |= pawns[1] & (shift_bitboard_left_up(pawns[1]));
      blocked_pawns |= pawns[1] & (shift_bitboard_right_up(pawns[1]));

      /* Being on the same diagonal as a blocked pawn is a bad idea. Find
       * the back diagonal spans of all blocked pawns and mark those
       * squares as bad. Squares on the intersection are marked twice, so
       * that bishops that are trapped from multiple sides get a double
       * penalty.
       */
      bb = blocked_pawns;
      bb |= shift_bitboard_left_up(bb);
      bb |= shift_bitboard_left_up(bb);
      bb |= shift_bitboard_left_up(bb);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[1][square] -= BISHOP_BLOCKED_PENALTY;
      }
      bb = blocked_pawns;
      bb |= shift_bitboard_right_up(bb);
      bb |= shift_bitboard_right_up(bb);
      bb |= shift_bitboard_right_up(bb);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[1][square] -= BISHOP_BLOCKED_PENALTY;
      }

      /* We give a smaller penalty for non-blocked pawns */
      bb = pawns[1] & ~blocked_pawns;
      bb |= shift_bitboard_left_up(bb);
      bb |= shift_bitboard_left_up(bb);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[1][square] -= BISHOP_SEMI_BLOCKED_PENALTY;
      }
      bb = pawns[1] & ~blocked_pawns;
      bb |= shift_bitboard_right_up(bb);
      bb |= shift_bitboard_right_up(bb);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[1][square] -= BISHOP_SEMI_BLOCKED_PENALTY;
      }
      }


      /* Blocking unadvanced central pawns is also bad */
      if (pawns[0] & D2_MASK)
         psq->data[0][D3] -= BISHOP_BLOCKING_PENALTY;

      if (pawns[0] & E2_MASK)
         psq->data[0][E3] -= BISHOP_BLOCKING_PENALTY;

      if (pawns[1] & D7_MASK)
         psq->data[1][D6] -= BISHOP_BLOCKING_PENALTY;

      if (pawns[1] & E7_MASK)
         psq->data[1][E6] -= BISHOP_BLOCKING_PENALTY;

      /* Bishop fiancetto, compensate for the weakness of the pawn structure */
      if ( (pawns[0] & (C2_MASK|B3_MASK)) == (C2_MASK|B3_MASK)) {
         //psq->data[0][C3] += backward_penalty[1]/2+1;
         ////psq->data[0][A3] += backward_penalty[1]/2+1;
         if (pawns[0] & A2_MASK) {
            psq->data[0][B2] += backward_penalty[2]+1;
         } else {
            psq->data[0][B2] += backward_penalty[1]+1;
         }
      }

      if ( (pawns[0] & (F2_MASK|G3_MASK)) == (F2_MASK|G3_MASK)) {
         //psq->data[0][F3] += backward_penalty[1]/2+1;
         //psq->data[0][H3] += backward_penalty[1]/2+1;
         if (pawns[0] & H2_MASK) {
            psq->data[0][G2] += backward_penalty[2]+1;
         } else {
            psq->data[0][G2] += backward_penalty[1]+1;
         }
      }

      if ( (pawns[1] & (C7_MASK|B6_MASK)) == (C7_MASK|B6_MASK)) {
         //psq->data[0][C3] += backward_penalty[1]/2+1;
         //psq->data[0][A3] += backward_penalty[1]/2+1;
         if (pawns[0] & A7_MASK) {
            psq->data[1][B7] += backward_penalty[2]+1;
         } else {
            psq->data[1][B7] += backward_penalty[1]+1;
         }
      }

      if ( (pawns[1] & (F7_MASK|G6_MASK)) == (F7_MASK|G6_MASK)) {
         //psq->data[0][F7] += backward_penalty[1]/2+1;
         //psq->data[0][H7] += backward_penalty[1]/2+1;
         if (pawns[0] & H7_MASK) {
            psq->data[1][G7] += backward_penalty[2]+1;
         } else {
            psq->data[1][G7] += backward_penalty[1]+1;
         }
      }

      /* Bishop trap; the PSQ value here is more than a pawn because we
       * *really* need to discourage this sort of thing, unless we can pull
       * the bishop back IMMEDIATELY and eliminate the bad score before it
       * is picked up by the evaluation function.
       */
      if ( (pawns[0] & (C2_MASK|B2_MASK)) == (C2_MASK|B2_MASK)||
           (pawns[0] & (C2_MASK|B3_MASK)) == (C2_MASK|B3_MASK))
         psq->data[1][A2] = -127;

      if ( (pawns[0] & (F2_MASK|G2_MASK)) == (F2_MASK|G2_MASK)||
           (pawns[0] & (F2_MASK|G3_MASK)) == (F2_MASK|G3_MASK))
         psq->data[1][H2] = -127;

      if ( (pawns[1] & (C7_MASK|B7_MASK)) == (C7_MASK|B7_MASK)||
           (pawns[1] & (C7_MASK|B6_MASK)) == (C7_MASK|B6_MASK))
         psq->data[0][A7] = -127;

      if ( (pawns[1] & (F7_MASK|G7_MASK)) == (F7_MASK|G7_MASK)||
           (pawns[1] & (F7_MASK|G6_MASK)) == (F7_MASK|G6_MASK))
         psq->data[0][H7] = -127;

      /* Same idea, but shifted by one file. The penalty is slightly
       * smaller.
       */
      if ( (pawns[0] & (C3_MASK|B3_MASK)) == (C3_MASK|B3_MASK)||
           (pawns[0] & (C3_MASK|B4_MASK)) == (C3_MASK|B4_MASK))
         psq->data[1][A3] = -80;

      if ( (pawns[0] & (F3_MASK|G3_MASK)) == (F3_MASK|G3_MASK)||
           (pawns[0] & (F3_MASK|G4_MASK)) == (F3_MASK|G4_MASK))
         psq->data[1][H3] = -80;

      if ( (pawns[1] & (C6_MASK|B6_MASK)) == (C6_MASK|B6_MASK)||
           (pawns[1] & (C6_MASK|B5_MASK)) == (C6_MASK|B5_MASK))
         psq->data[0][A6] = -80;

      if ( (pawns[1] & (F6_MASK|G6_MASK)) == (F6_MASK|G6_MASK)||
           (pawns[1] & (F6_MASK|G5_MASK)) == (F6_MASK|G5_MASK))
         psq->data[0][H6] = -80;

      /* Store entry for later retrieval */
#ifdef DEBUG_PIECE_SQUARE_TABLES
      int sum[2];
      sum[0] = sum[1] = 0;
      for (square = A1; square <= H8; square++) {
         for (n=0; n<2; n++) {
            sum[n] += psq->data[n][square];
         }
      }
      sum[0] /= 64;
      sum[1] /= 64;
      printf("Bishop <PSQ>: % 3d % 3d\n", sum[0], sum[1]);
#endif
   }

   /* We now have the dynamic bishop PSQ, adjust the score */
   bb = board->bbp[BISHOP] & board->bbc[0];
   while (bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      ev += psq->data[0][square];

      get_bishop_attacks(board, square, &a, &xa);
      attack->attacks[0][BISHOP] |= a;
      attack->xattacks[0][BISHOP] |= xa;
   }
   WRITE_LOG("  Evaluate bishops (psq)                % 6d\n", ev);

   /* Score mobility, extra for centre (including x-rays?) */
   ev += popcount64(attack->attacks[0][BISHOP]) - 12;
   ev += popcount64(attack->attacks[0][BISHOP] & board_xcentre);
   ev += popcount64(attack->xattacks[0][BISHOP] & board_xcentre)/2;
   ev += popcount64(attack->attacks[0][BISHOP] & board_centre);
   WRITE_LOG("  Evaluate bishops (white)              % 6d\n", ev);

   bb = board->bbp[BISHOP] & board->bbc[1];
   while (bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      ev -= psq->data[1][square];

      /* Score mobility */
      get_bishop_attacks(board, square, &a, &xa);
      attack->attacks[1][BISHOP] |= a;
      attack->xattacks[1][BISHOP] |= xa;
   }
   /* Score mobility, extra for centre (including x-rays?) */
   ev -= popcount64(attack->attacks[1][BISHOP]) - 12;
   ev -= popcount64(attack->attacks[1][BISHOP] & board_xcentre);
   ev -= popcount64(attack->xattacks[1][BISHOP] & board_xcentre)/2;
   ev -= popcount64(attack->attacks[1][BISHOP] & board_centre);
   WRITE_LOG("  Evaluate bishops (black)              % 6d\n", ev);

   return ev;
}

/* Asses the positions of the rooks by examining the pawn structure and the
 * position of the enemy king.
 * Computes new piece-square tables and then applies these to the current
 * position.
 * FIXME: it might be slightly better to calculate a hash key based on
 * things like the number of free pawns, the location of (semi-)open files
 * and the row of the kings. These things do not change as often as the
 * location of pawns themselves, so we might be able to avoid some costly
 * recalculations this way.
 */
static inline int get_rook_table_key_king_square(const board_t *board, const int *king_square)
{
   uint64_t key;

   /* The position key depends on the pawn structure and on the rows that the
    * two kings are on.
    */
   key = board->pawn_hash;
   key ^= piece_key[0][KING][unpack_rank(king_square[0])];
   key ^= piece_key[1][KING][unpack_rank(king_square[1])];

   return key;
}

static int evaluate_rooks(gamestate_t *game, const pawn_structure_t *pawn_structure, const int *king_square, attack_t *attack)
{
   const board_t *board = game->board;
   bitboard_t bb, bp, a, xa;
   piece_square_table_t *psq;
   int square, square2;
   int ev = 0;
   uint64_t key;
   bool connected;

   /* Nothing to do if there are no rooks left */
   if (!board->bbp[ROOK])
      return 0;

   /* The position key depends on the pawn structure and on the rows that the
    * two kings are on.
    */
   key = get_rook_table_key_king_square(board, king_square);

   /* See whether we calculated PSQs for this position before; if not,
    * generate them.
    */
   psq = query_piece_table_entry(game->rook_psq, key);
   if (!psq) {
      //const int base_file[] = { 0, 2, 0, 5, 5, 0, 2, 0 };
      bitboard_t pawns[2];
      uint8_t half_open_files[2];
      uint8_t open_files;
      int row, file;
      uint8_t king_row[2];

      psq = get_free_piece_table_entry(game->rook_psq, key);
      memset(&psq->data, 0, sizeof psq->data);

      /*
       * Construct new rook PSQ
       *
       * Good squares for rooks:
       *  - on the row of the base of the enemy pawn structure
       *  - one row before the enemy king (to cut off the king)
       *  - on open or semi-open files
       *  - behind free pawns
       * Avoid:
       *  - squares in front of friendly pawns, especially on closed files
       *    (exceptions: when doing a king attack, maybe)
       *  - blocking free pawns of the same colour
       *  - rows where horizontal movement is hindered by free pawns
       */

      /* Find the rows for the enemy kings */
      king_row[0] = min(7, unpack_rank(king_square[0]) + 1);
      king_row[1] = max(0, unpack_rank(king_square[1]) - 1);
      assert(king_row[0] <= 7);
      assert(king_row[1] <= 7);

      pawns[0] = board->bbp[PAWN] & board->bbc[0];
      pawns[1] = board->bbp[PAWN] & board->bbc[1];

      /* Find the base of black's pawn structure, award bonuses for
       * attacking it.
       * If this cuts off the enemy king as well, award a greater bonus.
       * The intention here is that (base of pawn structure) + (king) is
       * worth more than either feature on their own.
       */
      if (pawns[1]) {
         square = bitscan64(pawns[1]);
         row = unpack_rank(square);
         for (file = 0; file<8; file++) {
            psq->data[0][pack_row_file(row, file)] += ROOK_PAWN_BASE_BONUS;
            if (row == king_row[1])
               psq->data[0][pack_row_file(row, file)] += ROOK_KING_CUTOFF_BONUS;
         }

         /* Bonus for trapping black's king on the back rows */
         if (row >= 4) {
            for (file = 0; file<8; file++) {
               psq->data[0][pack_row_file(king_row[1], file)] += ROOK_KING_CUTOFF_BONUS;
            }
         }
      }

      /* Ditto black */
      if (pawns[0]) {
         square = forward_bitscan64(pawns[0]);
         row = unpack_rank(square);
         for (file = 0; file<8; file++) {
            psq->data[1][pack_row_file(row, file)] += ROOK_PAWN_BASE_BONUS;
            if (row == king_row[0])
               psq->data[1][pack_row_file(row, file)] += ROOK_KING_CUTOFF_BONUS;
         }

         /* Bonus for trapping white's king on the back rows */
         if (row <= 3) {
            for (file = 0; file<8; file++) {
               psq->data[1][pack_row_file(king_row[0], file)] += ROOK_KING_CUTOFF_BONUS;
            }
         }
      }

      /* Award bonuses for open and half open files */
      open_files = pawn_structure->open_files;
      half_open_files[0] = pawn_structure->half_open_files[0];
      half_open_files[1] = pawn_structure->half_open_files[1];

      for (file = 0; file < 8; file++) {
         /* Open files */
         if ( open_files & (1<<file) ) {
            for (row = 0; row<8; row++) {
               psq->data[0][pack_row_file(row, file)] = ROOK_OPEN_FILE_BONUS;
               psq->data[1][pack_row_file(row, file)] = ROOK_OPEN_FILE_BONUS;
            }
         }

         /* Half open files (white) */
         if ( half_open_files[0] & (1<<file) ) {
            int bonus = ROOK_HALF_OPEN_BONUS;
            /* Extra bonus if there is a weak pawn on this file */
            //if (pawn_structure->weak[1] & (board_afile << file))
            //   bonus += 5;
            for (row = 0; row<8; row++) {
               square = pack_row_file(row, file);
               psq->data[0][square] += bonus;
               if (board->bbp[PAWN] & make_bitboard_square(square))
                  break;
            }

            /* Not too bad: backing your own pawn on an opponent's semi-open
             * file (black).
             */
            for (; row<8; row++) {
               square = pack_row_file(row, file);
               psq->data[1][square] += 2;
            }
         }

         /* Half open files (black) */
         if ( half_open_files[1] & (1<<file) ) {
            int bonus = ROOK_HALF_OPEN_BONUS;
            /* Extra bonus if there is a weak pawn on this file */
            //if (pawn_structure->weak[1] & (board_afile << file))
            //   bonus += 5;
            for (row = 7; row>=0; row--) {
               square = pack_row_file(row, file);
               psq->data[1][square] += bonus;
               if (board->bbp[PAWN] & make_bitboard_square(square))
                  break;
            }
         
            /* Not too bad: backing your own pawn on an opponent's semi-open
             * file (white).
             */
            for (; row>=0; row--) {
               square = pack_row_file(row, file);
               psq->data[0][square] += 2;
            }
         }
      }

      /* Bad: squares that are between the pawn spans of the two players */
      bp = pawns[0];
      bp |= bp << 8;
      bp |= bp << 16;
      bp |= bp << 32;
      bb = pawns[1];
      bb |= bb >> 8;
      bb |= bb >> 16;
      bb |= bb >> 32;
      bb &= bp;
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[0][square] = -ROOK_INTERSPAN_PENALTY;
         psq->data[1][square] = -ROOK_INTERSPAN_PENALTY;
      }

      /* Bad: squares on the same row as a blocked pawn of the same colour */
      bb = pawns[0] & (pawns[1]>>8);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         row = unpack_rank(square);
         for(file = 0; file<8; file++) {
            psq->data[0][pack_row_file(row, file)] -= 5;
         }
      }

      /* Bad: squares on the same row as a blocked pawn of the same colour */
      bb = pawns[1] & (pawns[0]<<8);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         row = unpack_rank(square);
         for(file = 0; file<8; file++) {
            psq->data[1][pack_row_file(row, file)] -= 5;
         }
      }

      /* Bad: standing in front of a passed (or free) pawn of the same colour
       * Calculate the forward span of the free pawns, then mark squares in
       * this span as "bad".
       * Progressively so when we're directly blocking the pawn, but the
       * case where we're on the stop square itself is already taken care
       * of in the main evaluation.
       */
      bb = (pawn_structure->free & board->bbc[0])<<16;
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[0][square] -= STOP_SQUARE_PENALTY;
      }
      bb = (pawn_structure->free & board->bbc[0])<<24;
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[0][square] -= STOP_SQUARE_PENALTY/2;
      }

      bb = (pawn_structure->free & board->bbc[1])>>16;
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[1][square] -= STOP_SQUARE_PENALTY;
      }

      bb = (pawn_structure->free & board->bbc[1])>>24;
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[1][square] -= STOP_SQUARE_PENALTY/2;
      }

      /* Good: being behind a passed pawn */
      /* White */
      bb = (pawn_structure->free & board->bbc[0]);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         file = unpack_file(square);
         row = unpack_rank(square)-1;
         for (; row>=0; row--) {
            square = pack_row_file(row, file);
            psq->data[0][square] += ROOK_BACKUP_FREE_PAWN_BONUS;
            psq->data[1][square] += ROOK_BACKUP_FREE_PAWN_BONUS;
         }
      }

      /* Black */
      bb = (pawn_structure->free & board->bbc[1]);
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         file = unpack_file(square);
         row = unpack_rank(square)+1;
         for (; row<8; row++) {
            square = pack_row_file(row, file);
            psq->data[0][square] += ROOK_BACKUP_FREE_PAWN_BONUS;
            psq->data[1][square] += ROOK_BACKUP_FREE_PAWN_BONUS;
         }
      }

      /* Bad: being in the cul between connected pawns, that is, being
       * directly behind a defended pawn - especially on the side of the
       * board.
       */
      bb = board->bbp[PAWN] & board->bbc[0];
      bb &= (bb & board_minus_a) << 7 | (bb & board_minus_h) << 9;
      bb &= board_afile | board_hfile;
      bb >>= 8;
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[0][square] -= 7;
      }

      bb = board->bbp[PAWN] & board->bbc[1];
      bb &= (bb & board_minus_a) >> 9 | (bb & board_minus_h) >> 7;
      bb &= board_afile | board_hfile;
      bb <<= 8;
      while (bb) {
         square = bitscan64(bb);
         bb ^= make_bitboard_square(square);
         psq->data[1][square] -= 7;
      }

      /* Store entry for later retrieval */
#ifdef DEBUG_PIECE_SQUARE_TABLES
      int sum[2];
      sum[0] = sum[1] = 0;
      for (square = A1; square <= H8; square++) {
         int n;
         for (n=0; n<2; n++) {
            sum[n] += psq->data[n][square];
         }
      }
      sum[0] /= 64;
      sum[1] /= 64;
      printf("Rook <PSQ>:   % 3d % 3d\n", sum[0], sum[1]);
#endif
   }

   /* We now have the dynamic rook PSQ, adjust the score */
   bb = board->bbp[ROOK] & board->bbc[0];
   square2 = square = 0xff;
   connected = false;
   while (bb) {
      square2 = square;
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      ev += psq->data[0][square];

      /* Score mobility */
      get_rook_attacks(board, square, &a, &xa);
      attack->attacks[0][ROOK] |= a;
      attack->xattacks[0][ROOK] |= xa;
      ev += (popcount64(a) - 7) * ROOK_MOBILITY_WEIGHT / 64;
      connected = connected || (a & bb);
   }

   /* Extra bonus: connected rooks */
   if (connected && square2 != 0xff) {
      if (unpack_file(square) == unpack_file(square2))
         ev += ROOK_CONNECTED_FILE_BONUS;
      else if (unpack_rank(square) == unpack_rank(square2))
         ev += ROOK_CONNECTED_RANK_BONUS;
   }

   bb = board->bbp[ROOK] & board->bbc[1];
   square2 = square = 0xff;
   connected = false;
   while (bb) {
      square2 = square;
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      ev -= psq->data[1][square];

      /* Score mobility */
      get_rook_attacks(board, square, &a, &xa);
      attack->attacks[1][ROOK] |= a;
      attack->xattacks[1][ROOK] |= xa;
      ev -= (popcount64(a) - 7) * ROOK_MOBILITY_WEIGHT / 64;
      connected = connected || (a & bb);
   }

   /* Extra bonus: connected rooks */
   if (connected && square2 != 0xff) {
      if (unpack_file(square) == unpack_file(square2))
         ev -= ROOK_CONNECTED_FILE_BONUS;
      else if (unpack_rank(square) == unpack_rank(square2))
         ev -= ROOK_CONNECTED_RANK_BONUS;
   }

   return ev;
}

/* Asses the positions of the queens. 
 * Here the only consideration is that the queen should be close to the
 * enemy king.
 */
int evaluate_queens(gamestate_t *game, const pawn_structure_t *pawn_structure, const int *king_square, attack_t *attack)
{
   const board_t *board = game->board;
   bitboard_t bb, unsafe, a, xa;
   int n_moves, n_unsafe;
   int square;
   int ev = 0;

   /* Nothing to do if there are no queens left */
   if (!board->bbp[QUEEN])
      return 0;

   bb = board->bbp[QUEEN] & board->bbc[0];
   while (bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      get_queen_attacks(board, square, &a, &xa);
      attack->attacks[0][QUEEN] |= a;
      attack->xattacks[0][QUEEN] |= xa;

      ev += 2*popcount64(a & ~(attack->attacks[1][PAWN] | board->bbc[0]));
   }

   bb = board->bbp[QUEEN] & board->bbc[1];
   while (bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      get_queen_attacks(board, square, &a, &xa);
      attack->attacks[1][QUEEN] |= a;
      attack->xattacks[1][QUEEN] |= xa;

      ev -= 2*popcount64(a & ~(attack->attacks[0][PAWN] | board->bbc[1]));
   }

   return ev;

   /* Queen mobility: penalise a queen for whom many possible destination
    * squares are under enemy control.
    * Find out how many of the queen's normal attack squares have been
    * blocked by enemy attack; if many of them, then the queen gets a
    * severe penalty.
    */
   if (board->bbp[QUEEN] & board->bbc[0]) {
      int index;

      index = 0;
      /* Squares under enemy control (by pieces less valuable than the queen) */
      unsafe   = attack->attacks[1][PAWN] | attack->attacks[1][KNIGHT] |
                 attack->attacks[1][BISHOP] | attack->attacks[1][ROOK];

      n_moves  = popcount64(attack->attacks[0][QUEEN]);
      n_unsafe = popcount64(attack->attacks[0][QUEEN] & unsafe);
      
      /* Award a small penality initially if the queen cannot withdraw to
       * its own side of the board.
       */
      if (!(attack->attacks[0][QUEEN] & board_white))
         index += 2;

      if (n_unsafe > n_moves/2) {
         index += 2*(n_unsafe - n_moves/2);

         if (n_moves < 8)
            index += 8-n_moves;
      }

      ev -= 4*index*index;
   }

   if (board->bbp[QUEEN] & board->bbc[1]) {
      int index;

      index = 0;
      /* Squares under enemy control (by pieces less valuable than the queen) */
      unsafe   = attack->attacks[0][PAWN] | attack->attacks[0][KNIGHT] |
                 attack->attacks[0][BISHOP] | attack->attacks[0][ROOK];

      n_moves  = popcount64(attack->attacks[1][QUEEN]);
      n_unsafe = popcount64(attack->attacks[1][QUEEN] & unsafe);
      
      /* Award a small penality initially if the queen cannot withdraw to
       * its own side of the board.
       */
      if (!(attack->attacks[1][QUEEN] & board_black))
         index += 2;

      if (n_unsafe > n_moves/2) {
         index += 2*(n_unsafe - n_moves/2);

         if (n_moves < 8)
            index += 8-n_moves;
      }

      ev += 4*index*index;
   }

   return ev;

#if 0
   empty = ~(board->bbc[0] | board->bbc[1]);

   /* White */
   /* Squares that the queen can move to */
   bb = attack->attacks[0][QUEEN] & empty;

   /* Squares under enemy control (by pieces less valuable than the queen) */
   blocked = attack->attacks[1][PAWN] | attack->attacks[1][KNIGHT] |
               attack->attacks[1][BISHOP] | attack->attacks[1][ROOK];
   blocked &= bb;
   
   n_moves = popcount64(bb);
   n_blocked = popcount64(blocked);
   if (n_blocked > n_moves/2) ev -= n_blocked*4;

   /* Black */
   /* Squares that the queen can move to */
   bb = attack->attacks[1][QUEEN] & empty;

   /* Squares under enemy control (by pieces less valuable than the queen) */
   blocked = attack->attacks[0][PAWN] | attack->attacks[0][KNIGHT] |
               attack->attacks[0][BISHOP] | attack->attacks[0][ROOK];
   blocked &= bb;
   
   n_moves = popcount64(bb);
   n_blocked = popcount64(blocked);
   if (n_blocked > n_moves/2) ev += n_blocked*4;


   return ev;

   /* There is no queen psq, but in general a queen works well with rooks,
    * so if there are still rooks present, then we award the queen the same
    * bonuses that we would a rook.
    */

   /* The position key depends on the pawn structure and on the rows that the
    * two kings are on.
    */
   //key = get_rook_table_key_king_square(board, king_square);

   /* See whether we calculated PSQs for this position before; if not,
    * generate them.
    */
   //psq = query_piece_table_entry(game->rook_psq, key);

   bb = board->bbp[QUEEN] & board->bbc[0];
   while (bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      ev += 3*(8 - king_distance(square, king_square[1]))/2;
      //if (psq && psq->data[0][square]>0) ev += psq->data[0][square];
   }

   bb = board->bbp[ROOK] & board->bbc[1];
   while (bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      ev -= 3*(8 - king_distance(square, king_square[0]))/2;
      //if (psq && psq->data[1][square]>0) ev -= psq->data[1][square];
   }

   return ev;
#endif
}

/* Evaluate passed pawns. */
int evaluate_passed_pawns(gamestate_t *game, const pawn_structure_t *pawn_structure, const int *king_square, int side_to_move, const int *majors, const int *minors, attack_t *attack)
{
   static const bitboard_t back_rank[2] = {
      board_rank6|board_rank7|board_rank8, 
      board_rank1|board_rank2|board_rank3
   };
   board_t *board = game->board;
   bitboard_t passed[2];
   bitboard_t attacked[2];
   bitboard_t bb, bp;
   bitboard_t king_move, occ, square, pawns;
   int passed_pawns_mg = 0;
   int passed_pawns_eg = 0;
   int ev = 0;
   int n;

   passed[0] = board->bbc[0] & pawn_structure->free;
   passed[1] = board->bbc[1] & pawn_structure->free;

   attacked[0] = attack->attacks[0][QUEEN] | attack->attacks[0][ROOK] | attack->attacks[0][BISHOP] | attack->attacks[0][KNIGHT];
   attacked[1] = attack->attacks[1][QUEEN] | attack->attacks[1][ROOK] | attack->attacks[1][BISHOP] | attack->attacks[1][KNIGHT];
   occ = board->bbc[0] | board->bbc[1];


   /* Passed pawns */
   n =  popcount64(passed[0]);
   passed_pawns_mg += n*FREE_PAWN_BONUS_MG;
   passed_pawns_eg += n*FREE_PAWN_BONUS_EG;
   n =  popcount64(passed[1]);
   passed_pawns_mg -= n*FREE_PAWN_BONUS_MG;
   passed_pawns_eg -= n*FREE_PAWN_BONUS_EG;

   /* Passed pawns that are close to promotion are worth more */
   /* FIXME: passed pawns become more valuable in the END GAME */
   bb = passed[0];
   while (bb) {
      int p = bitscan64(bb);
      int dist;

      unset_bitboard(&bb, p);
      dist = unpack_rank(p);

      /* FIXME: should correct for the other king being in the "square"
       * (ie, having a shorter Manhatten-distance to the promotion square.
       */
      passed_pawns_mg += dist*FREE_PAWN_ADVANCE_BONUS_MG;
      passed_pawns_eg += (dist + (dist == 6)) * FREE_PAWN_ADVANCE_BONUS_EG;
   }

   bb = passed[1];
   while (bb) {
      int p = bitscan64(bb);
      int dist;

      unset_bitboard(&bb, p);
      dist = 7 - unpack_rank(p);

      /* FIXME: should correct for the other king being in the "square"
       * (ie, having a shorter Manhatten-distance to the promotion square.
       */
      passed_pawns_mg -= dist*FREE_PAWN_ADVANCE_BONUS_MG;
      passed_pawns_eg -= (dist + (dist == 6)) * FREE_PAWN_ADVANCE_BONUS_EG;
   }

   /* The evaluation of passed pawns depends on the game phase:
    *    14    - opening/early middle game, no bonus
    *   >10    - middle game, small bonus
    *    10-6  - transition middle to end game, strength increases
    *   <6     - end game
    */
   int phase = minors[0]+minors[1] + majors[1]+majors[0];
   if (phase < 14) {
      ev += passed_pawns_mg;
      WRITE_LOG("Score after passed pawns (phase 1)      % 6d\n", ev);
      if (phase < 10) {
         if (phase < 6)
            ev += passed_pawns_eg;
         else
            ev += passed_pawns_eg * (10 - phase) / 4;
      }
   }
   WRITE_LOG("Score after passed pawns                % 6d\n", ev);

   /* Passed duo's
    * These will defeat a rook - but not necessarily if the rook aided by
    * other pieces!
    * If we've evaluated these, remove them from the list of passed pawns
    * under consideration so we don't award the same bonus twice.
    */
   if (!onebit64(pawn_structure->connected_free & board->bbc[0] & back_rank[0]) ) {
      if (evaluate_pawn_duo(board, pawn_structure, side_to_move, 0, &ev))
         passed[0] &= ~(pawn_structure->connected_free & board->bbc[0] & back_rank[0]);
   }
   if (!onebit64(pawn_structure->connected_free & board->bbc[1] & back_rank[1])) {
      if (evaluate_pawn_duo(board, pawn_structure, side_to_move, 1, &ev))
         passed[1] &= ~(pawn_structure->connected_free & board->bbc[1] & back_rank[1]);
   }

   /* Passed pawns - white */
   while (passed[0]) {
      bool enemy_pieces = (board->bbc[1] & get_normal_pieces(board)) != board_empty;
      int pawn_square = bitscan64(passed[0]);
      bp = make_bitboard_square(pawn_square);
      passed[0] ^= bp;

      int index = 0;

      /* Does the pawn have a free path? */
      bb = fill_north(bp) & ~bp;
      if ( (bb & (attacked[1] | board->bbc[0])) == 0) index+=2;

      /* Is the stop square free? */
      if ( ((bp << 8) & occ) == 0) index++;

      /* Does the pawn have a free path while it is on the 7th rank? */
      if ( (bp & board_rank7) && (bb & (attacked[1] | board->bbc[0])) == 0) index++;

      /* Does the pawn have the support of its own pieces? */
      if ( bp & attacked[0] ) index++;

      /* Does the pawn have the support of its own pawns? */
      pawns = ( (bp << 1 | bp >> 9) & ~board_hfile ) | ( (bp >> 1 | bp >> 7) & ~board_afile );
      if (pawns & board->bbc[0] & board->bbp[PAWN]) index++;

      /* Is the pawn supported by a rook/queen from behind? */
      if (bb & board_file[unpack_file(pawn_square)] & attack->xattacks[0][ROOK]) index++;

      /* Is the enemy king inside the square while the own king is outside
       * the square?
       */
      square  = white_pawn_square[pawn_square];
      if (side_to_move == 1)
         king_move = king_attack[king_square[1]];
      else
         king_move = make_bitboard_square(king_square[1]);
      if ( enemy_pieces == 0 && (square & king_move) ) index /= 2;
      /* FIXME: what is the following term supposed to do? Should it
       * trigger in the middle game?
       */
      if ( !(square & king_move) || (square & make_bitboard_square(king_square[0])) ) index+=2;

      /* The more advanced the pawn is, the larger its value */
      if (unpack_rank(pawn_square) > 3) index += (unpack_rank(pawn_square) - 3);

      ev += passed_pawn_bonus[min(index, 16)];
   }


   /* Passed pawns - black */
   while (passed[1]) {
      bool enemy_pieces = (board->bbc[0] & get_normal_pieces(board)) != board_empty;
      int pawn_square = bitscan64(passed[1]);
      bp = make_bitboard_square(pawn_square);
      passed[1] ^= bp;

      int index = 0;

      /* Does the pawn have a free path? */
      bb = fill_south(bp) & ~bp;
      if ( (bb & (attacked[0] | board->bbc[1])) == 0) index+=2;

      /* Is the stop square free? */
      if ( ((bp >> 8) & occ) == 0) index++;

      /* Does the pawn have a free path while it is on the 7th rank? */
      if ( (bp & board_rank2) && (bb & (attacked[0] | board->bbc[1])) == 0) index++;

      /* Does the pawn have the support of its own pieces? */
      if ( bp & attacked[1] ) index++;

      /* Does the pawn have the support of its own pawns? */
      pawns = ( (bp << 1 | bp << 9) & ~board_hfile ) | ( (bp >> 1 | bp << 7) & ~board_afile );
      if (pawns & board->bbc[1] & board->bbp[KNIGHT]) index++;

      /* Is the pawn supported by a rook/queen from behind? */
      if (bb & board_file[unpack_file(pawn_square)] & attack->xattacks[1][ROOK]) index++;

      /* Is the enemy king inside the square while the own king is outside
       * the square?
       */
      square  = black_pawn_square[pawn_square];
      if (side_to_move == 0)
         king_move = king_attack[king_square[0]];
      else
         king_move = make_bitboard_square(king_square[0]);
      if ( enemy_pieces == 0 && (square & king_move) ) index /= 2;
      /* FIXME: what is the following term supposed to do? Should it
       * trigger in the middle game?
       */
      if ( !(square & king_move) || (square & make_bitboard_square(king_square[1])) ) index+=2;

      /* The more advanced the pawn is, the larger its value */
      if (unpack_rank(pawn_square) < 4) index += (4 - unpack_rank(pawn_square));

      ev -= passed_pawn_bonus[min(index, 16)];
   }

   return ev;
}

/* Load the generated dynamic piece-square tables into the game struct,
 * where they can be used for move ordering.
 * TODO: update dynamic PSQ at every iteration when possible. The simplest
 * way would be to store the key sigantures used to generate the table and
 * test whether the signature is different from the last one used.
 */
void load_dynamic_psq(gamestate_t *game)
{
#ifdef USE_DYNAMIC_PSQ_FOR_MOVE_ORDER
   const board_t *board = game->board;
   pawn_structure_t *pawn_structure;
   attack_t attack;
   piece_square_table_t *psq;
   int king_square[2];
   uint64_t key;
   int c, square;

   memset(game->dynamic_psq, 0, sizeof game->dynamic_psq);

   pawn_structure = evaluate_pawn_structure(game);

   king_square[0] = bitscan64(board->bbp[KING] & board->bbc[0]);
   king_square[1] = bitscan64(board->bbp[KING] & board->bbc[1]);

   /* Asses the position of the different pieces */
   evaluate_rooks(game, pawn_structure, king_square, &attack);
   evaluate_knights(game, pawn_structure, king_square, &attack);
   evaluate_bishops(game, pawn_structure, king_square, &attack);

   /* Load the different piece square tables and copy */
   /* Rooks */
   key = get_rook_table_key_king_square(board, king_square);
   psq = query_piece_table_entry(game->rook_psq, key);

   if (psq) {
      for (c=0; c<2; c++) {
         for (square = 0; square<64; square++) {
            game->dynamic_psq[3][c][square] = psq->data[c][square];
         }
      }
   }

   /* Bishops */
   key = get_bishop_table_key(board);
   psq = query_piece_table_entry(game->bishop_psq, key);

   if (psq) {
      for (c=0; c<2; c++) {
         for (square = 0; square<64; square++) {
            game->dynamic_psq[2][c][square] = psq->data[c][square];
         }
      }
   }

   /* Knights */
   key = get_knight_table_key(board);
   psq = query_piece_table_entry(game->knight_psq, key);

   if (psq) {
      for (c=0; c<2; c++) {
         for (square = 0; square<64; square++) {
            game->dynamic_psq[1][c][square] = psq->data[c][square];
         }
      }
   }
#endif
}

#ifdef ALLOW_LAZY_EVALUATION
static bool lazy_evaluation_safe(const board_t *board, int side_to_move)
{
   /* If there are advanced passed pawns (and they're not blocked) then we
    * don't allow lazy evaluation (not safe).
    */
   bitboard_t passed = get_passed_pawns(board);
   if (passed) {
      bitboard_t bb;

      /* White */
      bb = (passed & board->bbc[0]) & (board_rank5 | board_rank6 | board_rank7);
      if (bb) return false;

      /* Black */
      bb = (passed & board->bbc[1]) & (board_rank4 | board_rank3 | board_rank2);
      if (bb) return false;
   }

   /* King safety issues - no lazy eval */

   return true;
}
#endif

/* Evaluate the current state of the game from the point of view of the
 * player to move.
 * Returns a positive score if the player to move is ahead or a negative
 * score if he is behind. Returns 0 iff the game is a draw.
 */
int static_evaluation(gamestate_t *game, int side_to_move, const int alpha, const int beta)
{
   static const int last_rank[2] = {7, 0};
   bitboard_t stop_squares[2];
   eval_info_t ei;
   const board_t *board = game->board;
   pawn_structure_t *pawn_structure;
   int majors[2], minors[2], pawns[2], pieces[2];
   int16_t material[2] = { 0, 0 };
   int piece_types[2] = { 0, 0 };
   int num_pieces[2] = { 0, 0 };
   int mate_potential[2] = { 0, 0 };
   int army_strength[2] = { 8, 8 };    /* Initialise with king present */
   bool can_win[2] = { true, true };
   int minor_idx, major_idx;
   int mat = 0;
   int psq = 0;
   int ev = 0;
   int sign;
   int king_square[2];
#ifdef ALLOW_LAZY_EVALUATION
   int lazy_ev;
   bool allow_lazy_evaluation = true;
#if !defined ALLOW_LAZY_EVALUATION_IN_PV
   const bool pv_node = (beta - alpha == 1);
#endif
#endif
   bool have_eval_hash;
   int eh_score;

#if !defined ALLOW_LAZY_EVALUATION_IN_PV && ALLOW_LAZY_EVALUATION
   allow_lazy_evaluation = !pv_node;
#endif

   have_eval_hash = query_eval_table_entry(game->eval_hash, game->board->hash, &eh_score);
#ifndef DEBUG_EVHASH
   if (have_eval_hash) return eh_score;
#endif

   memset(&ei.attack, 0, sizeof ei.attack);

   /* Assess whether it's safe to do lazy evaluation at this node.
    * Exceptions: presence of advanced passed pawns, exposed king.
    * In principle we could handle this with a wider margin as well, but
    * this way we can use narrower margins in positions where it's safe to
    * do so.
    */
#ifdef ALLOW_LAZY_EVALUATION
   allow_lazy_evaluation = allow_lazy_evaluation && lazy_evaluation_safe(board, side_to_move);
#endif

   /* Prefetch pawn hash table entry (if it exists), so we can access it
    * more quickly below.
    */
   prefetch_pawntable(game->pawn_structure, board->pawn_hash);

   /* Convert to 1-based counting for black and white */
   side_to_move >>= 7;
   sign = 1 - 2*side_to_move;

   /* Count material */
   for (int side=0; side<2; side++) {
      pieces[side] = 0;
      piece_types[side] = 1;
      for (int n=PAWN; n<KING; n++) {  /* We don't need to count the king */
         ei.piece_count[side][n] = sparse_popcount64(board->bbc[side]&board->bbp[n]);
         pieces[side] += ei.piece_count[side][n];
         piece_types[side] += (ei.piece_count[side][n]!=0);
         if (n) material[side] += chess_piece_values[n]*ei.piece_count[side][n];
         army_strength[side] += piece_attack_weight[n] * ei.piece_count[side][n];
      }
      pawns[side] = ei.piece_count[side][PAWN];
      material[side] += pawn_count_value[pawns[side]];

      /* Total number of playing pieces, including kings and pawns */
      num_pieces[side] = pieces[side] + 1;

      /* "Officers", excludining kings and pawns */
      pieces[side] -= pawns[side];

      /* Number of minors: knights + bishops */
      minors[side] = ei.piece_count[side][KNIGHT] + ei.piece_count[side][BISHOP];

      /* Number of majors: rooks + 2*qeens (1 queen == 2 rooks approximately) */
      majors[side] = ei.piece_count[side][ROOK] + 2*ei.piece_count[side][QUEEN];

      /* Assess mate potential: number of rooks, queens, bishop pairs,
       * bishop+knight. As a refinement could include number of passed
       * pawns that are not blocked.
       * Assumes that bishops are on different colours, which is normally
       * correct; could add an evaluation term that reduces the value of
       * the second bishop if it is on the same colour, like a negative
       * pair bonus, which is probably good enough.
       */
      mate_potential[side] = ei.piece_count[side][ROOK] +
                             ei.piece_count[side][QUEEN]*2 +
                             ei.piece_count[side][BISHOP]/2 +
                             (ei.piece_count[side][BISHOP] + ei.piece_count[side][KNIGHT])/2;

      /* Lack of mate potential - side cannot win without pawns */
      if (pawns[side] == 0 && !mate_potential[side])
         can_win[side] = false;
   }

   /* Recognise position of rook vs. minor+(passed) pawn, these are dead
    * draws if the rook side has no pawns at all.
    * In other cases the score should be scaled (TODO)
    */
   for (int side=0; side<2; side++) {
      int other = side^1;
      if (pieces[side] == 1 && ei.piece_count[side][ROOK] && pawns[side] == 0 && pieces[other])
         can_win[side] = false;
   }

   /* TODO: keep track of mating potential, so we can (correctly) evaluate
    * end games like KRNKR (and pawn-less endings in general) as draw-ish.
    * Mate potential is: a queen, a rook, a bishop pair, a bishop+knight, a
    * passed pawn that can promote without issue.
    *
    * TODO: scale the value of a passed pawn with the strongest piece that
    * can stop it.
    */

   king_square[0] = bitscan64(board->bbp[KING] & board->bbc[0]);
   king_square[1] = bitscan64(board->bbp[KING] & board->bbc[1]);

   /* Check for lone king endings. Our function may or may not handle these
    * in detail, so we can't return the score without checking for this.
    */
   if (onebit64(board->bbc[0]) || onebit64(board->bbc[1])) {
      bool handled;
      handled = evaluate_lone_king(game, side_to_move, minors, majors, pawns, material, &ev);
      if (handled)
         goto scale_fifty;
   }

   /* Assess material balance */
   mat = material[0] - material[1];
   WRITE_LOG("Score after material balance:           % 6d\n", mat);

   /* Award a bonus for having the bishop pair */
   bitboard_t bishops = board->bbc[0] & board->bbp[BISHOP];
   if (bishops && !onebit64(bishops)) mat += EVAL_BISHOP_PAIR;
   bishops = board->bbc[1] & board->bbp[BISHOP];
   if (bishops && !onebit64(bishops)) mat -= EVAL_BISHOP_PAIR;
   WRITE_LOG("Score after bishop pair                 % 6d\n", mat);

   /* Material imbalances: the number of majors and minors does not match
    * between sides. Pawns may make up the difference, but in general pawns
    * for pieces is not worth it. This code was adapted from Crafty
    */
   minor_idx = minors[0] - minors[1];
   major_idx = majors[0] - majors[1];

   mat += imbalance[max(min(4+major_idx, 8), 0)][max(min(4+minor_idx, 8), 0)];

   WRITE_LOG("Score after material imbalance:         % 6d\n", mat);

   /* Drawish material combinations (pawns make all the difference here and
    * are required to win):
    *  - rook vs minor one or two minors
    *  - rook + minor vs rook or minor
    *  - same with an extra rook on either side (less drawish, however)
    * The idea here is that the defending piece must be won, but cannot be
    * traded by sacrificing a rook. In other words, the player must be
    * careful to maintain mating potential.
    */


   /* Give a minute bonus to the side to move. The idea behind this is to
    * break the degeneracy between positions that are exactly balanced, and
    * those that are legitimate draws.
    * This value should be small enough that it doesn't affect the
    * evaluation too much.
    */
   //ev += SIDE_TO_MOVE_BONUS;

   /* Exchange bonus.
    * If ahead in material, award a penalty for having pieces on the
    * board, award a bonus for having pawns on the board.
    * This is to encourage piece exchanges when ahead and pawn exchanges
    * when behind.
    * Exception: if there are only rooks left, it's better to have two than
    * to have one.
    */
   if ((pieces[0] + pieces[1]) < 6) {
      if ( board->bbp[KNIGHT]|board->bbp[BISHOP]|board->bbp[QUEEN] ) {
         if (material[0] > material[1]) {
            mat -= (pieces[0] + pieces[1])*2;
            mat += (pawns [0] + pawns [1])*4;
            //mat -= pieces[1]*4;
            //mat += pawns [0]*12;
         } else if (material[0] < material[1]) {
            mat += (pieces[0] + pieces[1])*2;
            mat -= (pawns [0] + pawns [1])*4;
            //mat += pieces[0]*4;
            //mat -= pawns [1]*12;
         }
      } else if ( board->bbp[ROOK] ) {
         /* We still don't want to exchange pawns if we're ahead */
         if (material[0] > material[1]) {
            mat += (pawns[0] + pawns[1])*4;
         } else if (material[0] < material[1]) {
            mat -= (pawns[0] + pawns[1])*4;
         }
         /* Draw score slightly closer to draw if there is only one rook on
          * each side.
          */
         if ( (majors[0]+majors[1]) == 2 ) {
            mat -= mat/4;
         }
      }
   }
   WRITE_LOG("Score after trade-down bonus            % 6d\n", mat);

   /* Adjust the material score: rooks become stronger as pawns disappear,
    * knights become a bit weaker.
    * FIXME: does this just reflect increased rook mobility?
    * In a sense we're already doing this by awarding rook bonuses on open
    * files: when there are few pawns, most files are open.
    */
   mat += ei.piece_count[0][ROOK] * (5-pawns[0]) * ROOK_VALUE_SCALE;
   mat -= ei.piece_count[1][ROOK] * (5-pawns[1]) * ROOK_VALUE_SCALE;
   WRITE_LOG("Score after rook score adjustment       % 6d\n", mat);

   /* TODO: only apply the knight bonus for the first knight. Kauffman says
    * knight pairs should be penalised, which in itself seems odd to me. I
    * think it makes more sense to apply the bonus only to the first
    * knight.
    */
   mat -= ( ei.piece_count[0][KNIGHT] != 0) * (5-pawns[0]) * KNIGHT_VALUE_SCALE;
   mat += ( ei.piece_count[1][KNIGHT] != 0) * (5-pawns[1]) * KNIGHT_VALUE_SCALE;
   WRITE_LOG("Score after knight score adjustment     % 6d\n", mat);

   /* Bad material combination: two knights in the end game */
   if (mat > 0 && majors[0] == 0 && ei.piece_count[0][KNIGHT] == 2 && minors[0] == 2 && pawns[0] == 0) mat /= 8;
   if (mat < 0 && majors[1] == 0 && ei.piece_count[1][KNIGHT] == 2 && minors[1] == 2 && pawns[1] == 0) mat /= 8;

   /* No pawns, advantage of a single minor is meaningless */
   if (!board->bbp[PAWN]) {
      if (majors[0] == majors[1] && abs(minors[0]-minors[1]) == 1) {
         if (majors[0] <= 1)
            mat = 0;
         else
            mat /= 16;
      }
   }

   ev += mat;

   /* Lazy evaluation: bail out if we're already well-outside the
    * alpha-beta window and can't hope to make up with the rest of the
    * evaluation.
    */
#ifdef ALLOW_LAZY_EVALUATION
   lazy_ev = ev * sign;
   if ( allow_lazy_evaluation && ((lazy_ev+LAZY_MARGIN2) < alpha || (lazy_ev-LAZY_MARGIN2) > beta) )
      return lazy_ev;
#endif

   /* Evaluate castling status: give a penalty for the current position if
    * castling rights have been lost since the root of the search, but the
    * player has not castled.
    * This should really be based on the king safety evaluation after
    * castling.
    */
   if (!board->did_castle[0] && may_castle(game->root_board, 0) && !may_castle(board, 0)) {
      ev -= CASTLE_LOST_PENALTY;
   }

   if (!board->did_castle[1] && may_castle(game->root_board, 1) && !may_castle(board, 1)) {
      ev += CASTLE_LOST_PENALTY;
   }

   /* A slightly smaller penalty if the player can castle, but hasn't */
   if (!board->did_castle[0] && may_castle(board, 0)) {
      ev -= CASTLE_LOST_PENALTY/2;
   }

   if (!board->did_castle[1] && may_castle(board, 1)) {
      ev += CASTLE_LOST_PENALTY/2;
   }

   WRITE_LOG("Score after castle status check         % 6d\n", ev);

   /*
    * Evaluate pawn structure and piece scores that depend on it
    */
   pawn_structure = evaluate_pawn_structure(game);
   assert(pawn_structure);

   ev += pawn_structure->score;
   WRITE_LOG("Score after pawn structure              % 6d\n", ev);

   /* Evaluate some special pawn formations that also depend on the
    * presence of other pieces: double pawns, duos.
    */

   /* Doubled pawns.
    * These are normally bad, but the value depends on the presence of
    * major pieces (Kaufman "All About Doubled Pawns" 2005, Chess Life)
    */
   major_idx = min(4, majors[0] + majors[1]);
   ev -= popcount64(pawn_structure->doubled_pawns & board->bbc[0]) * double_pawn_penalty[major_idx];
   ev += popcount64(pawn_structure->doubled_pawns & board->bbc[1]) * double_pawn_penalty[major_idx];
   WRITE_LOG("Score after doubled pawns               % 6d\n", ev);

   /* Bail out if we're already well-outside the alpha-beta window and
    * can't hope to make up with the rest of the evaluation
    */
#ifdef ALLOW_LAZY_EVALUATION
   lazy_ev = ev * sign;
   if ( allow_lazy_evaluation && ((lazy_ev+LAZY_MARGIN1) < alpha || (lazy_ev-LAZY_MARGIN1) > beta) )
      return lazy_ev;
#endif

   /* Some general piece placement evaluations that are independent of the
    * detailed piece-square tables below, although these may modify the
    * results from the evaluation here.
    */

   /* Evaluate stop squares */
   stop_squares[0] = (pawn_structure->free & board->bbc[1]) >> 8;
   stop_squares[1] = (pawn_structure->free & board->bbc[0]) << 8;

   /* Pieces on stop square: bonus for own piece on stop square, penalty
    * for own piece on enemy stop square (a piece blocking a free pawn of
    * the same colour)
    * FIXME: this could be handled in the PSQs...
    * FIXME: the bonus for a piece on a stop square is larger than the
    * bonus of a free pawn, so it's good if the opponent has free pawns and
    * we get to stop them...?! Conversely, it's bad to exchange a free pawn
    * for a non-free pawn if we can stop it instead?! It's good to exchange
    * a free pawn for a non-free pawn if the opponent is blocking it?!
    */
   ev += sparse_popcount64(stop_squares[0] & board->bbc[0]) * STOP_SQUARE_BONUS;
   ev -= sparse_popcount64(stop_squares[1] & board->bbc[0]) * STOP_SQUARE_PENALTY;
   ev -= sparse_popcount64(stop_squares[1] & board->bbc[1]) * STOP_SQUARE_BONUS;
   ev += sparse_popcount64(stop_squares[0] & board->bbc[1]) * STOP_SQUARE_PENALTY;
   WRITE_LOG("Score after stop square bonus           % 6d\n", ev);

   /* Outposts and weak/strong squares.
    * For our purposes, we mean the following:
    * A weak square is a square in front of a backward pawn.
    * One player's weak square is the other player's strong square.
    * An outpost is a strong square defended by an own pawn
    * Note that an outpost is also a strong square.
    * FIXME: this could be handled in the PSQs...
    */
   ev += sparse_popcount64(pawn_structure->strong_squares[0]&board->bbc[0]) * STRONG_SQUARE_BONUS;
   ev -= sparse_popcount64(pawn_structure->strong_squares[1]&board->bbc[1]) * STRONG_SQUARE_BONUS;
   WRITE_LOG("Score after strong squares              % 6d\n", ev);

   /* Asses the position of the different pieces.
    * Build up attack tables to be used for king safety evaluation.
    * TODO: factor in game phase as well as pawn structure
    */
   get_pawn_attacks(game->board, &ei.attack.attacks[0][PAWN], &ei.attack.attacks[1][PAWN]);

   psq += evaluate_rooks(game, pawn_structure, king_square, &ei.attack);
   WRITE_LOG("Score after rook evaluation             % 6d\n", ev + psq);

   psq += evaluate_knights(game, pawn_structure, king_square, &ei.attack);
   WRITE_LOG("Score after knight evaluation           % 6d\n", ev + psq);

   psq += evaluate_bishops(game, pawn_structure, king_square, &ei.attack);
   WRITE_LOG("Score after bishop evaluation           % 6d\n", ev + psq);

   psq += evaluate_queens(game, pawn_structure, king_square, &ei.attack);
   WRITE_LOG("Score after queen evaluation            % 6d\n", ev + psq);

   psq += evaluate_passed_pawns(game, pawn_structure, king_square, side_to_move, majors, minors, &ei.attack);
   WRITE_LOG("Score after passed pawn evaluation      % 6d\n", ev + psq);

   /* King safety */
   psq += evaluate_king(game, pawn_structure, king_square, majors, minors, &ei);
   WRITE_LOG("Score after king evaluation             % 6d\n", ev + psq);

   ev += psq;

   for (int side=0; side<2; side++) {
      ei.attack.all_attacks[side] = board_empty;
      for (int n=PAWN; n<=KING; n++) {
         ei.attack.all_attacks[side] |= ei.attack.attacks[side][n];
      }
   }

   /* Pinned pieces */
#if PINNED_PIECE_PENALTY > 0
   bitboard_t pinned;
   pinned = get_pinned_pieces(game->board, board_all, 0, king_square[0]);
   ev -= PINNED_PIECE_PENALTY*popcount64(pinned);
   pinned = get_pinned_pieces(game->board, board_all, 1, king_square[1]);
   ev += PINNED_PIECE_PENALTY*popcount64(pinned);
#endif

   /* Hanging pieces
    * A piece is hanging if it is attacked but not (adequately) defended.
    * We could consider a minor bonus for pieces that are defended in
    * addition.
    * TODO: tune constants
    */
   bitboard_t hanging, minor_atk;
   int hc, hp_key;
   minor_atk = ei.attack.attacks[1][KNIGHT] | ei.attack.attacks[1][BISHOP];
   hanging = board->bbc[0] & ei.attack.all_attacks[1] & ~ei.attack.all_attacks[0];
   hanging |= board->bbc[0] & ei.attack.attacks[1][PAWN];
   //hanging |= board->bbc[0] & board->bbp[QUEEN] & minor_atk;
   hc = 0;
   hp_key = 0;
   while (hanging) {
      int square = bitscan64(hanging);
      hanging ^= make_bitboard_square(square);
      int piece = get_piece(board, square) & PIECE;
      if (piece == PAWN)   hp_key |= 0x01;
      if (piece == KNIGHT) hp_key |= 0x02;
      if (piece == BISHOP) hp_key |= 0x02;
      if (piece == ROOK)   hp_key |= 0x04;
      if (piece == QUEEN)  hp_key |= 0x08;
      hc++;
   }
   if (hc > 1) hp_key |= 0x10;
   ev -= hanging_piece_penalty[hp_key];

   minor_atk = ei.attack.attacks[0][KNIGHT] | ei.attack.attacks[0][BISHOP];
   hanging = board->bbc[1] & ei.attack.all_attacks[0] & ~ei.attack.all_attacks[1];
   hanging |= board->bbc[1] & ei.attack.attacks[0][PAWN];
   //hanging |= board->bbc[1] & board->bbp[QUEEN] & minor_atk;
   hc = 0;
   hp_key = 0;
   while (hanging) {
      int square = bitscan64(hanging);
      hanging ^= make_bitboard_square(square);
      int piece = get_piece(board, square) & PIECE;
      if (piece == PAWN)   hp_key |= 0x01;
      if (piece == KNIGHT) hp_key |= 0x02;
      if (piece == BISHOP) hp_key |= 0x02;
      if (piece == ROOK)   hp_key |= 0x04;
      if (piece == QUEEN)  hp_key |= 0x08;
      hc++;
   }
   if (hc > 1) hp_key |= 0x10;
   ev += hanging_piece_penalty[hp_key];

   /* Evaluate overall board control/space advantage.
    * This is the full set of squares attacked by either player with any
    * piece (except for the king), except for squares that are attacked
    * only by enemy pawns. We give a small extra bonus for squares
    * on the opponent's half of the board and for central squares.
    * This also allows us to define a generic "piece safety", by analogy
    * with "king safety": the more threatened a piece is, the more likely
    * that there will be some combination whereby that piece is lost.
    */
   bitboard_t attackers[2];
   attackers[0] = ei.attack.attacks[0][PAWN] | ei.attack.attacks[0][KNIGHT] |
                  ei.attack.attacks[0][BISHOP] | ei.attack.attacks[0][ROOK] |
                  ei.attack.attacks[0][QUEEN];
   attackers[1] = ei.attack.attacks[1][PAWN] | ei.attack.attacks[1][KNIGHT] |
                  ei.attack.attacks[1][BISHOP] | ei.attack.attacks[1][ROOK] |
                  ei.attack.attacks[1][QUEEN];
   attackers[0] &= ~(ei.attack.attacks[1][PAWN] & ~ei.attack.attacks[0][PAWN]);
   attackers[1] &= ~(ei.attack.attacks[0][PAWN] & ~ei.attack.attacks[1][PAWN]);
   int bc = 0;
   bc += popcount64(attackers[0]) - popcount64(attackers[1]);
   bc += popcount64(attackers[0]&(board_black|board_centre)) - popcount64(attackers[1]&(board_white|board_centre));
   ev += 2*bc;
   WRITE_LOG("Board control:                          % 6d\n", ev);

   /* Colour weakness */
   ev -= COLOUR_WEAKNESS_PENALTY*(abs(popcount64(attackers[0] & board_light) -
                                      popcount64(attackers[0] & board_dark)));
   ev += COLOUR_WEAKNESS_PENALTY*(abs(popcount64(attackers[1] & board_light) -
                                      popcount64(attackers[1] & board_dark)));
   WRITE_LOG("Colour weakness:                        % 6d\n", ev);

   /* Simple end-game evaluation, using rules-of-thumb. These could
    * be replaced by bitbases. Note that lone king endings are already
    * taken care of.
    */
   switch (pieces[0]+pieces[1]) {
      case 0:     /* Pawn ending */
         /* Try the specialised evaluation function */
         if (evaluate_pawn_ending(game, side_to_move, &ev))
            return ev;

         /* we assume this is handled by either the lone king evaluation,
          * or by the king piece square tables.
          */
         break;

      case 1:     /* Lone piece or piece against or with pawn */
         /* Although not quite so simple, we let Q/R against pawn win, but
          * set minor against pawn or lone king to be at most a draw.
          * The latter is normally correct, the former may not be.
          */
         if (ev>0 && minors[0] && !pawns[0]) ev = 0;
         if (ev<0 && minors[1] && !pawns[1]) ev = 0;

         if (board->bbp[BISHOP])
            evaluate_lone_bishop(board, side_to_move, &ev);

         /* Queen against lone pawn. We know it's not queen and pawn, because then the opponent
          * would have a bare king and that's already been taken care of.
          */
         if (board->bbp[QUEEN] && (pawns[0]+pawns[1] == 1)) {
            if (board->bbp[PAWN] &
               (board_afile|board_hfile|board_cfile|board_gfile)) {   /* Queen vs rook or knight pawn */
               bool handled;
               handled = evaluate_queen_pawn(board, side_to_move, &ev, &ei.attack);
               if (handled)
                  goto done;
            }
         }

         /* Rook against lone pawn. */
         if (board->bbp[PAWN] && (pawns[0]+pawns[1] == 1)) {
            bool handled;
            handled = evaluate_king_rook_king_pawn_ending(board, side_to_move, &ei.attack, &ev);
            if (handled)
               goto done;
         }

         break;

      case 2:     /* Two pieces and (possibly) pawns */
         if (board->bbp[PAWN] == 0) {  /* No pawns */
            if (minors[0] && minors[1])   /* Draw, but keep king near the centre */
               ev = centre_table[king_square[0]] - centre_table[king_square[1]];
            else if (board->bbp[QUEEN] == 0 && pieces[0] && pieces[1]) { /* No queen -> draw(ish), but keep king near the centre */
               /* Rook vs. minor? */
               if (board->bbp[ROOK] && (minors[0]+minors[1])) {
                  bool handled = evaluate_king_rook_minor_ending(board, side_to_move, &ei.attack, &ev);
                  if (handled)
                     goto scale_fifty;
               }
            }

            /* Queen vs Rook */
            if (board->bbp[ROOK] && board->bbp[QUEEN]) {
               evaluate_queen_rook(board, side_to_move, &ei.attack, &ev);
               goto scale_fifty;
            }

         } else {
            /* Pawns+minor vs (pawns)+minor */
            if (minors[0] && minors[1]) {
               /* Check knight vs bishop and bishop vs bishop */
               if (board->bbp[KNIGHT] && board->bbp[BISHOP]) {
                  /* Knight vs. bishop: drawish if only pawns on one side, as
                   * long as the knight is on the same side of the board to
                   * stop the pawns.
                   * FIXME: doesn't properly score results when the knight
                   * is on the *attacking* team.
                   */
                  if ( !(board->bbp[PAWN] & board_qwing) || !(board->bbp[PAWN] & board_kwing) ) {
                     if ( ((board->bbp[PAWN] & board_qwing) && (board->bbp[KNIGHT] & board_qwing)) ||
                          ((board->bbp[PAWN] & board_kwing) && (board->bbp[KNIGHT] & board_kwing)) )
                        ev /= 2;

                  }
               } else { /* Bishop vs. bishop, drawish if opposite coloured bishops */
                  if ((board->bbp[BISHOP]&board_dark) && (board->bbp[BISHOP]&board_light))
                     ev /= 4;
               }
            } else if (majors[0] && majors[1]) {
               /* Pawns+major vs (pawns)+major */
               if (board->bbp[ROOK] == 0) {  /* Queen+pawn(s) vs queen+(pawns) */
                  /* More complicated than this, but the position is
                   * normally drawish. Leave it to the search to find the
                   * exceptions.
                   */
                  ev /= 4;
               } else {
                  /* Rook + pawn(s) vs. rook + (pawns),
                   * Queen + pawn(s) vs. rook + (pawns),
                   * Rook + pawn(s) vs. queen + (pawns)
                   */
                  if (!pawns[0] || !pawns[1]) {   /* Pawns on one side */
                     if ( (pawns[0] + pawns[1]) == 1) {  /* Exactly one pawn */
                        /* Probe special KRPKR ending */
                        if (board->bbp[QUEEN] == 0) {
                           bool handled = evaluate_king_rook_pawn_rook_ending(board, side_to_move, &ei.attack, &ev);
                           if (handled)
                              goto done;
                        } else {
                           /* Check fortress draws by KRP vs KQ */
                           bool handled = evaluate_king_rook_pawn_queen_ending(board, side_to_move, &ei.attack, &ev);
                           if (handled)
                              goto scale_fifty;
                        }
                     }
                     /* FIXME: we can deal with this more efficiently in
                      * the special-purpose function, which has convenient
                      * bitboards for these things.
                      */
                     if (pawns[1] == 1) {
                        /* Check for defending king in front of the pawn */
                        int pawn_square = bitscan64(board->bbp[PAWN]);
                        int promotion_square;
                        promotion_square = (pawn_square&7)|(last_rank[1]<<3);
                        if (king_distance(king_square[0], promotion_square) < 2)
                           ev /= 2;
                     } else if (pawns[0] == 1) {
                        /* Check for defending king in front of the pawn */
                        int pawn_square = bitscan64(board->bbp[PAWN]);
                        int promotion_square;
                        promotion_square=(pawn_square&7)|(last_rank[0]<<3);
                        if (king_distance(king_square[1], promotion_square) < 2)
                           ev /= 2;
                     }
                  }
               }
            }
         }

      case 3:
         if (board->bbp[KNIGHT] && board->bbp[QUEEN] && !onebit64(board->bbp[KNIGHT])) {
            if (!((board->bbp[KNIGHT] & board->bbc[0]) && (board->bbp[KNIGHT] & board->bbc[1]))) {
               /* Knights on one side, recognise draw-ish positions */
               int attacker = WHITE;
               if (board->bbp[QUEEN] & board->bbc[1])
                  attacker ^= 1;
               int defender = attacker^1;

               /* Draw-ish if the knights are next to eachother on a rank
                * or file and the defending king is between the knights and
                * the attacking king.
                */
               bitboard_t knights = board->bbp[KNIGHT];
               if ( (knights & shift_bitboard_file_left(knights)) ||
                    (knights & shift_bitboard_file_right(knights)) ||
                    (knights & shift_bitboard_row_up(knights)) ||
                    (knights & shift_bitboard_row_down(knights)) ) {
                  bitboard_t king = board->bbp[KING] & board->bbc[attacker];
                  int square = bitscan64(board->bbp[KING] & board->bbc[defender]);
                  int file = unpack_file(square);
                  int rank = unpack_rank(square);

                  if (square_separates_pieces(square, king | knights))
                     ev /= 16;

                  if ((board_rank[rank] & knights) == knights ||
                      (board_file[file] & knights) == knights) {
                     ev /= 16;
                  }
               } else {
                  int square = bitscan64(knights);
                  if (knights & knight_attack[square])
                     ev /= 4;
               }
               goto scale_fifty;
            }
         }

      default:
         break;
   }

done:
   /* Adjust the score: if the side that is nominally ahead can't win, drop the score to 0(ish) */
   if (ev > 0 && !can_win[0]) ev = psq;
   if (ev < 0 && !can_win[1]) ev = psq;

   /* If we're only up against pieces and have only one piece with mate potential, things become more
    * difficult because we can't just exchange one piece.
    */
   if (ev > 0 && pawns[0] == 0 && pawns[1] == 0 && piece_types[1] > 1 && mate_potential[0] < 2) ev /= 8;
   if (ev < 0 && pawns[0] == 0 && pawns[1] == 0 && piece_types[0] > 1 && mate_potential[1] < 2) ev /= 8;

scale_fifty:
   /* Tapered evaluation: if we approach the 50-move limit, then let the
    * score go to draw-ish smoothly.
    * Idea taken from Crafty
    */
   if (game->moves_played && game->fifty_counter[game->moves_played-1]>80)
      ev = (101 - game->fifty_counter[game->moves_played-1])*ev/20;

   WRITE_LOG("Final evaluation:                       % 6d\n", ev);


   /* Correct sign, for side to move */
   ev *= sign;

   /* Sanity check: the hashed score should equal the current score.
    * We only ever get here if we're debugging the evaluation hash.
    */
   assert(!have_eval_hash || (eh_score == ev));

   store_eval_hash_entry(game->eval_hash, game->board->hash, ev);

   /* Store score with reversed colours, but be careful: there are
    * positions where this is wrong because tempo is taken into account.
    * Mainly affects fortress-like positions in the end game and positions
    * with passed pawns.
    */
   if (piece_types[1] > 2 && piece_types[0] > 2 &&
       pawns[0] > 0 && pawns[1] > 0 &&
       pawn_structure->free == board_empty)
      store_eval_hash_entry(game->eval_hash, side_to_move_key^game->board->hash, -ev);

   return ev;
}
