/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <ctype.h>
#include "names.h"
#include "board.h"
#include "game.h"

char *make_fen_string(const gamestate_t *game, char *buffer)
{
   static char static_buffer[4096];
   char *fen = buffer;
   int empty_count;
   int file;
   int row;
   char p;
   int n = 0;

   if (!fen) fen = static_buffer;
   fen[0] = '\0';

   /* Write FEN string for this position */
   for (row=7; row>=0; row--) {
      empty_count = 0;
      for (file=0; file<8; file++) {
         int square = pack_row_file(row, file);
         int piece = -1;
         if (get_occupied(game->board) & make_bitboard_square(square))
            piece = get_piece(game->board, square);
         if (piece==-1) {
            empty_count++;
         } else {
            if (empty_count)
               n += snprintf(fen+n, 4096-n, "%d", empty_count);
            empty_count = 0;
            p = piece_str[piece&PIECE][0];
            if (piece & BLACK)
               p = tolower(p);
            n += snprintf(fen+n, 4096-n, "%c", p);
         }
      }
      if (empty_count)
         n += snprintf(fen+n, 4096-n, "%d", empty_count);
      n += snprintf(fen+n, 4096-n, "/");
   }

   /* Second record: colour */
   if (game->side_to_move & BLACK)
      n += snprintf(fen+n, 4096-n, " b");
   else
      n += snprintf(fen+n, 4096-n, " w");

   /* Third record: castling rights */
   bool may_castle = false;
   n += snprintf(fen+n, 4096-n, " ");
   bitboard_t bb = game->board->init & (game->board->bbp[KING] | game->board->bbp[ROOK]);
   if ((bb & game->board->bbc[0]) == (E1_MASK|H1_MASK)) { n += snprintf(fen+n, 4096-n, "K"); may_castle = true; }
   if ((bb & game->board->bbc[0]) == (E1_MASK|A1_MASK)) { n += snprintf(fen+n, 4096-n, "Q"); may_castle = true; }
   if ((bb & game->board->bbc[1]) == (E8_MASK|H8_MASK)) { n += snprintf(fen+n, 4096-n, "k"); may_castle = true; }
   if ((bb & game->board->bbc[1]) == (E8_MASK|A8_MASK)) { n += snprintf(fen+n, 4096-n, "q"); may_castle = true; }
   if (!may_castle) n += snprintf(fen+n, 4096-n, "-");

   /* Fourth record: En-passant square */
   if (game->board->ep_square) {
      n += snprintf(fen+n, 4096-n, " %s ", square_str[game->board->ep_square]);
   } else {
      n += snprintf(fen+n, 4096-n, " - ");
   }

   /* Fifth and sixth record: half-move counter and full-move counter */
   int par = (game->moves_played&1) && (game->side_to_move == WHITE);
   n += snprintf(fen+n, 4096 - n, "%d ", game->fifty_counter[game->moves_played]);
   n += snprintf(fen+n, 4096 - n, "%d", (int)(game->moves_played + par)/2 + 1);

   n += snprintf(fen+n, 4096-n, "\n");

   return fen;
}

void write_epd_file(gamestate_t *game, const char *filename)
{
   FILE *f;

   f = fopen(filename, "w");
   if (!f)
      return;

   fprintf(f, "%s\n", make_fen_string(game, NULL));

   fclose(f);
}
