/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <ctype.h>
#include "names.h"
#include "board.h"
#include "game.h"
#include "movegen.h"

void setup_fen_position(gamestate_t *game, const char *str)
{
   const char *s = str;
   int square;

   assert(game);
   if (!s) return;
   game->board = game->board_list;
   game->moves_played = 0;
   clear_board(game->board);

   /* Parse first record: piece positions */
   square = A8;
   while(*s && (*s != ' ')) {
      switch (*s) {
         case '/':
            square -= 15;
         case '1':
         case '2':
         case '3':
         case '4':
         case '5':
         case '6':
         case '7':
         case '8':
            square += (*s - '1')+1;
            break;

         case 'p':
            place_piece(game->board, square, piece_for_side(PAWN, BLACK));
            square++;
            break;
         case 'n':
            place_piece(game->board, square, piece_for_side(KNIGHT, BLACK));
            square++;
            break;
         case 'b':
            place_piece(game->board, square, piece_for_side(BISHOP, BLACK));
            square++;
            break;
         case 'r':
            place_piece(game->board, square, piece_for_side(ROOK, BLACK));
            square++;
            break;
         case 'q':
            place_piece(game->board, square, piece_for_side(QUEEN, BLACK));
            square++;
            break;
         case 'k':
            place_piece(game->board, square, piece_for_side(KING, BLACK));
            square++;
            break;

         case 'P':
            place_piece(game->board, square, piece_for_side(PAWN, WHITE));
            square++;
            break;
         case 'N':
            place_piece(game->board, square, piece_for_side(KNIGHT, WHITE));
            square++;
            break;
         case 'B':
            place_piece(game->board, square, piece_for_side(BISHOP, WHITE));
            square++;
            break;
         case 'R':
            place_piece(game->board, square, piece_for_side(ROOK, WHITE));
            square++;
            break;
         case 'Q':
            place_piece(game->board, square, piece_for_side(QUEEN, WHITE));
            square++;
            break;
         case 'K':
            place_piece(game->board, square, piece_for_side(KING, WHITE));
            square++;
            break;
            
      }
      s++;
   }

   game->board->init = game->board->bbc[0]|game->board->bbc[1];

   /* Second record: colour */
   while(*s && (*s == ' ')) s++;
   game->side_to_move = WHITE;
   if (*s == 'b') game->side_to_move = BLACK;
   if (*s) s++;
   if (game->side_to_move == BLACK)
      game->board->hash ^= side_to_move_key;

   /* Third record: castling rights */
   while(*s && (*s == ' ')) s++;
   game->board->init &= ~(game->board->bbp[KING] | game->board->bbp[ROOK]);
   while(*s && (*s != ' ')) {
      switch (*s) {
         case 'K':
            game->board->init |= E1_MASK;
            game->board->init |= H1_MASK;
            break;
         case 'Q':
            game->board->init |= E1_MASK;
            game->board->init |= A1_MASK;
            break;

         case 'k':
            game->board->init |= E8_MASK;
            game->board->init |= H8_MASK;
            break;
         case 'q':
            game->board->init |= E8_MASK;
            game->board->init |= A8_MASK;
            break;

         default:
            break;
      }
      s++;
   }
   if (*s) s++;

   /* Fourth record: En-passant square */
   while(*s && (*s == ' ')) s++;
   game->board->ep_square = 0;
   if (*s && (*s != '-') && isalpha(*s)) {
      int col = *s - 'a';
      int row = s[1] - '1';
      s+=2;
      game->board->ep_square = pack_row_file(row, col);
   }

   while (*s && !isspace(*s)) s++;

   /* Half-move counter, needed for the 50-move rule */
   while(*s && (*s == ' ')) s++;
   if (*s) {
      int c = 0;
      sscanf(s, "%d", &c);
      game->fifty_counter[0] = c;
   }

   /* Don't care about the rest for now... */ 

#ifndef SMP
   memset(game->repetition_hash_table, 0, sizeof game->repetition_hash_table);
   game->repetition_hash_table[game->board->hash&0xFFFF] = 1;
#endif

   /* Set the in-check flag */
   game->board->in_check = player_in_check(game->board, game->side_to_move);
   game->board->piece_count = popcount64(game->board->bbc[0]|game->board->bbc[1]);
}

gamestate_t *load_fen_position(const char *fen)
{
   gamestate_t *game = NULL;

   game = create_game();
   start_new_game(game);
   setup_fen_position(game, fen);

   return game;
}

gamestate_t *load_epd_file(const char *filename, int index)
{
   gamestate_t *game = NULL;
   char *str;
   FILE *file;
   int n;

   file = fopen(filename, "r");
   if (!file) return NULL;

   str = malloc(256);
   for (n=0; n<=index; n++) {
      fgets(str, 256, file);
   }
   fclose(file);

   game = load_fen_position(str);

   free(str);
   return game;
}
