/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <ctype.h>
#include "names.h"
#include "board.h"
#include "game.h"
#include "fen.h"
#include "movestring.h"
#include "movegen.h"

static char *chomp(char *str)
{
   char *s;
   if (str) {
      s = strstr(str, "\n");
      if (s) {
         *s = '\0';
         s--;
         while (s > str && isspace(*s)) { *s = '\0'; s--; }
      }
   }
   return str;
}

gamestate_t *load_pgn_file(const char *filename, int index)
{
   gamestate_t *game = NULL;
   move_t move;
   char *str;
   FILE *file;
   bool in_move_list = false;

   file = fopen(filename, "r");
   if (!file) return NULL;

   str = malloc(65536);

   /* Initialise game */
   game = create_game();
   start_new_game(game);

   while(true) {
      char *s;

      fgets(str, 65535, file);
      if (feof(file)) break;

      /* Remove trailing white space */
      chomp(str);

      /* Check for empty lines, which indicate the start of the next
       * section.
       */
      if (strlen(str) == 0) {
         if (in_move_list) break;
         in_move_list = true;
         continue;
      }

      /* Read in split lines */
      s = str + strlen(str)-1;
      while (*str == '[' && *s != ']') {
         s++;
         fgets(s, 65535, file);
         chomp(s);
         s = str + strlen(str)-1;
      }

      //printf("|%s|\n", str);

      /* Check for position strings */
      s = strstr(str, "FEN");
      if (s) {
         char *p;
         while (*s != '"') s++;
         s++;
         p = s;
         while (*p != '"') p++;
         *p = '\0';
         setup_fen_position(game, s);
      }

      /* Check for player name strings */
      s = strstr(str, "White");
      if (s) {
         char *p;
         while (*s != '"') s++;
         s++;
         p = s;
         while (*p != '"') p++;
         *p = '\0';
         game->name_white = strdup(s);
      }

      s = strstr(str, "Black");
      if (s) {
         char *p;
         while (*s != '"') s++;
         s++;
         p = s;
         while (*p != '"') p++;
         *p = '\0';
         game->name_black = strdup(s);
      }

      /* Moves */
      int in_comment = 0;
      if (in_move_list) {
         char *p;
         s = str;
         while (*s) {

            /* Skip move numbers */
            if (isdigit(*s)) {
               while (*s && (*s != '.')) { s++; }
               if (!*s) break;
               s++;
            }

            /* Skip spaces */
            while (*s && isspace(*s)) s++;
            if (!*s) break;

            /* Strip comments */
            if (s[0] == '{') {
               in_comment++;
               s++;
            }

            if (s[0] == '}') {
               in_comment--;
               s++;
            }

            p = s;

            while (*p && !isspace(*p)) p++;
            *p = '\0';

            /* Now s is the move string */
            //printf("->- %s\n", s);

            /* Translate the move string to a move */
            if (in_comment == 0) {
               movelist_t movelist;
               generate_legal_moves(&movelist, game->board, game->side_to_move);

               move = move_string_to_move(s, &movelist);
               //printf("-<- %s\n", move_string(move, NULL));

               if (move == 0) return game;

               playmove(game, move);
               game->last_move = game->moves_played;
            }
            s = p+1;
            //printf(">>>%s<<<\n", s);
         }
      }
   }


   fclose(file);
   free(str);
   return game;
}
