/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <inttypes.h>

#include "jazz.h"

extern uint64_t get_timer(void);

uint64_t perft(gamestate_t *game, int depth, int root)
{
   movelist_t movelist;
   int me = game->side_to_move;
   uint64_t nodes = 0;
   bool check = game->board->in_check;
   int n;

   if (depth == 0) return 1;

   generate_moves(&movelist, game->board, game->side_to_move);
   for (n=0; n<movelist.num_moves; n++) {
      uint64_t count = 0;

      playmove(game, movelist.move[n]);
      //if (check || !player_in_check(game, me))  /* Don't count illegal moves */
      if (check || !move_leaves_player_in_check(game, movelist.move[n], me))  /* Don't count illegal moves */
         count = perft(game, depth-1, root - 1);
      nodes += count;
      if (root > 0)
         printf("%8s %10"PRIu64" %10"PRIu64"\n", move_string(movelist.move[n], NULL), count, nodes);
      takeback_no_check(game);
   }
   return nodes;
}

int main(int argc, char **argv)
{
   gamestate_t *game = NULL;
   uint64_t t;
   int n;
   int depth = 6;

   printf("Welcome to Jazz\n");
   printf("Memory sizes:\n"
          " * move_t     : %lu bytes\n"
          " * bitboard_t : %lu bytes\n"
          " * hash_table_: %lu bytes\n"
          " * board_t    : %lu bytes\n",
          sizeof(move_t),
          sizeof(bitboard_t),
          sizeof(hash_table_entry_t),
          sizeof(board_t));
   printf("Size of transposition table: %lu bytes\n",
         sizeof(hash_table_entry_t) * HASH_TABLE_SIZE);

   printf("Initialising Jazz engine\n");
   initialise_jazz();

   if (argc>1) {
      int n;
      for (n=1; n<argc; n++) {
         if (strstr(argv[n], ".epd")) {
            char fen[4096];
            FILE *f;

            f = fopen(argv[n], "r");
            fgets(fen, sizeof fen, f);
            fclose(f);

            if (game) {
               end_game(game);
            }
            game = create_game();
            start_new_game(game);
            setup_fen_position(game, fen);
         } else {
            int new_depth, result;
            
            result = sscanf(argv[n], "%d", &new_depth);
            if (result == 1)
               depth = new_depth;
         }
      }
   }


   //printf("Initialising new game\n");
   if (!game) {
      game = create_game();
      start_new_game(game);
   }
   //game = load_epd_file("perft_test_position.fen", 0);

   t = get_timer();
   for (n = 1; n<depth+1; n++) {
      uint64_t nodes = perft(game, n, 0);
      uint64_t tt = get_timer();

      if (tt == t) tt++;

      printf("%2d %10lld %5.2f %12.2fnps\n", n, (long long int)nodes,
         (double)(tt - t)/1000000.0,nodes*1.0e6/(tt-t));

      t = tt;
   }

   return 0;
}

