/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>

#include "jazz.h"

#if defined __LP64__ || defined _WIN64
#define ARCHSTR "(x86_64)"
#elif defined __i386__ || defined _WIN32
#define ARCHSTR "(i386)"
#elif defined POWERPC
#define ARCHSTR "(powerpc)"
#else
#define ARCHSTR "(unknown)"
#endif 

#ifndef SVNVERSION
#define SVNVERSION "unknown revision "
#endif
#define VERSIONSTR SVNVERSION

char *chomp(char *s)
{
   char *p;
   p = strstr(s, "\n");
   if (p) *p = '\0';
   return s;
}

static bool streq(const char *s1, const char *s2)
{
   return strcmp(s1, s2) == 0;
}

static FILE *f = NULL;
static char *buf;

static void log_engine_output(const char *msg, ...) __attribute__((unused));
static void log_engine_output(const char *msg, ...)
{
   va_list ap;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   if (f) {
      fprintf(f, "  %s", buf);
      fflush(f);
   }
}

static void log_uci_output(const char *msg, ...)
{
   va_list ap;
   static bool newline = true;
   va_start(ap, msg);
   vsnprintf(buf, 65535, msg, ap);
   va_end(ap);

   if (f) {
      if (newline)
         fprintf(f, "> ");
      newline = false;
      fprintf(f, "%s", buf);
      if (strstr(buf, "\n"))
         newline = true;
   }

   printf("%s", buf);
}

/* Play a sequence of moves from the initial position */
void replay_game(gamestate_t *game, char *moves)
{
   movelist_t movelist;
   int from, to, n;
   char *s;
   /* First rewind to the beginning if not there yet */
   while (game->moves_played)
      takeback(game);

#ifndef SMP
   /* Clear the repetition hash table - it will be rebuild as we play
    * through the game.
    */
   memset(game->repetition_hash_table, 0, sizeof game->repetition_hash_table);
   game->repetition_hash_table[game->board->hash&0xFFFF] = 1;
#endif

   s = moves;
   while(s && *s) {
      generate_moves(&movelist, game->board, game->side_to_move);
      while (*s == ' ') s++;
      if (*s) {
         move_t move;
         int col = *s - 'a'; s++;
         int row = *s - '1'; s++;
         from = pack_row_file(row, col);
         col = *s - 'a'; s++;
         row = *s - '1'; s++;
         to = pack_row_file(row, col);

         n = validate_move(&movelist, from, to);
         if (n<0) {
            int k;
            for (k=0; k<movelist.num_moves; k++) {
               fprintf(stderr, "%s ", move_string(movelist.move[k], NULL));
            }
            fprintf(stderr, "\n");
            break;
         }
         move = movelist.move[n];
         if (is_promotion_move(move)) {
            switch (*s) {
               case 'q':
                  set_promotion_piece(&move, get_move_player(move) | QUEEN);
                  break;
               case 'r':
                  set_promotion_piece(&move, get_move_player(move) | ROOK);
                  break;
               case 'n':
                  set_promotion_piece(&move, get_move_player(move) | KNIGHT);
                  break;
               case 'b':
                  set_promotion_piece(&move, get_move_player(move) | BISHOP);
                  break;
            }
            s++;
         }
         playmove(game, move);
      }
   }

   write_epd_file(game, "uci_next.epd");
}

int num_games = 0;

int main(void)
{
   gamestate_t *game = NULL;
   char input[65536];
   const char *lower_piece_str[] = {
      "p", "n", "b", "r", "q", "k"
   };
   size_t hash_size = HASH_TABLE_SIZE;

   buf = malloc(65536);

   initialise_jazz();

   set_default_output_function(NULL);
   //set_default_output_function(log_engine_output);
   set_uci_output_function(log_uci_output);

   /* Turn off buffering for stdout and stdin */
   setvbuf(stdout, NULL, _IONBF, 0);
   setvbuf(stdin, NULL, _IONBF, 0);

   /* Write log output (for debugging purposes) */
   //f = fopen("uci.log", "a");

   while (true) {
      input[0] = '\0';
      if (!fgets(input, sizeof input, stdin))
         break;
      chomp(input);
      if (f) {
         fprintf(f, "< %s\n", input);
         fflush(f);
      }
      if (streq(input, "uci")) {
         printf("id name Jazz %s %s\n", VERSIONSTR, ARCHSTR);
         printf("id author Evert Glebbeek\n");
         //printf("option name OwnBook type check default true\n");
         /* Hash size: 1MB - 1 GB */
         printf("option name Hash type spin default 48 min 1 max 1024\n");
         printf("option name Ponder type check default true\n");
         printf("uciok\n");
      } else if (streq(input, "isready")) {
         printf("readyok\n");
      } else if (strstr(input, "setoption name Hash") == input) {
         /* TODO: set the size of the hash table */
         size_t memory_size;
         char *s = strstr(input, "value");
         if (s) {
            sscanf(s+6, "%lu", &memory_size);

            /* Convert to bytes */
            memory_size <<= 20;

            /* Reserve default */
            if (memory_size > 10<<20)
               memory_size -= 5<<20;

            hash_size = memory_size / sizeof(hash_table_entry_t);
         }
      } else if (streq(input, "ucinewgame")) {
         if (game) {
            end_game(game);
         }
         game = create_game();
         set_transposition_table_size(game, hash_size);
         start_new_game(game);
         num_games++;
         if (f) fprintf(f, "# game %d\n", num_games);
      } else if (strstr(input, "position")) {
         char *p = strstr(input, " ");
         while (p && *p && p[0]==' ') p++;

         if (!game) {
            game = create_game();
            set_transposition_table_size(game, hash_size);
            start_new_game(game);
         }

         if (strstr(p, "startpos") == p) {   /* Game from start position */
            /* We need to return to the first node in the game struct and
             * then replay all the moves from the beginning to the current
             * position.
             */
            p = strstr(p, "moves");
            if (p && *p) p+=6;
            replay_game(game, p);
         } else if (strstr(p, "fen") == p) { /* Game from FEN string */
            p+=4;
            setup_fen_position(game, p);

            p = strstr(p, "moves");
            if (p && *p) p+=6;
            replay_game(game, p);
         }
      } else if (strstr(input, "go")) {
         move_t move;
         const char *p = "";
         int depth = 60;
         char *s;

         /* Set defaults */
         set_time_per_move(game, 5000);
         game->movestogo = 0;
         game->time_inc[0] = game->time_inc[1] = 0;

         /* parse options */
         if ((s = strstr(input, "ponder"))) {   /* Can't ponder */
            computer_ponder(game);
            continue;
         }
         if ((s = strstr(input, "movetime"))) {
            int time_per_move = 5000;
            sscanf(s+9, "%d", &time_per_move);
            set_time_per_move(game, time_per_move);
         }
         if ((s = strstr(input, "movestogo"))) {
            sscanf(s+10, "%d", &game->movestogo);
         }
         if ((s = strstr(input, "infinite"))) {
            set_infinite_time(game);
         }
         if ((s = strstr(input, "wtime"))) {
            sscanf(s+6, "%d", &game->time_left[0]);
            set_time_for_game(game);
         }
         if ((s = strstr(input, "btime"))) {
            sscanf(s+6, "%d", &game->time_left[1]);
            set_time_for_game(game);
         }
         if ((s = strstr(input, "winc"))) {
            sscanf(s+5, "%d", &game->time_inc[0]);
         }
         if ((s = strstr(input, "binc"))) {
            sscanf(s+5, "%d", &game->time_inc[1]);
         }
         if ((s = strstr(input, "mate"))) {  /* Just do a normal N-ply search instead */
            sscanf(s+5, "%d", &depth);
            set_infinite_time(game);
         }
         if ((s = strstr(input, "depth"))) {
            sscanf(s+6, "%d", &depth);
            set_infinite_time(game);
         }

         computer_play(game, depth);

         move = game->move_list[game->moves_played-1];
         if (is_promotion_move(move)) {
            p = lower_piece_str[get_promotion_piece(move) & PIECE];
         }
         if (game->ponder_move) {
            const char *pp = "";
            move_t pmove = game->ponder_move;
            if (is_promotion_move(pmove))
               pp = lower_piece_str[get_promotion_piece(pmove) & PIECE];
            printf("bestmove %s%s%s ponder %s%s%s\n",
                  square_str[get_move_origin(move)], square_str[get_move_destination(move)], p,
                  square_str[get_move_origin(pmove)], square_str[get_move_destination(pmove)], pp);
         } else {
            printf("bestmove %s%s%s\n", square_str[get_move_origin(move)],
                  square_str[get_move_destination(move)], p);
         }
      } else if (strstr(input, "ponderhit")) {  /* Nothing special yet... */
      } else if (streq(input, "quit")) {
         exit(0);
      }
   }

   return 0;
}

