/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include "config.h"
#include "expect.h"
#include "square.h"
#include "movegen.h"
#include "board.h"
#include "game.h"
#include "hashtable.h"
#include "pawn_table.h"
#include "evalhash.h"
#include "fen.h"

#define PIECESTRING   " NBRQK"

static void (*default_iteration_output)(const char *, ...) = NULL;
static void (*default_uci_output)(const char *, ...) = NULL;
static void (*default_xboard_output)(const char *, ...) = NULL;

void playgame_init(void)
{
   default_iteration_output = (void *)printf;
   default_uci_output = NULL;
}

void playgame_shutdown(void)
{
}

void set_default_output_function(void (*func)(const char *, ...))
{
   default_iteration_output = func;
}

void set_uci_output_function(void (*func)(const char *, ...))
{
   default_uci_output = func;
}

void set_xboard_output_function(void (*func)(const char *, ...))
{
   default_xboard_output = func;
}

void set_transposition_table_size(gamestate_t *game, uint64_t nelem)
{
   if (!game)
      return;

   /* Round to the next-lowest power of 2 */
   nelem |= (nelem >> 1);
   nelem |= (nelem >> 2);
   nelem |= (nelem >> 4);
   nelem |= (nelem >> 8);
   nelem |= (nelem >> 16);
   nelem |= (nelem >> 32);
   nelem >>= 1;
   nelem++;

   game->default_hash_size = nelem;

   /* Reallocate table if it was already allocated */
   if (game->default_hash_size) {
      destroy_hash_table(game->transposition_table);
      game->transposition_table = create_hash_table(game->default_hash_size);
   }
}

gamestate_t *create_game(void)
{
   return calloc(1, sizeof(gamestate_t));
}

/* Initialize the game datastructure to correspond to a new game */
void start_new_game(gamestate_t *game)
{
   /* Load the starting position */
   if (game->default_hash_size == 0) game->default_hash_size = HASH_TABLE_SIZE;
   game->max_moves = 1000;
   game->score = calloc(game->max_moves, sizeof *game->score);
   game->fifty_counter = calloc(game->max_moves, sizeof(*game->fifty_counter));
   game->move_list = calloc(game->max_moves, sizeof *game->move_list);
   game->board_list = calloc(game->max_moves, sizeof *game->board_list);
   game->board = game->board_list;

   game->transposition_table = create_hash_table(game->default_hash_size);
   game->pawn_structure = create_pawn_hash_table(PAWN_TABLE_SIZE);
   game->queen_psq = create_piece_hash_table(PIECE_TABLE_SIZE);
   game->rook_psq = create_piece_hash_table(PIECE_TABLE_SIZE);
   game->knight_psq = create_piece_hash_table(PIECE_TABLE_SIZE);
   game->bishop_psq = create_piece_hash_table(PIECE_TABLE_SIZE);
   game->king_psq = create_piece_hash_table(PIECE_TABLE_SIZE);
   game->eval_hash = create_eval_hash_table(game->default_hash_size);

   setup_fen_position(game, "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1");

   
   game->moves_played = 0;
   game->last_move = 0;

   game->analysing = false;
   game->analyse_move = 0;
   game->analyse_fen = NULL;
   game->analyse_new = 0;
   game->analyse_undo = 0;
   game->pondering = false;
   game->random_ok = true;
   game->ponder_move = 0;

   game->name_white = NULL;
   game->name_black = NULL;
   game->out_book = false;

   game->output_iteration = default_iteration_output;
   game->uci_output = default_uci_output;
   game->xboard_output = default_xboard_output;
}

/* Frees the memory used by a gamestate */
void end_game(gamestate_t *game)
{
   if (game) {
      close_opening_book(game->book);

      free(game->name_white);
      free(game->name_black);
      free(game->move_list);
      free(game->board_list);
      free(game->score);
      free(game->fifty_counter);
      destroy_hash_table(game->transposition_table);
      destroy_eval_hash_table(game->eval_hash);
      destroy_pawn_hash_table(game->pawn_structure);
      destroy_piece_hash_table(game->queen_psq);
      destroy_piece_hash_table(game->rook_psq);
      destroy_piece_hash_table(game->knight_psq);
      destroy_piece_hash_table(game->bishop_psq);
      destroy_piece_hash_table(game->king_psq);
      free(game);
   }
}

/* Play a null-move; we don't simply flip the current player because that
 * will mess up the killer tables (and possibly the transposition table,
 * since the hash keys are not properly updated).
 */
void play_null_move(gamestate_t *game)
{

   game->move_list[game->moves_played] = 0;
   game->moves_played++;
   game->board++;
   memcpy(game->board, game->board-1, sizeof *game->board);

   game->side_to_move ^= BLACK;
   game->board->hash ^= side_to_move_key;
   game->board->hash ^= en_passant_key[game->board->ep_square];
   game->board->hash ^= en_passant_key[0];
   game->board->ep_square = 0;
#ifndef SMP
   game->repetition_hash_table[game->board->hash&0xFFFF]++;
#endif
}

void playmove(gamestate_t *game, move_t move)
{
   assert(game);

   /* The following assertion should be prevented from a higher level: we
    * don't want to check whether we have the space to store the next move
    * in a low level function like this. The search algorithm should make
    * sure that there is enough space to store to the end of the current
    * search depth before trying to play any moves. In other words,
    * max_moves-moves_played needs to be larger than the horizon depth,
    * allowing for quiescence search. We could take the horizon depth + 100
    * and always be safe.
    */
   assert(game->moves_played < game->max_moves);

   game->board++;
   memcpy(game->board, game->board-1, sizeof *game->board);

   /* This should not be a NULL move */
   assert(move);

   if (is_capture_move(move)) {
      uint8_t ptaken = get_captured_piece(move);

      assert((ptaken & PIECE) < KING);
      assert((ptaken & PIECE) >= PAWN);

      /* Remove piece from the capture square (normally the same as the
       * destination, except on e-p moves)
       */
      remove_known_piece(game->board, ptaken, get_capture_square(move));
      game->board->piece_count--;
   }

   game->move_list[game->moves_played] = move;
   game->moves_played++;

   /* Advance the 50-move counter if this move was reversible, reset if it
    * wasn't.
    */
   game->fifty_counter[game->moves_played] =
      is_irreversible_move(move) ? 0 : (game->fifty_counter[game->moves_played-1] + 1);

   /* Move the piece to its new location */
   move_piece(game->board, get_move_piece(move), get_move_origin(move), get_move_destination(move));

   /* Store en-passant square
    * The hash key for this needs some clarification. If there is no EP
    * square, then move.ep = board->ep_square = 0 and this is a NOP. If
    * there is, then this places an extra white and black pawn on the board
    * on this square, which should be a fairly unique combination.
    * Note that the en-passant square is not set if no enemy pawn is
    * present to exact the capture, which is the common case. This
    * prevents pollution of the hash table where identical positions would
    * otherwise be stored differently.
    */
   if (expect(game->board->ep_square != get_ep_square(move), false)) {
      game->board->hash ^= en_passant_key[get_ep_square(move)];
      game->board->hash ^= en_passant_key[game->board->ep_square];
      game->board->ep_square = get_ep_square(move);
   }

   /* Check special moves: pawn promotions and castling */
   if (expect(is_promotion_move(move), false)) {
      int piece = get_move_piece(move);
      assert((piece & PIECE) == PAWN);
      remove_known_piece(game->board, piece, get_move_destination(move));
      piece = get_promotion_piece(move);
      assert((piece & PIECE) >= KNIGHT);
      assert((piece & PIECE) <= QUEEN);
      place_promotion_piece(game->board, get_move_destination(move), piece);
   }

   /* Castling; the King has already been moved, now just need to move the
    * rook.
    */
   if (expect(is_castle_move(move), false)) {
      game->board->did_castle[get_move_player(move)>>7] = true;
      castle_rook(game->board, get_move_player(move)>>7, get_castle_flags(move));
   }

   game->side_to_move ^= BLACK;
   game->board->hash ^= side_to_move_key;
#ifndef SMP
   game->repetition_hash_table[game->board->hash&0xFFFF]++;
#endif

   /* Update check state */
   /* FIXME: do we really need this here? It slows things down quite a
    * bit... there might be a more clever way to do this.
    * Note that the perft test is misleading here - the actual search will
    * also do the test that we do here.
    */
   game->board->in_check = move_checked_player(game->board, game->side_to_move, move);
}

/* Like the regular takeback function, but doesn't check if any moves have
 * been played. Can be used instead of takeback in the search function to
 * avoid an unnecessary branch there.
 */
void takeback_no_check(gamestate_t *game)
{
   __builtin_prefetch(game->board-1);
#ifndef SMP
   __builtin_prefetch(&(game->repetition_hash_table[game->board->hash&0xFFFF]));
#endif
   game->side_to_move ^= BLACK;
   game->moves_played--;
#ifndef SMP
   game->repetition_hash_table[game->board->hash&0xFFFF]--;
#endif
   game->board--;
}

/* To take back a move, simply go back one spot in the array. We could save
 * memory by actually reversing the move, but what's the point? Memory is
 * cheap these days!
 */
void takeback(gamestate_t *game)
{
   assert(game);
   if (game->moves_played) {
      takeback_no_check(game);
   }
}

/* Redo the last move: if a move has been reverted, but no new move has
 * been played yet.
 * Mainly for player interaction.
 */
void redo_last_move(gamestate_t *game)
{
   if (game->moves_played < game->last_move) {
      game->moves_played++;
      game->board++;
      game->side_to_move ^= BLACK;
#ifndef SMP
      game->repetition_hash_table[game->board->hash&0xFFFF]++;
#endif
   }
}

