/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdbool.h>
#include <string.h>
#include "bits.h"
#include "move.h"
#include "movestring.h"
#include "board.h"
#include "bitboard.h"
#include "square.h"
#include "names.h"
#include "movegen.h"

#undef WRITE_OUTPUT_FROM_MOVE_GENERATOR

/* Attack bitboards: for a given square for each piece and for selected
 * occupation numbers, return all squares attacked by a piece (ie, all
 * possible moves).
 * We have row/file bitboards, diagonal bitboards, knight and king
 * bitboards.
 * We can safe quite a bit of space (important for caching more than
 * anything else) by discarding the end points on the bitboard (a piece
 * will always attack those and can't pass beyond them anyway).
 * NB: although we conceptually use rotated bitboards to store
 * row/column/diagonal occupation numbers, the bitboards in the below data
 * structure are strictly regular.
 */
bitboard_t king_attack[64];
bitboard_t king_attack2[64];
bitboard_t knight_attack[64];
bitboard_t knight_attack2[64];
bitboard_t super_attack[64];
bitboard_t rays[64];
bitboard_t orth[64];
bitboard_t diag[64];
bitboard_t attack_row[64][64];
bitboard_t attack_file[64][64];
bitboard_t attack_diagonal_a8h1[64][64];
bitboard_t attack_diagonal_a1h8[64][64];
bitboard_t ep_neighbours[16];
bitboard_t connecting_ray[64][64];

/* Discovered or indirect attacks: squares along the direction beyond the
 * first blocker. This is useful for detecting discovered attacks, X-ray
 * attacks and pins. The logical and between these bitboards and the normal
 * attack bitboards indicates the blockers.
 *  1. For a rook attack, if a blocker is a rook or queen, we have X-ray
 *     attacks of the rook through the queen and the queen through the rook.
 *  2. For a bishop, if a blocker is a queen (or bishop, but that is
 *     unlikely) then there is an X-ray attack of the bishop through the
 *     queen and of the queen through the bishop
 *  3. For a rook, if a blocker is an enemy rook, there is possibly an
 *     X-ray defence of squares behind the enemy rook. Vice versa for
 *     bishops and queens.
 *  4. If there is an 'X-ray' looking at the opposing king, we have either
 *     a pin or a discovered check.
 */
bitboard_t xattack_row[64][64];
bitboard_t xattack_file[64][64];
bitboard_t xattack_diagonal_a8h1[64][64];
bitboard_t xattack_diagonal_a1h8[64][64];

/* Population count/mobility */
int8_t mobility_row[64][64];
int8_t mobility_file[64][64];
int8_t mobility_diagonal_a8h1[64][64];
int8_t mobility_diagonal_a1h8[64][64];

static void print_uint8_t(uint8_t b) __attribute__((unused));
static void print_uint8_t(uint8_t b)
{
   int c;
   
   for (c=0; c<8; c++) {
      printf ("%d", (b>>c)&1);
   }
}

//#define DEBUG_DIAGONAL_BITBOARD_MAPPING
/* Compute the bitboards and other static data structures needed for move
 * generation.
 */
void generate_movement_tables(void)
{
   int row, file, square;
   int occ, bit;
   int attack, xattack;

#ifdef DEBUG_DIAGONAL_BITBOARD_MAPPING
   for (row=7; row>=0; row--) {
      /* Write sequential number of diagonal */
      for (file=0; file<8; file++) {
         int square = pack_row_file(row, file);
         int diag_nr = get_a8h1_diagonal(square);
         printf(" %02d", diag_nr);
      }
      printf("  ");
      /* The position of each square within the diagonal */
      for (file=0; file<8; file++) {
         int square = pack_row_file(row, file);
         printf(" %02d", get_a8h1_diagonal_ord(square));
      }
      printf("  ");
      /* The offset of the start of the diagonal in the rotated bitboard */
      for (file=0; file<8; file++) {
         int square = pack_row_file(row, file);
         printf(" %02d", get_a8h1_diagonal_offset(square));
      }
      printf("  ");
      /* The permutation of numbers along the bitboard */
      for (file=0; file<8; file++) {
         int square = pack_row_file(row, file);
         printf(" %02d", get_a8h1_diagonal_square(square));
      }
      printf("\n");
   }

   printf("\n");
   for (row=7; row>=0; row--) {
      /* Write sequential number of diagonal */
      for (file=0; file<8; file++) {
         int square = pack_row_file(row, file);
         int diag_nr = get_a1h8_diagonal(square);
         printf(" %02d", diag_nr);
      }
      printf("  ");
      /* The position of each square within the diagonal */
      for (file=0; file<8; file++) {
         int square = pack_row_file(row, file);
         printf(" %02d", get_a1h8_diagonal_ord(square));
      }
      printf("  ");
      /* The offset of the start of the diagonal in the rotated bitboard */
      for (file=0; file<8; file++) {
         int square = pack_row_file(row, file);
         printf(" %02d", get_a1h8_diagonal_offset(square));
      }
      printf("  ");
      /* The permutation of numbers along the bitboard */
      for (file=0; file<8; file++) {
         int square = pack_row_file(row, file);
         printf(" %02d", get_a1h8_diagonal_square(square));
      }
      printf("\n");
   }
   exit(0);
#endif

   /* Initialise general purpose bitboards */
   compute_pawn_bitboard_masks();

   /* Generate movement tables for the king and the knight.
    * The movement tables for the knight are relatively simple,
    * because knights can't be blocked. On the discrete grid of the chess
    * board, the possible destinations of the knight lie on a circle with
    * radius 5.
    * The king likewise can't be blocked, because it can only move one
    * square at a time.
    */
   for (square=0; square<64; square++) {
      int dx, dy;
      row = unpack_rank(square);
      file = unpack_file(square);

      king_attack[square] = board_empty;
      for (dx=-1; dx<2; dx++)
         for (dy=-1; dy<2; dy++) {
            if ( (dx||dy) &&
                  (row+dx>=0 && row+dx<=7 && file+dy>=0 && file+dy<=7) ) {
               set_bitboard(&(king_attack[square]), pack_row_file(row+dx,file+dy));
            }
         }

      knight_attack[square] = board_empty;
      for (dx=-3; dx<4; dx++)
         for (dy=-3; dy<4; dy++) {
            if ( (dx*dx + dy*dy == 5) &&
                  (row+dx>=0 && row+dx<=7 && file+dy>=0 && file+dy<=7) ) {
               set_bitboard(&(knight_attack[square]), pack_row_file(row+dx,file+dy));
            }
         }
   }

   /* Generate row and file attack bitboards */
   for (file=0; file<8; file++) {
      for (occ=0; occ<256; occ++) {
         int occupied = occ & ~(1<<file);
         attack = 0;
         xattack = 0;
         /* Normal attacks */
         bit = file;
         while ( bit>0 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit-1);
            bit--;
         }
         /* X-ray/indirect attacks */
         xattack |= 1<<bit;
         bit--;
         xattack |= 1<<bit;
         while ( bit>0 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit-1);
            bit--;
         }

         /* Normal attacks */
         bit = file;
         while ( bit<7 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit+1);
            bit++;
         }
         /* X-ray/indirect attacks */
         xattack |= 1<<bit;
         bit++;
         xattack |= 1<<bit;
         while ( bit<7 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit+1);
            bit++;
         }

         /* store the generated bitboards */
         for(row=0; row<8; row++) {
            attack_row[pack_row_file(row, file)][(occ>>1)&63] = make_bitboard_row(attack, row);
            attack_file[pack_row_file(file, row)][(occ>>1)&63] = make_bitboard_file(attack, row);

            xattack_row[pack_row_file(row, file)][(occ>>1)&63] = make_bitboard_row(xattack, row);
            xattack_file[pack_row_file(file, row)][(occ>>1)&63] = make_bitboard_file(xattack, row);
         }
      }
   }

   /* Generate attack bitboards for the diagonals */
   for (square=0; square<64; square++) {
      for (occ=0; occ<256; occ++) {
         int diag_nr = get_a8h1_diagonal(square);
         /* Find the length of the diagonal: this is the number of bits in
          * the occupancy number that we actually care about.
          */
         int diag_length = 8 - abs(7 - diag_nr);
         int occupied = 0;
         int occ_index = occ;

         /* The occupancy lookup for diagonals is slighty different when
          * not using rotated bitboards.
          */
#ifndef USE_ROTATED_BITBOARDS
         bitboard_t bb;
         bb = make_bitboard_a8h1_diagonal(occ, square);
         occ_index = (bb * board_bfile) >> 57;
#endif

         bit = get_a8h1_diagonal_ord(square);

         /* Compute effective occupancy number: we don't care about the
          * square the piece we're computing the moves for is on (we know
          * there's a piece there) and we don't care about bits that
          * are on the extension of diagonal beyond the edge of the board.
          * We unset the bit corresponding to the current square and set
          * the bits that are off the board. This makes the code below a
          * little bit simpler.
          */
         occupied = (uint8_t)((occ & ~(1<<bit)) | ~((1<<diag_length)-1));
         attack = 0;
         xattack = 0;

         /* Scan both directions of the ray */
         /* Normal attacks */
         bit = get_a8h1_diagonal_ord(square);
         while ( bit>0 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit-1);
            bit--;
         }
         /* X-ray attacks */
         xattack |= 1<<bit;
         bit--;
         xattack |= 1<<bit;
         while ( bit>0 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit-1);
            bit--;
         }

         /* Normal attacks */
         bit = get_a8h1_diagonal_ord(square);
         while ( bit<7 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit+1);
            bit++;
         }
         /* X-ray attacks */
         xattack |= 1<<bit;
         bit++;
         xattack |= 1<<bit;
         while ( bit<7 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit+1);
            bit++;
         }

         /* We don't care about the edge bits; they're irrelevant.
          * For X-rays, make sure we mask out the home square.
          */
         attack_diagonal_a8h1[square][(occ_index>>1)&63] =
                     make_bitboard_a8h1_diagonal(attack, square);
         xattack_diagonal_a8h1[square][(occ_index>>1)&63] =
                     make_bitboard_a8h1_diagonal(xattack, square) & ~make_bitboard_square(square);

         /* Now we repeat this exercise for diagonals parallel to a1-h8 */
         diag_nr = get_a1h8_diagonal(square);
         /* Find the length of the diagonal: this is the number of bits in
          * the occupancy number that we actually care about.
          */
         diag_length = 8 - abs(7 - diag_nr);

         /* The occupancy lookup for diagonals is slighty different when
          * not using rotated bitboards.
          */
#ifndef USE_ROTATED_BITBOARDS
         bb = make_bitboard_a1h8_diagonal(occ, square);
         occ_index = (bb * board_bfile) >> 57;
#endif

         bit = get_a1h8_diagonal_ord(square);

         /* Compute effective occupancy number: we don't care about the
          * square the piece we're computing the moves for is on (we know
          * there's a piece there) and we don't care about bits that
          * are on the extension of diagonal beyond the edge of the board.
          * We unset the bit corresponding to the current square and set
          * the bits that are off the board. This makes the code below a
          * little bit simpler.
          */
         occupied = (uint8_t)((occ & ~(1<<bit)) | ~((1<<diag_length)-1));
         attack = 0;
         xattack = 0;

         /* Scan both directions of the ray */
         /* Normal attacks */
         bit = get_a1h8_diagonal_ord(square);
         while ( bit>0 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit-1);
            bit--;
         }
         /* X-ray attacks */
         xattack |= 1<<bit;
         bit--;
         xattack |= 1<<bit;
         while ( bit>0 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit-1);
            bit--;
         }

         /* Normal attacks */
         bit = get_a1h8_diagonal_ord(square);
         while ( bit<7 && !(occupied&1<<bit) ) {
            attack |= 1<<(bit+1);
            bit++;
         }
         xattack |= 1<<bit;
         bit++;
         xattack |= 1<<bit;
         while ( bit<7 && !(occupied&1<<bit) ) {
            xattack |= 1<<(bit+1);
            bit++;
         }

         /* We don't care about the edge bits; they re irrelevant. */
         attack_diagonal_a1h8[square][(occ_index>>1)&63] =
                     make_bitboard_a1h8_diagonal(attack, square);
         xattack_diagonal_a1h8[square][(occ_index>>1)&63] =
                     make_bitboard_a1h8_diagonal(xattack, square) & ~make_bitboard_square(square);
      }
   }

   /* Make sure squares are not recorded as attacking themselves */
   for (square=0; square<64; square++) {
      for (occ=0; occ<64; occ++) {
         attack_row[square][occ] &= ~make_bitboard_square(square);
         attack_file[square][occ] &= ~make_bitboard_square(square);
         attack_diagonal_a1h8[square][occ] &= ~make_bitboard_square(square);
         attack_diagonal_a8h1[square][occ] &= ~make_bitboard_square(square);
      }
   }

   /* Generate test bitboards for en-passant captures: neighbouring squares
    * for each file (shifted such that A4 = 0)
    */
   for (square = 0; square < 16; square++)
      ep_neighbours[square] = 0;
   for (square = 1; square < 8; square++) {
      ep_neighbours[square] |= make_bitboard_square(A4+square-1);
      ep_neighbours[8+square] |= make_bitboard_square(A5+square-1);
   }
   for (square = 0; square < 7; square++) {
      ep_neighbours[square] |= make_bitboard_square(A4+square+1);
      ep_neighbours[8+square] |= make_bitboard_square(A5+square+1);
   }

   /* Connecting rays and superpiece attacks */
   memset(connecting_ray, 0, sizeof connecting_ray);
   for (square = A1; square <= H8; square++) {
      for (attack = square+1; attack<=H8; attack++) {
         int row = unpack_rank(square);
         int file = unpack_file(square);
         if (row == unpack_rank(attack)) {
            int n;
            for (n=file;n<=unpack_file(attack);n++)
               connecting_ray[square][attack] |=
                  make_bitboard_square(pack_row_file(row, n));
         }
         if (file == unpack_file(attack)) {
            int n;
            for (n=row;n<=unpack_rank(attack);n++)
               connecting_ray[square][attack] |=
                  make_bitboard_square(pack_row_file(n, file));
         }
         if (get_a1h8_diagonal(square) == get_a1h8_diagonal(attack)) {
            int n;
            for (n=square;n<=(attack);n+=9)
               connecting_ray[square][attack] |= make_bitboard_square(n);
         }
         if (get_a8h1_diagonal(square) == get_a8h1_diagonal(attack)) {
            int n;
            for (n=square;n<=(attack);n+=7)
               connecting_ray[square][attack] |= make_bitboard_square(n);
         }
         connecting_ray[attack][square] = connecting_ray[square][attack];
      }

      orth[square] = attack_row[square][0] |
                     attack_file[square][0];
      diag[square] = attack_diagonal_a8h1[square][0] |
                     attack_diagonal_a1h8[square][0];
      rays[square] = attack_row[square][0] |
                     attack_file[square][0] |
                     attack_diagonal_a8h1[square][0] |
                     attack_diagonal_a1h8[square][0];
      super_attack[square] = rays[square] | knight_attack[square];
   }

   /* Mobility information.
    * A bit crude, because we're still masking off the edge of the board
    */
   for (square = A1; square <= H8; square++) {
      for (occ=0; occ<64; occ++) {
         /* Rook */
         mobility_row[square][occ]           = popcount64(attack_row[square][occ]) - 2;
         mobility_file[square][occ]          = popcount64(attack_file[square][occ]) - 1;

         /* Bishop */
         mobility_diagonal_a8h1[square][occ] = popcount64(attack_diagonal_a8h1[square][occ]) +
                                               popcount64(attack_diagonal_a8h1[square][occ] & board_centre) +
                                               popcount64(attack_diagonal_a8h1[square][occ] & board_xcentre) +
                                               popcount64(xattack_diagonal_a8h1[square][occ] & board_centre) - 4;
         mobility_diagonal_a1h8[square][occ] = popcount64(attack_diagonal_a1h8[square][occ]) +
                                               popcount64(attack_diagonal_a1h8[square][occ] & board_centre) +
                                               popcount64(attack_diagonal_a1h8[square][occ] & board_xcentre) +
                                               popcount64(xattack_diagonal_a1h8[square][occ] & board_centre) - 4;
      }
   }

   /* Squares that can be attacked by a king in at most 2 moves, used for
    * some end game evaluation terms.
    */
   for (square = A1; square <= H8; square++) {
      bitboard_t bb = king_attack2[square] = king_attack[square];
      while (bb) {
         int s = bitscan64(bb);
         king_attack2[square] |= king_attack[s];
         unset_bitboard(&bb, s);
      }
   }

   /* Squares that can be attacked by a knight in 2 moves, used for
    * some evaluation terms.
    */
   for (square = A1; square <= H8; square++) {
      bitboard_t bb = knight_attack[square];
      knight_attack2[square] = 0;
      while (bb) {
         int s = bitscan64(bb);
         knight_attack2[square] |= knight_attack[s];
         unset_bitboard(&bb, s);
      }
   }


   /* Sanity test */
#if 0
   for (square = A1; square <= H8; square++) {
      for (attack = A1; attack<=H8; attack++) {
         bool f1, f2;
         f1 = (rays[square] & make_bitboard_square(attack)) != 0;
         f2 = connecting_ray[square][attack] != 0;
         if (f1 ^ f2) {
            printf("Discrepancy!\n");
            exit(0);
         }
      }
   }
#endif
}

/* Helper to generate pawn moves, except double moves and en-passant moves */
#define generate_pawn_single_move(bb, disp, cap)               \
      while(promote_pawns) {                                   \
         int dest = bitscan64(promote_pawns);                  \
         int pp = allowed_promotion_pieces;                    \
         int p;                                                \
         promote_pawns ^= make_bitboard_square(dest);          \
         while (pp) {                                          \
            p = bitscan16(pp);                                 \
            pp ^= (1<<p);                                      \
         *move = encode_promotion_move(piece_for_side(p, player), dest + disp, dest);\
         cap;                                                  \
         move++;                                               \
         }                                                     \
      }                                                        \
                                                               \
      while(bb) {                                              \
         int dest = bitscan64(bb);                             \
         bb ^= make_bitboard_square(dest);                     \
         *move = encode_normal_move(piece_for_side(PAWN, player), dest + disp, dest);\
         cap;                                                  \
         move++;                                               \
      }

/* Helper macro to generate pawn moves.
 * Apart from a few signs and constants, the code for these is exactly the
 * same for white and for black.
 */
#define generate_pawn_moves(shift_f,shift_l,shift_r,sign,dl,dr,rank3,rank8)\
{                                                                       \
   /* First we separate the pawns into three groups: those that can move\
    * two paces, those that can promote and the rest.                   \
    * shift all pawns forward one row, check against empty squares. This\
    * gives us all pawns that can move.                                 \
    */                                                                  \
   bb = and_bitboards(~occupied, shift_f(pawns));                       \
   /* Keep seperate track of pawns that can do a double move or promote \
    * Only pawns that can get_move_destination(move) the third row could get_move_destination(move) the fourth\
    */                                                                  \
   double_pawns = and_bitboards(~occupied, shift_f(and_bitboards(bb, rank3)));\
                                                                        \
   /* Do not double check pawns we already check for promotion */       \
   bb &= destination_mask;                                              \
   promote_pawns = and_bitboards(rank8, bb);                            \
   bb ^= promote_pawns;                                                 \
                                                                        \
   /* Now generate pawn moves */                                        \
   /* Single steps */                                                   \
   generate_pawn_single_move(bb, sign 8, (void)0);                      \
   /* Double steps */                                                   \
   double_pawns = and_bitboards(double_pawns, destination_mask);        \
   while(double_pawns) {                                                \
      int dest;                                                         \
      int ep = 0;                                                       \
                                                                        \
      dest = bitscan64(double_pawns);                                   \
      /* Only set E-P square if enemy pawns are present */              \
      if (enemy_pawns & ep_neighbours[dest-A4]) ep = dest sign 8;       \
      *move = encode_doublep_move(piece_for_side(PAWN, player),                          \
                                       dest sign 16, dest, ep);         \
      move++;                                                           \
                                                                        \
      double_pawns ^= make_bitboard_square(dest);                       \
   }                                                                    \
   /* Captures right */                                                 \
   bb = enemy & shift_r(pawns);                                         \
   bb &= destination_mask;                                              \
   promote_pawns = and_bitboards(rank8, bb);                            \
   bb ^= promote_pawns;                                                 \
   generate_pawn_single_move(bb, sign dr, set_captured_piece(move, get_piece(board, get_capture_square(*move)));set_capture_flag(move));      \
   /* Captures left */                                                  \
   bb = enemy & shift_l(pawns);                                         \
   bb &= destination_mask;                                              \
   promote_pawns = and_bitboards(rank8, bb);                            \
   bb ^= promote_pawns;                                                 \
   generate_pawn_single_move(bb, sign dl, set_captured_piece(move, get_piece(board, get_capture_square(*move)));set_capture_flag(move));      \
   /* En-passant */                                                     \
   if (en_passant) {                                                    \
      /* en-passant captures, right */                                  \
      bb = and_bitboards(en_passant, shift_r(pawns));                   \
      if(bb) {                                                          \
         int dest = bitscan64(bb);                                          \
         *move = encode_enpassant_capture(piece_for_side(PAWN, player), dest sign dr,    \
                                                   dest, dest sign 8);  \
         set_captured_piece(move, get_piece(board, get_capture_square(*move)));      \
         move++;                                                        \
      }                                                                 \
      /* en-passant captures, left */                                   \
      bb = and_bitboards(en_passant, shift_l(pawns));                   \
      if(bb) {                                                          \
         int dest = bitscan64(bb);                                          \
         *move = encode_enpassant_capture(piece_for_side(PAWN, player), dest sign dl,    \
                                                   dest, dest sign 8);  \
         set_captured_piece(move, get_piece(board, get_capture_square(*move)));      \
         move++;                                                        \
      }                                                                 \
   }                                                                    \
}

static move_t *serialise_normal_moves(move_t *move, bitboard_t moves, int piece, int from)
{
   while(moves) {
      int to = bitscan64(moves);
      moves ^= make_bitboard_square(to);

      *move = encode_normal_move(piece, from, to);
      move++;
   }

   return move;
}

static move_t *serialise_capture_moves(move_t *move, bitboard_t moves, const board_t *board, int piece, int from)
{
   while(moves) {
      int to = bitscan64(moves);
      moves ^= make_bitboard_square(to);

      *move = encode_normal_move(piece, from, to);
      uint8_t ptaken = get_piece(board, to);
      set_captured_piece(move, ptaken);
      set_capture_flag(move);
      move++;
   }

   return move;
}

/* General move generation function: generate all pseudo-legal moves for
 * pieces on the board under the 'source_mask' to the 'destination_mask'
 */
static void generate_moves_mask(movelist_t *movelist, const board_t *board, bitboard_t source_mask, bitboard_t destination_mask, int player, int allowed_promotion_pieces)
{
   static const bitboard_t castle_mask[2][2] = {
      /* King side */
      { board_castlek&board_rank1, board_castlek&board_rank8 },
      /* Queen side */
      { board_castleq&board_rank1, board_castleq&board_rank8 }
   };
   bitboard_t own_movers;
   bitboard_t own;
   bitboard_t enemy;
   bitboard_t enemy_attacks;
   bitboard_t occupied;
   bitboard_t pawns;
   bitboard_t enemy_pawns;
   bitboard_t knights;
   bitboard_t bishops;
   bitboard_t queens;
   bitboard_t rooks;
   bitboard_t king;
   bitboard_t double_pawns;
   bitboard_t promote_pawns;
   bitboard_t en_passant;
   bitboard_t castle_row;
   bitboard_t bb;
   bitboard_t bp;
   bitboard_t moves;
   int square;
   int piece;
   int c_index = (player != 0);
   move_t *move;

   /* Bookkeeping: we keep a pointer to the next move in the move list, and
    * update the number of moves in the list at the end of this function
    */
   move = &(movelist->move[movelist->num_moves]);

   /* Initilise appropriate bitboards */
   en_passant = board_ep & make_bitboard_square(board->ep_square);
   en_passant &= destination_mask;

   own = board->bbc[c_index];
   enemy = board->bbc[1-c_index];
   enemy_attacks = 0;

   occupied = or_bitboards(own, enemy);
   destination_mask &= ~own;

   /* Now apply bitmasks: we don't generate moves for pieces that are not
    * on the source mask.
    */
   own_movers = and_bitboards(own, source_mask);

   __builtin_prefetch(attack_row);
   __builtin_prefetch(attack_file);
   __builtin_prefetch(attack_diagonal_a8h1);
   __builtin_prefetch(attack_diagonal_a1h8);
   __builtin_prefetch(king_attack);
   __builtin_prefetch(knight_attack);

   /* Bitboards representing the different piece types */
   enemy_pawns = and_bitboards(enemy, board->bbp[PAWN]);
   pawns = and_bitboards(own_movers, board->bbp[PAWN]);
   queens = and_bitboards(own_movers, board->bbp[QUEEN]);
   rooks = and_bitboards(own_movers, board->bbp[ROOK]);
   bishops = and_bitboards(own_movers,board->bbp[BISHOP]);
   knights = and_bitboards(own_movers, board->bbp[KNIGHT]);
   king = and_bitboards(own_movers, board->bbp[KING]);

   /* Return if we can't generate any moves with these masks */
   if ((own_movers == 0) || (enemy==0))
      return;
   
   /* Generate rook moves */
   bb = rooks;
   piece = piece_for_side(ROOK, player);
   while(bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);

      moves = get_orth_attack_bitboard(occupied, square) & destination_mask;

      move = serialise_normal_moves (move, moves & ~occupied,        piece, square);
      move = serialise_capture_moves(move, moves &  occupied, board, piece, square);
   }

   /* Queen moves */
   bb = queens;
   piece = piece_for_side(QUEEN, player);
   while(bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);

      moves = (get_orth_attack_bitboard(occupied, square) | get_diag_attack_bitboard(occupied, square)) & destination_mask;

      move = serialise_normal_moves (move, moves & ~occupied,        piece, square);
      move = serialise_capture_moves(move, moves &  occupied, board, piece, square);
   }

   /* Generate bishop moves */
   bb = bishops;
   piece = piece_for_side(BISHOP, player);
   while(bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);

      moves = get_diag_attack_bitboard(occupied, square) & destination_mask;

      move = serialise_normal_moves (move, moves & ~occupied,        piece, square);
      move = serialise_capture_moves(move, moves &  occupied, board, piece, square);
   }

   /* Generate knight moves */
   bb = knights;
   piece = piece_for_side(KNIGHT, player);
   while(bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      
      moves = and_bitboards(knight_attack[square], destination_mask);

      move = serialise_normal_moves (move, moves & ~occupied,        piece, square);
      move = serialise_capture_moves(move, moves &  occupied, board, piece, square);
   }

   /* Generate pawn moves. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators - so we
    * use a macro to generate these.
    */
   if (player==WHITE) {
      enemy_attacks |= shift_bitboard_right_down(enemy_pawns);
      enemy_attacks |= shift_bitboard_left_down(enemy_pawns);
      generate_pawn_moves(shift_bitboard_row_up,
                          shift_bitboard_right_up,
                          shift_bitboard_left_up, 
                          -,9,7,board_rank3,board_rank8);
      castle_row = board_rank1;
   } else {
      enemy_attacks |= shift_bitboard_right_up(enemy_pawns);
      enemy_attacks |= shift_bitboard_left_up(enemy_pawns);
      generate_pawn_moves(shift_bitboard_row_down,
                          shift_bitboard_right_down,
                          shift_bitboard_left_down,
                          +, 7, 9, board_rank6, board_rank1);
      castle_row = board_rank8;
   }

   /* Generate (legal) king moves.
    * It seems that the if statement should not be necessary because there
    * is always a king on the board, but this is false: the king may be
    * outside of the origin mask and therefore be "invisible"
    */
   if (king) {
      square = bitscan64(king);
      piece = piece_for_side(KING, player);

      moves = mask_bitboards(king_attack[square], enemy_attacks);

      /* Here, we could already mask out squares attacked by the enemy
       * king, pawns and knights. Sliders are a bit more tricky though.
       */

      /* "moves" now has all possible moves for this piece encoded, but we
       * should only generate the ones that satisfy the destination mask.
       */
      moves = and_bitboards(moves, destination_mask);

      move = serialise_normal_moves (move, moves & ~occupied,        piece, square);
      move = serialise_capture_moves(move, moves &  occupied, board, piece, square);

      if (king & (E1_MASK | E8_MASK) & board->init) {
         /* Castling, only if the king is on the e-file */
         rooks = and_bitboards(own, board->bbp[ROOK]);

         /* Castling, king side */
         if (rooks&(H1_MASK | H8_MASK)) {
         //if (rooks&(H1_MASK | H8_MASK) && destination_mask & (king << 2)) {
            bp = and_bitboards(castle_row, board_castleki);
            bb = and_bitboards(bp, ~occupied);
            if (bb && (bb == bp)) {          /* No pieces between king&rook */
               bb = and_bitboards(castle_mask[0][c_index], board->init);
               if (bb == castle_mask[0][c_index]) {
                  uint8_t from = bitscan64(king);
                  /* make sure the King does not pass through check */
                  if (!square_is_attacked(board, board_all, from+1, player^BLACK)&&
                        !square_is_attacked(board, board_all, from, player^BLACK)&&
                        !square_is_attacked(board, board_all, from+2, player^BLACK)) {
                     *move=encode_castling(piece_for_side(KING, player), from, from+2, castlek_rook_bits);
                     move++;
                  }
               }
            }
         }

         /* Castling, queen side */
         //if (rooks&(A1_MASK | A8_MASK) && destination_mask & (king >> 2)) {
         if (rooks&(A1_MASK | A8_MASK)) {
            bp = and_bitboards(castle_row, board_castleqi);
            bb = and_bitboards(bp, ~occupied);
            if (bb && (bb == bp)) {          /* No pieces between king&rook */
               bb = and_bitboards(castle_mask[1][c_index], board->init);
               if (bb == castle_mask[1][c_index]) {
                  uint8_t from = bitscan64(king);
                  /* make sure the King does not pass through check */
                  if (!square_is_attacked(board, board_all, from-1, player^BLACK)&&
                        !square_is_attacked(board, board_all, from, player^BLACK)&&
                        !square_is_attacked(board, board_all, from-2, player^BLACK)) {
                     *move=encode_castling(piece_for_side(KING, player), from, from-2, castleq_rook_bits);
                     move++;
                  }
               }
            }
         }
      }
   }

   /* Store the number of moves currently in the list */
   movelist->num_moves = (move - &(movelist->move[0]));

   /* Sanity check: make sure that the target square is valid.
    * If it isn't this indicates a memory overwrite somewhere in the code
    * that invalidates the move generation tables.
    */
#ifdef DEBUGMODE
   int n;
   for (n=0; n<movelist->num_moves; n++) {
      move_t move = movelist->move[n];
      assert(super_attack[get_move_origin(move)] & make_bitboard_square(get_move_destination(move)));
   }
#endif
}

/* Generate all check evasion moves */
static void inline generate_check_evasion_moves(movelist_t *movelist, const board_t *board, int side_to_move)
{
   bitboard_t destination_mask = board_all;
   bitboard_t source_mask = board_all;
   bitboard_t attackers;
   bitboard_t bp;
   int king_square;

   king_square = bitscan64(and_bitboards(board->bbc[side_to_move>>7], board->bbp[KING]));

   /* Get a list of all attackers; generate only legal moves out
    * of check
    */
   attackers = get_attackers(board, side_to_move^BLACK, king_square);
   assert(attackers != 0);

   /* Step 1.
    * Generate legal evasions.
    */
   source_mask = board->bbp[KING];
   destination_mask = king_attack[king_square];
   /* Mask attack rays from checking piece */
   bp = attackers;
   while (bp) {
      int square = bitscan64(bp);
      bp ^= make_bitboard_square(square);
      if (board->bbp[KNIGHT] & make_bitboard_square(square)) { /* Knight */
         destination_mask &= ~knight_attack[square];
      } else if (board->bbp[BISHOP] & make_bitboard_square(square)) { /* Bishop */
         /* Mask entire ray */
         bitboard_t mask;
         if (get_a1h8_diagonal(king_square) == get_a1h8_diagonal(square)) {
            mask = board_a1h8_diagonal[get_a1h8_diagonal(king_square)];
         } else {
            mask = board_a8h1_diagonal[get_a8h1_diagonal(king_square)];
         }
         destination_mask &= ~(mask & ~make_bitboard_square(square));
      } else if (board->bbp[ROOK] & make_bitboard_square(square)) { /* Rook */
         bitboard_t mask;
         if (unpack_rank(king_square) == unpack_rank(square)) {
            mask = board_rank[unpack_rank(king_square)];
         } else {
            mask = board_file[unpack_file(king_square)];
         }
         destination_mask &= ~(mask & ~make_bitboard_square(square));
      } else if (board->bbp[QUEEN] & make_bitboard_square(square)) { /* Queen */
         bitboard_t mask;
         if (unpack_rank(king_square) == unpack_rank(square)) {
            mask = board_rank[unpack_rank(king_square)];
         } else if (unpack_file(king_square) == unpack_file(square)) {
            mask = board_file[unpack_file(king_square)];
         } else if (get_a1h8_diagonal(king_square) == get_a1h8_diagonal(square)) {
            mask = board_a1h8_diagonal[get_a1h8_diagonal(king_square)];
         } else {
            mask = board_a8h1_diagonal[get_a8h1_diagonal(king_square)];
         }
         destination_mask &= ~(mask & ~make_bitboard_square(square));
      }
   }
   /* Mask attacks to other possible destinations */
   bp = destination_mask;
   while (bp) {
      int square = bitscan64(bp);
      bp ^= make_bitboard_square(square);
      if (square_is_attacked(board, board_all, square, side_to_move^BLACK))
         destination_mask ^= make_bitboard_square(square);
   }
   /* Generate king moves */
   generate_moves_mask(movelist, board, source_mask, destination_mask, side_to_move, ANY_PROMOTION);

   /* Step 2.
    * Generate interpositions and captures of the checking piece if there
    * is only one.
    */
   if (onebit64(attackers)) {
      int attacker_square = bitscan64(attackers);

      source_mask = board->bbc[side_to_move>>7] ^ make_bitboard_square(king_square);
      destination_mask = get_connecting_ray(king_square, attacker_square) | attackers;

      source_mask &= ~get_pinned_pieces(board, board_all, side_to_move>>7, king_square);

      generate_moves_mask(movelist, board, source_mask, destination_mask, side_to_move, ANY_PROMOTION);

      /* En-passant captures of checking pawns */
      if (board->ep_square && board->bbp[PAWN]&attackers &&
          unpack_file(attacker_square) == unpack_file(board->ep_square)) {
         generate_moves_mask(movelist, board,
                             source_mask&board->bbp[PAWN], make_bitboard_square(board->ep_square),
                             side_to_move, ANY_PROMOTION);
      }
   }
}

/* Generate only checking moves */
void generate_checking_moves(movelist_t *movelist, const board_t *board, int side_to_move, bool include_captures)
{
   bitboard_t destination_mask = board_all;
   bitboard_t source_mask = board_all;
   bitboard_t own_movers;
   bitboard_t mask;
   int king_square;

   /* Avoid this function if we're in check; we should consider all
    * evasions there.
    */
   assert(!board->in_check);

   if (!include_captures)
      destination_mask = ~board->bbc[1-(side_to_move>>7)];

   king_square = bitscan64(and_bitboards(board->bbc[1-(side_to_move>>7)], board->bbp[KING]));
   own_movers = board->bbc[side_to_move>>7];

   movelist->num_moves = 0;
   movelist->cur_move = 0;

   /* Direct checks - knight */
   generate_moves_mask(movelist, board,
                       board->bbp[KNIGHT]&source_mask, knight_attack[king_square]&destination_mask,
                       side_to_move, ANY_PROMOTION);

   /* Direct checks - rook/queen */
   bitboard_t rooks_or_queens = and_bitboards(own_movers, or_bitboards(board->bbp[ROOK], board->bbp[QUEEN]));
   mask = get_rook_attack_bitboard(board, king_square);
   generate_moves_mask(movelist, board,
                       rooks_or_queens&source_mask, mask&destination_mask,
                       side_to_move, ANY_PROMOTION);

   /* Direct checks - bishop/queen */
   bitboard_t bishops_or_queens = and_bitboards(own_movers, or_bitboards(board->bbp[BISHOP], board->bbp[QUEEN]));
   mask = get_bishop_attack_bitboard(board, king_square);
   generate_moves_mask(movelist, board,
                       bishops_or_queens&source_mask, mask&destination_mask,
                       side_to_move, ANY_PROMOTION);

   /* Direct checks - pawn */
   /* TODO */

   /* Discovered checks */
   bitboard_t occ = get_occupied(board);
   bitboard_t pinned = get_orth_attack_bitboard(occ, king_square) | get_diag_attack_bitboard(occ, king_square);
   pinned &= own_movers & source_mask;
   while (pinned) {
      int square = bitscan64(pinned);
      pinned ^= make_bitboard_square(square);
      if (get_orth_attack_bitboard(occ ^ make_bitboard_square(square), king_square) & rooks_or_queens) {
         generate_moves_mask(movelist, board,
                             make_bitboard_square(square), destination_mask & ~(orth[king_square]&orth[square]),
                             side_to_move, ANY_PROMOTION);
      }
      if (get_diag_attack_bitboard(occ ^ make_bitboard_square(square), king_square) & bishops_or_queens) {
         generate_moves_mask(movelist, board,
                             make_bitboard_square(square), destination_mask & ~(diag[king_square]&diag[square]),
                             side_to_move, ANY_PROMOTION);
      }
   }
   /* TODO */
}

static void generate_legal_moves_mask(movelist_t *movelist, const board_t *board, bitboard_t source_mask, bitboard_t destination_mask, int colour, int allowed_promotion_pieces)
{
   board_t b = *board;
   b.ep_square = 0;
   int side_to_move = (colour >> 7);
   bitboard_t king  = b.bbp[KING] & b.bbc[side_to_move];
   int king_square  = bitscan64(king);

   /* Get pinned pieces */
   bitboard_t pinned = get_pinned_pieces(&b, board_all, side_to_move, king_square) & ~king;

   /* Avoid pinned pieces and king moves for the first batch */
   generate_moves_mask(movelist, &b, (board_all^(pinned|king))&source_mask, destination_mask, colour, allowed_promotion_pieces);

   /* King moves to non-attacked squares (and possibly castling moves) */
   bitboard_t mask = king_attack[king_square] & ~b.bbc[side_to_move] & destination_mask;
   bitboard_t bp = mask;
   while (bp) {
      int square = bitscan64(bp);
      bp ^= make_bitboard_square(square);
      if (square_is_attacked(&b, board_all, square, colour^BLACK))
         mask ^= make_bitboard_square(square);
   }
   if (king & b.init) mask |= side_to_move ? (G8_MASK | C8_MASK) : (G1_MASK | C1_MASK);
   generate_moves_mask(movelist, &b, king & source_mask, mask, colour, allowed_promotion_pieces);

   /* Generate moves for pinned pieces: they can only move along the ray
    * connecting them to the king.
    */
   bitboard_t p = pinned;
   while (p) {
      int psq = bitscan64(p);
      p ^= make_bitboard_square(psq);

      if (orth[psq] & king) {
         mask = orth[psq] & orth[king_square];
      } else {
         assert(diag[psq] & king);
         mask = diag[psq] & diag[king_square];
      }
      generate_moves_mask(movelist, &b, make_bitboard_square(psq) & source_mask, mask, colour, allowed_promotion_pieces);
   }

   /* Generate en-passant captures (if legal) */
   /* Special: pinned en-passant capture pawn */
   if (__builtin_expect(board->ep_square, false)) {
      int ep_capture_square;
      if (board->ep_square > A4)
         ep_capture_square =board->ep_square - 8;
      else
         ep_capture_square =board->ep_square + 8;

      if (unpack_rank(king_square) == unpack_rank(ep_capture_square)) {
         bitboard_t occ = get_occupied(board);
         bitboard_t xa = get_rank_xattacks(occ, king_square);
         occ ^= make_bitboard_square(ep_capture_square);
         xa ^= get_rank_xattacks(occ, king_square);

         if ((xa & get_rook_movers(board) & board->bbc[side_to_move^0x01]) == board_empty) {
            generate_moves_mask(movelist, board, board->bbp[PAWN], make_bitboard_square(board->ep_square), colour, allowed_promotion_pieces);
         }
      } else {
         generate_moves_mask(movelist, board, board->bbp[PAWN]&~pinned, make_bitboard_square(board->ep_square), colour, allowed_promotion_pieces);
      }
   }
}

/* Generate all legal moves on the board for the specified colour */
void generate_legal_moves(movelist_t *movelist, const board_t *board, int colour)
{
   movelist->num_moves = 0;
   movelist->cur_move = 0;

   if (board->in_check) {
      generate_check_evasion_moves(movelist, board, colour);
   } else {
      generate_legal_moves_mask(movelist, board, board_all, board_all, colour, ANY_PROMOTION);
   }
}

/* Generate all legal moves on the board for the specified colour */
void generate_legal_quiescence_moves(movelist_t *movelist, const board_t *board, int colour)
{
   bitboard_t destination_mask = board_all;
   bitboard_t source_mask = board_all;

   movelist->num_moves = 0;
   movelist->cur_move = 0;

   if (board->in_check) {
      generate_check_evasion_moves(movelist, board, colour);
      return;
   }

   /* Generate only captures */
   destination_mask = board->bbc[1-(colour>>7)];
   generate_legal_moves_mask(movelist, board, board_all, destination_mask, colour, QUEEN_PROMOTION);

   /* Now generate all normal (non-capture) pawn promotions */
   destination_mask ^= board_rank1 | board_rank8;
   source_mask = board->bbc[colour>>7] & board->bbp[PAWN];

   if (source_mask && destination_mask)
      generate_legal_moves_mask(movelist, board, source_mask, destination_mask, colour, QUEEN_PROMOTION);
}

/* Generate all pseudo-legal moves on the board for the specified colour */
void generate_moves(movelist_t *movelist, const board_t *board, int colour)
{
#ifdef WRITE_OUTPUT_FROM_MOVE_GENERATOR
   printf("Generating moves for player %d\n", colour);
#endif
   movelist->num_moves = 0;
   movelist->cur_move = 0;

   /* If the king is in check, then only moves that end in the region of
    * allowed moves for a "super piece" at the location of the king can be
    * valid. A "super piece" is a piece that moves as a knight or a queen.
    * The idea behind this is that we only need to consider moves that can
    * break the check, all other moves will be discarded anyway so why
    * generate them?
    * We don't actually have to test for this here in order to play legal
    * chess, the search algorithm will automatically cull moves that leave
    * the king in check, but this will avoid some unnecessary evaluations.
    * Note that the search function tests whether the king is in check at
    * the beginning of the move anyway in order to decide whether to tag on
    * a search extension anyway, so it doesn't cost us extra to do the test
    * here and store the result.
    */
   if (board->in_check) {
      generate_check_evasion_moves(movelist, board, colour);
   } else {
      generate_moves_mask(movelist, board, board_all, board_all, colour, ANY_PROMOTION);
   }
#ifdef WRITE_OUTPUT_FROM_MOVE_GENERATOR
   printf("Generated %d moves\n", movelist->num_moves);

   int n;
   for (n=0; n<movelist->num_moves; n++) {
      printf("%s ", move_string(movelist->move[n], NULL));
   }
   printf("\n");
#endif
}

/* Test whether a particular move is legal */
bool check_move_exists(move_t move, const board_t *board)
{
   movelist_t movelist;
   bitboard_t orig, dest;
   int colour = get_move_player(move);
   int n;

   movelist.num_moves = 0;

   orig = make_bitboard_square(get_move_origin(move));
   dest = make_bitboard_square(get_move_destination(move)) | make_bitboard_square(get_capture_square(move));

   generate_moves_mask(&movelist, board, orig, dest, colour, QUEEN_PROMOTION);

   for (n=0; n<movelist.num_moves; n++)
      if (movelist.move[n] == move)
         return true;
   return false;
}

/* Only generate tactical moves: captures and pawn promotions */
void generate_tactical_moves(movelist_t *movelist, const board_t *board, int colour)
{
   bitboard_t destination_mask = board_all;
   bitboard_t source_mask = board_all;

#ifdef WRITE_OUTPUT_FROM_MOVE_GENERATOR
   printf("Generating moves for player %d\n", colour);
#endif

   /* Generate only captures */
   destination_mask = board->bbc[1-(colour>>7)];
   generate_moves_mask(movelist, board, board_all, destination_mask, colour, ANY_PROMOTION);

   /* Now generate all normal (non-capture) pawn promotions */
   destination_mask ^= board_rank1 | board_rank8;
   source_mask = and_bitboards(board->bbc[colour>>7], board->bbp[PAWN]);

   if (source_mask && destination_mask)
      generate_moves_mask(movelist, board, source_mask, destination_mask, colour, ANY_PROMOTION);

   movelist->num_moves = 0;
   movelist->cur_move = 0;

#ifdef WRITE_OUTPUT_FROM_MOVE_GENERATOR
   printf_bitboard(destination_mask);
   printf("Generated %d moves\n", movelist->num_moves);

   int n;
   for (n=0; n<movelist->num_moves; n++) {
      printf("%s ", move_string(movelist->move[n], NULL));
   }
   printf("\n");
#endif
}

/* Generate moves for quiescence search. Generally, this is restricted to
 * captures, check evasions and queen promotions.
 */
void generate_quiescence_moves(movelist_t *movelist, const board_t *board, int colour)
{
   bitboard_t destination_mask = board_all;
   bitboard_t source_mask = board_all;

#ifdef WRITE_OUTPUT_FROM_MOVE_GENERATOR
   printf("Generating moves for player %d\n", colour);
#endif

   /* Generate all moves by setting the "source" and "destination" masks to
    * the full board.
    */
   movelist->num_moves = 0;
   movelist->cur_move = 0;

   /* If the king is in check, then only moves that end in the region of
    * allowed moves for a "super piece" at the location of the king can be
    * valid.
    */
   if (board->in_check) {
      generate_check_evasion_moves(movelist, board, colour);
      return;
   } else {
      /* Generate only captures */
      destination_mask = board->bbc[1-(colour>>7)];
      generate_moves_mask(movelist, board, board_all, destination_mask, colour, QUEEN_PROMOTION);

      /* Now generate all normal (non-capture) pawn promotions */
      destination_mask ^= board_rank1 | board_rank8;
      source_mask = board->bbc[colour>>7] & board->bbp[PAWN];
   }

   if (source_mask && destination_mask)
      generate_moves_mask(movelist, board, source_mask, destination_mask, colour, QUEEN_PROMOTION);

#ifdef WRITE_OUTPUT_FROM_MOVE_GENERATOR
   printf_bitboard(destination_mask);
   printf("Generated %d moves\n", movelist->num_moves);

   int n;
   for (n=0; n<movelist->num_moves; n++) {
      printf("%s ", move_string(movelist->move[n], NULL));
   }
   printf("\n");
#endif
}


/* Return the bitboard of attacked squares for the given side.
 * FIXME: if we need this information regularly: it is easy to get for the
 * side that is to move as a side-effect of the move generation...
 */
bitboard_t get_attack_bitboard(const board_t *board, int side)
{
   bitboard_t attackers;
   bitboard_t own;
   bitboard_t enemy;
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops_or_queens;
   bitboard_t rooks_or_queens;
   bitboard_t king;
   bitboard_t bb;
   bitboard_t occ;
   int square;
   int c_index = (side != 0);

   own = board->bbc[c_index];
   enemy = board->bbc[1-c_index];
   occ = own | enemy;

   /* Bitboards representing the different piece types */
   pawns = and_bitboards(own, board->bbp[PAWN]);
   rooks_or_queens = and_bitboards(own,
                                    or_bitboards(board->bbp[ROOK], board->bbp[QUEEN]));
   bishops_or_queens = and_bitboards(own,
                                    or_bitboards(board->bbp[BISHOP], board->bbp[QUEEN]));
   knights = and_bitboards(own, board->bbp[KNIGHT]);
   king = and_bitboards(own, board->bbp[KING]);

   attackers = 0;

   /* Generate rook moves and horizontal/vertical queen moves */
   bb = rooks_or_queens;
   while(bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      attackers |= get_orth_attack_bitboard(occ, square);
   }

   /* Generate bishop moves and diagonal queen moves */
   bb = bishops_or_queens;
   while(bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      attackers |= get_diag_attack_bitboard(occ, square);
   }

   /* Generate knight moves */
   bb = knights;
   while(bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      
      attackers |= knight_attack[square];
   }

   /* Generate king moves */
   bb = king;
   while(bb) {
      square = bitscan64(bb);
      bb ^= make_bitboard_square(square);
      
      attackers |= king_attack[square];
   }

   /* Generate pawn attacks. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators.
    */
   if (side==WHITE) {
      attackers |= shift_bitboard_right_up(pawns);
      attackers |= shift_bitboard_left_up(pawns);
   } else {
      attackers |= shift_bitboard_right_down(pawns);
      attackers |= shift_bitboard_left_down(pawns);
   }

   return attackers;
}

/* List all attackers for side for a given square */
void list_attackers(board_t *board, int player, int square)
{
   bitboard_t bb_where = make_bitboard_square(square);
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops;
   bitboard_t queens;
   bitboard_t rooks;
   bitboard_t king;
   bitboard_t bb;
   bitboard_t moves;
   bitboard_t attackers;
   bitboard_t own_movers;
   int c_index = (player != 0);
   uint8_t row_nr, col_nr, a8h1_nr, a1h8_nr;

   /* Initilize appropriate bitboards */
   own_movers = board->bbc[c_index];

   /* Bitboards representing the different piece types */
   pawns = and_bitboards(own_movers, board->bbp[PAWN]);
   queens = and_bitboards(own_movers, board->bbp[QUEEN]);
   bishops = and_bitboards(own_movers, board->bbp[BISHOP]);
   rooks = and_bitboards(own_movers, board->bbp[ROOK]);
   knights = and_bitboards(own_movers, board->bbp[KNIGHT]);
   king = and_bitboards(own_movers, board->bbp[KING]);

   /* Pieces that can attack this square are pieces that can be captured by
    * a super piece -> generate all moves for a super piece on this square
    */
   row_nr = (get_row_occupancy_number(board, square) >> 1) & 63;
   col_nr = (get_file_occupancy_number(board,square) >> 1) & 63;
   a8h1_nr = (get_diagonal_a8h1_occupancy_number(board, square) >> 1) & 63;
   a1h8_nr = (get_diagonal_a1h8_occupancy_number(board, square) >> 1) & 63;

   /* attacking knights */
   attackers = and_bitboards(knights, knight_attack[square]);
   while(attackers) {
      int where = bitscan64(attackers);

      printf("N%s ", square_str[where]);
      unset_bitboard(&attackers, where);
   }

   /* Attacking kings */
   attackers = and_bitboards(king, king_attack[square]);
   while(attackers) {
      int where = bitscan64(attackers);

      printf("K%s ", square_str[where]);
      unset_bitboard(&attackers, where);
   }

   /* Attacking rooks */
   moves = or_bitboards(attack_row[square][row_nr],
                        attack_file[square][col_nr]);
   attackers = and_bitboards(moves, rooks);
   while(attackers) {
      int where = bitscan64(attackers);

      printf("R%s ", square_str[where]);
      unset_bitboard(&attackers, where);
   }

   /* Check bishop moves */
   moves = or_bitboards(attack_diagonal_a8h1[square][a8h1_nr],
                        attack_diagonal_a1h8[square][a1h8_nr]);
   attackers = and_bitboards(moves, bishops);
   while(attackers) {
      int where = bitscan64(attackers);

      printf("R%s ", square_str[where]);
      unset_bitboard(&attackers, where);
   }

   /* Queen */
   moves = or_bitboards( or_bitboards(attack_row[square][row_nr],
                                      attack_file[square][col_nr]),
                         or_bitboards(attack_diagonal_a8h1[square][a8h1_nr],
                                      attack_diagonal_a1h8[square][a1h8_nr]) );
   attackers = and_bitboards(moves, queens);
   while(attackers) {
      int where = bitscan64(attackers);

      printf("Q%s ", square_str[where]);
      unset_bitboard(&attackers, where);
   }

   /* Finally check for pawns. */
   /* Generate pawn moves. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators - so we
    * use a macro to generate these.
    */
   if (player==WHITE) {
      bb = and_bitboards(bb_where, shift_bitboard_right_up(pawns));
      if (bb) printf("P%s ", square_str[bitscan64(bb)]);
      bb = and_bitboards(bb_where, shift_bitboard_left_up(pawns));
      if (bb) printf("P%s ", square_str[bitscan64(bb)]);
   } else {
      bb = and_bitboards(bb_where, shift_bitboard_right_down(pawns));
      if (bb) printf("P%s ", square_str[bitscan64(bb)]);
      bb = and_bitboards(bb_where, shift_bitboard_left_down(pawns));
      if (bb) printf("P%s ", square_str[bitscan64(bb)]);
   }
}

/* get all attackers for the target square */
bitboard_t get_attackers(const board_t *board, int player, int square)
{
   bitboard_t bb_where = make_bitboard_square(square);
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops;
   bitboard_t queens;
   bitboard_t rooks;
   bitboard_t king;
   bitboard_t attackers;
   bitboard_t own_movers;
   bitboard_t occ;
   int c_index = (player != 0);

   /* Initilize appropriate bitboards */
   own_movers = board->bbc[c_index];

   /* Bitboards representing the different piece types */
   pawns = and_bitboards(own_movers, board->bbp[PAWN]);
   queens = and_bitboards(own_movers, board->bbp[QUEEN]);
   bishops = and_bitboards(own_movers, board->bbp[BISHOP]);
   rooks = and_bitboards(own_movers, board->bbp[ROOK]);
   knights = and_bitboards(own_movers, board->bbp[KNIGHT]);
   king = and_bitboards(own_movers, board->bbp[KING]);
   occ = get_occupied(board);

   /* attacking knights */
   attackers = and_bitboards(knights, knight_attack[square]);

   /* Attacking kings */
   attackers |= and_bitboards(king, king_attack[square]);

   /* Attacking rooks and queens */
   attackers |= and_bitboards(get_orth_attack_bitboard(occ, square), rooks|queens);

   /* Check bishop moves */
   attackers |= and_bitboards(get_diag_attack_bitboard(occ, square), bishops|queens);

   /* Finally check for pawns. */
   /* Generate pawn moves. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators - so we
    * use a macro to generate these.
    */
   if (player==WHITE) {
      bitboard_t attacking_pawns = shift_bitboard_right_down(bb_where)|
                                   shift_bitboard_left_down(bb_where);
      attackers |= pawns&attacking_pawns;
   } else {
      bitboard_t attacking_pawns = shift_bitboard_right_up(bb_where)|
                                   shift_bitboard_left_up(bb_where);
      attackers |= pawns&attacking_pawns;
   }

   return attackers;
}

/* get all attackers for the target square, from either side */
bitboard_t get_all_attackers(const board_t *board, bitboard_t mask, int square)
{
   bitboard_t bb_where = make_bitboard_square(square);
   bitboard_t pawns;
   bitboard_t knights;
   bitboard_t bishops;
   bitboard_t queens;
   bitboard_t rooks;
   bitboard_t king;
   bitboard_t moves;
   bitboard_t attackers;
   bitboard_t occ;

   /* Initilise appropriate bitboards */
   /* Bitboards representing the different piece types */
   pawns = board->bbp[PAWN];
   queens = board->bbp[QUEEN];
   bishops = board->bbp[BISHOP];
   rooks = board->bbp[ROOK];
   knights = board->bbp[KNIGHT];
   king = board->bbp[KING];
   occ = get_occupied(board);

   /* attacking knights */
   attackers = and_bitboards(knights, knight_attack[square]);

   /* Attacking kings */
   attackers |= and_bitboards(king, king_attack[square]);

   /* Attacking rooks and queens */
   moves = get_orth_attack_bitboard(occ & mask, square);
   attackers |= and_bitboards(moves, rooks|queens);

   /* Check bishop moves */
   moves = get_diag_attack_bitboard(occ & mask, square);
   attackers |= and_bitboards(moves, bishops|queens);

   /* Finally check for pawns. */
   /* Generate pawn moves. These do depend on the current player, but the
    * code is identical apart from a few sign and shift operators - so we
    * use a macro to generate these.
    */
   attackers |= shift_bitboard_right_up(bb_where)&pawns&board->bbc[1];
   attackers |= shift_bitboard_left_up(bb_where)&pawns&board->bbc[1];
   attackers |= shift_bitboard_right_down(bb_where)&pawns&board->bbc[0];
   attackers |= shift_bitboard_left_down(bb_where)&pawns&board->bbc[0];

   return attackers & mask;
}

/* Find pinned pieces for a given side (0 or 1). The square is the shielded
 * square (typically the location of the king).
 * The mask is so we can exclude pins by certain piece types: a queen
 * cannot pin a piece to another queen if that queen is defended, for
 * instance.
 */
bitboard_t get_pinned_pieces(const board_t *board, bitboard_t mask, int side, int square)
{
   bitboard_t pinned, p, a, xa;

   pinned = board_empty;

   /* Pins by a rook: see if a rook on the square would perform an X-ray
    * attack on an enemy rook. (or queen)
    */
   p = get_rook_movers(board) & board->bbc[1-side] & mask;
   get_rook_attacks(board, square, &a, &xa);
   xa &= ~make_bitboard_square(square);
   p &= xa;
   while (p) {
      int pinning_square = bitscan64(p);
      p ^= make_bitboard_square(pinning_square);
      pinned |= board->bbc[side] & get_connecting_ray(square, pinning_square);
   }

   /* Pins by a bishop: see if a bishop on the square would perform an X-ray
    * attack on an enemy bishop (or queen).
    */
   p = get_bishop_movers(board) & board->bbc[1-side] & mask;
   get_bishop_attacks(board, square, &a, &xa);
   xa &= ~make_bitboard_square(square);
   p &= xa;
   while (p) {
      int pinning_square = bitscan64(p);
      p ^= make_bitboard_square(pinning_square);
      pinned |= board->bbc[side] & get_connecting_ray(square, pinning_square);
   }

   //print_bitboard_squares(pinned);
   //printf("\n");
   return pinned;
}

/* Returns the index in the movelist of a valid move corresponding to the
 * given origin/destination. Intended for player interaction.
 * Returns -1 if no moves in the list match (ie, the requested move is
 * invalid). In the case of a promotion, there will be more than one move
 * that matches.
 */
int validate_move(const movelist_t *movelist, int source, int dest)
{
   int n;
   if (!movelist)
      return -1;

   for (n=0; n<movelist->num_moves; n++) {
      if (get_move_origin(movelist->move[n]) == source &&
          get_move_destination(movelist->move[n]) == dest) {
         return n;
      }
   }
   return -1;
}

void print_movelist(const movelist_t *movelist)
{
   int n;

   for (n=0; n<movelist->num_moves; n++) {
      printf("%s ", move_string(movelist->move[n], NULL));
   }
   printf("\n");
}
