/*  Jazz, a program for playing chess
 *  Copyright (C) 2009, 2011  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <ctype.h>
#include "move.h"
#include "names.h"
#include "pieces.h"
#include "squares.h"
#include "square.h"
#include "movestring.h"

static const char *file_names[8] = { "a", "b", "c", "d", "e", "f", "g", "h" };
static const char *row_names[8] = { "1", "2", "3", "4", "5", "6", "7", "8" };
static char static_output_buffer[16];

static char *trim(char *s)
{
   char *p = s;
   while (*p && isspace(*p)) p++;
   if (p!= s) {
      for (int n = 0; n<=strlen(p); n++)
         s[n] = p[n];
   }
   return s;
}

char *short_move_string(const move_t move, const movelist_t *movelist, char *buffer)
{
   char *s = buffer;
   const char *piece = "";
   const char *suffix = "";
   const char *token = "";
   const char *origin = "";
   int p = get_move_piece(move)&PIECE;

   if (!s) s = static_output_buffer;

   if (move == 0) {
      snprintf(s, 15, "...");
      return s;
   }

   if (is_castle_move(move)) {
      if (get_castle_flags(move) & 1)
         snprintf(s, 15, "O-O-O");
      else
         snprintf(s, 15, "O-O");
      return s;
   }

   if (is_capture_move(move)) token = "x";
   if (is_capture_move(move) && is_enpassant_move(move)) suffix = "ep";
   if (is_promotion_move(move)) suffix = piece_str[get_promotion_piece(move)&PIECE];
   if (p != PAWN) piece = piece_str[p];

   /* Slightly special case: pawn capture */
   if (p == PAWN && is_capture_move(move)) {
      origin = file_names[unpack_file(get_move_origin(move))];
   } else if (p && movelist) {
      /* The information we have now might be ambiguous - check */
      int count = 0;
      int n;
      for (n=0; n<movelist->num_moves; n++) {
         if (get_move_piece(move) == get_move_piece(movelist->move[n]) &&
             get_move_destination(move) == get_move_destination(movelist->move[n])) count++;
      }
      if (count <= 1) goto disambiguous;

      /* Try to disambiguate by file */
      count = 0;
      for (n=0; n<movelist->num_moves; n++) {
         if (get_move_piece(move) == get_move_piece(movelist->move[n]) &&
             get_move_destination(move) == get_move_destination(movelist->move[n]) &&
             unpack_file(get_move_origin(move)) == unpack_file(get_move_origin(movelist->move[n])))
            count++;
      }
      if (count == 1) {
         origin = file_names[unpack_file(get_move_origin(move))];
      } else {
         /* Disambiguate by row */
         origin = row_names[unpack_rank(get_move_origin(move))];
      }
   }
disambiguous:

   snprintf(s, 15, "%s%s%s%s%s", piece,
            origin, token, square_str[get_move_destination(move)],
            suffix);

   return s;
}

char *move_string(const move_t move, char *buffer)
{
   char *s = buffer;
   const char *piece = "";
   const char *suffix = "";
   const char *token = "-";
   int p = get_move_piece(move)&PIECE;

   if (!s) s = static_output_buffer;

   if (move == 0) {
      snprintf(s, 15, "...");
      return s;
   }

   if (is_castle_move(move)) {
      if (get_castle_flags(move) & 1)
         snprintf(s, 15, "O-O-O");
      else
         snprintf(s, 15, "O-O");
      return s;
   }

   if (is_capture_move(move)) token = "x";
   if (is_capture_move(move) && is_enpassant_move(move)) suffix = "ep";
   if (is_promotion_move(move)) suffix = piece_str[get_promotion_piece(move)&PIECE];
   if (p != PAWN) piece = piece_str[p];

   snprintf(s, 15, "%s%2s%s%2s%s", piece,
            square_str[get_move_origin(move)], token, square_str[get_move_destination(move)],
            suffix);

   return s;
}

const char *lan_move_string(const move_t move, char *buffer)
{
   char *s = buffer;
   char suffix = '\0';

   if (!s) s = static_output_buffer;

   if (move == 0) {
      snprintf(s, 15, "(pass)");
      return s;
   }

   if (is_promotion_move(move)) suffix = tolower(*piece_str[get_promotion_piece(move)&PIECE]);

   snprintf(s, 15, "%2s%2s%c", square_str[get_move_origin(move)], square_str[get_move_destination(move)],
            suffix);

   return s;
}


move_t move_string_to_move(const char *move_str, const movelist_t *movelist)
{
   char ms[16];
   int n;

   while (isspace(*move_str) && *move_str) move_str++;
   snprintf(ms, sizeof(ms), "%s", move_str);
   char *s = strstr(ms, "=");
   if (s) {
      while (*s) {
         *s = *(s+1);
         s++;
      }
   }
   s = strstr(ms, " ");
   if (s) *s = '\0';
   s = strstr(ms, "+");
   if (s) *s = '\0';

   for (n = 0; n<movelist->num_moves; n++) {
      move_t move = movelist->move[n];
      char buffer[128];

      short_move_string(move, movelist, buffer);
      trim(buffer);
      if (strcmp(buffer, ms) == 0) return move;

      move_string(move, buffer);
      trim(buffer);
      if (strcmp(buffer, ms) == 0) return move;

      lan_move_string(move, buffer);
      trim(buffer);
      if (strcmp(buffer, ms) == 0) return move;
   }

   return 0;
}

