/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef BOARD_H
#define BOARD_H

#include <stdint.h>
#include "assert.h"
#include "move.h"
#include "hashkey.h"

#define expect(x,y) __builtin_expect((x), (y))

typedef struct board_t {
   bitboard_t bbp[NUM_PIECES];
   bitboard_t bbc[2];
   bitboard_t occ;
   bitboard_t init;

   uint8_t piece[64];
   uint64_t hash;
   sides side_to_move;
   uint8_t ep, ep_capture;
   int8_t fifty_counter;
   bool in_check;

   /* Holdings, can only hold 1 piece of each type */
   int16_t holdings[2];
} board_t;

extern void clear_board(board_t *board);

extern void print_bitboards(const board_t *board);

static inline void place_piece(board_t *board, int piece, int side, int square)
{
   bitboard_t bb = make_bitboard_square(square);
   board->bbc[side] |= bb;
   board->occ |= bb;
   board->bbp[piece] |= bb;
   board->init |= bb;
   board->piece[square] = piece;
   board->hash ^= piece_key[piece][side][square];
}

static inline uint8_t get_piece(const board_t *board, const int where)
{
   assert(where >= A1);
   assert(where <= H8);
   assert(board);
   assert((board->bbc[WHITE]|board->bbc[BLACK]) & make_bitboard_square(where));
   return board->piece[where];
}

static inline bool may_castle(const board_t *board, sides side)
{
   return (board->init & (short_castle_mask[side] | long_castle_mask[side])) != board_empty;
}

static inline bitboard_t get_knight_movers(const board_t *board)
{
   return board->bbp[KNIGHT] | board->bbp[WARLORD] | board->bbp[MARSHAL];
}

static inline bitboard_t get_rook_movers(const board_t *board)
{
   return board->bbp[ROOK] | board->bbp[QUEEN] | board->bbp[GENERAL] | board->bbp[MARSHAL];
}

static inline bitboard_t get_bishop_movers(const board_t *board)
{
   return board->bbp[BISHOP] | board->bbp[QUEEN] | board->bbp[WARLORD];
}

static inline bitboard_t get_royal(const board_t *board)
{
   return board->bbp[KING] | board->bbp[BASILEUS];
}

static inline bitboard_t get_sliders(const board_t *board)
{
   return board->bbp[ROOK] | board->bbp[QUEEN] | board->bbp[GENERAL] | board->bbp[BISHOP] |
          board->bbp[WARLORD] | board->bbp[MARSHAL];
}

static inline bitboard_t get_leapers(const board_t *board)
{
   return board->bbp[KNIGHT] | board->bbp[LIEUTENANT] | board->bbp[CAPTAIN] | board->bbp[MARSHAL];
}

static inline bitboard_t get_mate_potential(const board_t *board)
{
   bitboard_t bb = board->bbp[BASILEUS];
   if (bb && onebit64(bb)) bb = board_empty;
   return bb | board->bbp[ROOK] | board->bbp[QUEEN] | board->bbp[GENERAL] | board->bbp[CAPTAIN] | board->bbp[WARLORD] | board->bbp[MARSHAL];
}

static inline bitboard_t get_pawns(const board_t *board)
{
   return board->bbp[WPAWN] | board->bbp[BHOPLITE];
}

static inline bitboard_t get_occupied(const board_t *board)
{
   return board->occ;
   //return board->bbc[WHITE] | board->bbc[BLACK];
}

static inline void assert_move(board_t *board, move_t move)
{
#ifdef DEBUGMODE
    int n, c;

    /* First: check all pickups */
    n = get_move_pickups(move);
    for (c=0; c<n; c++) {
       uint16_t p = get_move_pickup(move, c);
       int square = decode_pickup_square(p);
       int piece  = decode_pickup_piece(p);
       sides side = decode_pickup_side(p);
       bitboard_t bb = make_bitboard_square(square);

       if (!(board->bbc[side]&bb)){
          print_bitboards(board);
          printf("%s\n", move_string(move, NULL));
       }
       if (!(board->bbp[piece]&bb)){
          print_bitboards(board);
          printf("%s\n", move_string(move, NULL));
       }
       assert(board->bbc[side]&bb);
       assert(board->bbp[piece]&bb);
       assert(board->piece[square] == piece);
    }

    /* Second: check all drops */
    n = get_move_drops(move);
    if (!is_gate_move(move))
    for (c=0; c<n; c++) {
       uint16_t p = get_move_drop(move, c);
       int square = decode_drop_square(p);
       int piece  = decode_drop_piece(p);
       sides side = decode_drop_side(p);
       bitboard_t bb = make_bitboard_square(square);

       assert((board->bbc[side]&bb) == 0);
    }

    assert((board->bbc[0] & board->bbc[1]) == 0);
    assert((board->bbc[0]  | board->bbc[1]) == board->occ);
    assert( (board->bbc[0]  | board->bbc[1]) ==
          (board->bbp[0]  | board->bbp[1]  | board->bbp[2]  | board->bbp[3]  | board->bbp[4]  |
           board->bbp[5]  | board->bbp[6]  | board->bbp[7]  | board->bbp[8]  | board->bbp[9]  |
           board->bbp[10] | board->bbp[11] | board->bbp[12] ) );
#endif
}



static inline void makemove(board_t *board, move_t move)
{
   int n, c;

   /* Do some basic sanity checks up-front */
   assert_move(board, move);

   /* First: resolve all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->occ ^= bb;
      board->bbp[piece] ^= bb;
      board->init &= ~bb;
      board->hash ^= piece_key[piece][side][square];
      //if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      assert((board->bbc[side]&bb) == 0);
      assert((board->bbp[piece]&bb) == 0);
   }

   /* Second: resolve all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->occ ^= bb;
      board->bbp[piece] ^= bb;
      board->piece[square] = piece;
      board->hash ^= piece_key[piece][side][square];
      //if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      assert(board->bbc[side]&bb);
      assert(board->bbp[piece]&bb);
   }

   /* Third: update holdings */
   if (expect(get_move_holdings(move), 0)) {
      uint16_t p = get_move_holding(move);
      int piece  = decode_holding_piece(p);
      sides side = decode_holding_side(p);
      board->holdings[side] ^= 1<<piece;
      board->hash ^= holding_key[side][piece];
   }

   /* Fourth: flip side to move */
   board->side_to_move = next_side[board->side_to_move];
   board->hash ^= side_to_move_key;

   /* Assume we're not in check */
   board->in_check = false;

   assert((board->bbc[0] & board->bbc[1]) == 0);
   assert( (board->bbc[0]  | board->bbc[1]) ==
           (board->bbp[0]  | board->bbp[1]  | board->bbp[2]  | board->bbp[3]  | board->bbp[4]  |
            board->bbp[5]  | board->bbp[6]  | board->bbp[7]  | board->bbp[8]  | board->bbp[9]  |
            board->bbp[10] | board->bbp[11] | board->bbp[12] ) );
}

static inline void unmakemove(board_t *board, move_t move)
{
   int n, c;

   /* First: flip side to move */
   if (expect((move & MOVE_KEEP_TURN) == 0, true)) {
      board->side_to_move = next_side[board->side_to_move];
      board->hash ^= side_to_move_key;
   }

   /* Second: reverse all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->occ ^= bb;
      board->bbp[piece] ^= bb;
      board->hash ^= piece_key[piece][side][square];
      //if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      assert((board->bbc[side]&bb) == 0);
      assert((board->bbp[piece]&bb) == 0);
   }

   /* Third: reverse all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->occ ^= bb;
      board->bbp[piece] ^= bb;
      board->piece[square] = piece;
      board->hash ^= piece_key[piece][side][square];
      //if (piece == board->piece_types->pawn_index[side]) board->pawn_hash ^= piece_key[piece][side][square];

      assert(board->bbc[side]&bb);
      assert(board->bbp[piece]&bb);
   }

   /* Fourth: update holdings */
   if (expect(get_move_holdings(move), 0)) {
      uint16_t p = get_move_holding(move);
      int piece  = decode_holding_piece(p);
      sides side = decode_holding_side(p);
      board->holdings[side] ^= 1<<piece;
      board->hash ^= holding_key[side][piece];
   }

   assert((board->bbc[0] & board->bbc[1]) == 0);
   assert( (board->bbc[0]  | board->bbc[1]) ==
           (board->bbp[0]  | board->bbp[1]  | board->bbp[2]  | board->bbp[3]  | board->bbp[4]  |
            board->bbp[5]  | board->bbp[6]  | board->bbp[7]  | board->bbp[8]  | board->bbp[9]  |
            board->bbp[10] | board->bbp[11] | board->bbp[12] ) );
}

static inline void makemove_see(board_t *board, move_t move)
{
   int n, c;

   /* First: resolve all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->occ ^= bb;
      board->bbp[piece] ^= bb;

      assert((board->bbc[side]&bb) == 0);
      assert((board->bbp[piece]&bb) == 0);
   }

   /* Second: resolve all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      bitboard_t bb = make_bitboard_square(square);
      board->bbc[side] ^= bb;
      board->occ ^= bb;
      board->bbp[piece] ^= bb;
      board->piece[square] = piece;

      assert(board->bbc[side]&bb);
      assert(board->bbp[piece]&bb);
   }

   /* Next player */
   if (expect((move & MOVE_KEEP_TURN) == 0, true))
      board->side_to_move = next_side[board->side_to_move];
}

static inline uint64_t update_hash_from_move(move_t move, uint64_t hash)
{
   int n, c;

   /* First: resolve all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int square = decode_pickup_square(p);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);
      hash ^= piece_key[piece][side][square];
   }

   /* Second: resolve all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int square = decode_drop_square(p);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);
      hash ^= piece_key[piece][side][square];
   }

   /* Third: update holdings */
   if (expect(get_move_holdings(move), 0)) {
      uint16_t p = get_move_holding(move);
      int piece  = decode_holding_piece(p);
      sides side = decode_holding_side(p);
      hash ^= holding_key[side][piece];
   }

   /* Fourth: flip side to move */
   if (expect((move & MOVE_KEEP_TURN) == 0, true)) {
      hash ^= side_to_move_key;
   }

   return hash;
}

#endif
