/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include "pieces.h"
#include "movegen.h"
#include "board.h"
#include "see.h"
#include "bits.h"
#include "evaluate.h"

#undef PRINT_SEE_DEBUG_MESSAGES

static inline int max(int x, int y)
{
   return (x>y)?x:y;
}

static inline int find_least_valued_piece(const board_t *board, bitboard_t own)
{
   int perm[NUM_PIECES] = { WPAWN, BHOPLITE, KNIGHT, BISHOP, LIEUTENANT, CAPTAIN, ROOK, GENERAL, WARLORD, MARSHAL, QUEEN, KING, BASILEUS };
   int n;

   for (n=0; n<NUM_PIECES; n++) {
      if (board->bbp[perm[n]] & own)
         return bitscan64(board->bbp[perm[n]] & own);
   }

   /* should never get here if everything works correctly for own!=0 */
   assert(own);

   return -1;
}

static int move_value(const int *piece_value, move_t move, sides side_to_move)
{
   int value = 0;
   int n, c;

   /* First: resolve all pickups */
   n = get_move_pickups(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_pickup(move, c);
      int piece  = decode_pickup_piece(p);
      sides side = decode_pickup_side(p);

      value -= (side == WHITE) ? piece_value[piece] : -piece_value[piece];
   }

   /* Second: resolve all drops */
   n = get_move_drops(move);
   for (c=0; c<n; c++) {
      uint16_t p = get_move_drop(move, c);
      int piece  = decode_drop_piece(p);
      sides side = decode_drop_side(p);

      value += (side == WHITE) ? piece_value[piece] : -piece_value[piece];
   }

   return (side_to_move == WHITE) ? value : -value;
}

/* Perform static exchange evaluation for a given capture move.
 * Uses the "swap" algorithm
 * (http://chessprogramming.wikispaces.com/SEE+-+The+Swap+Algorithm).
 * returns the evaluation in centipawns.
 */
int static_exchange_evaluation(const board_t *const_board, const move_t move)
{
   int score[32];
   int depth;
   int piece = get_move_piece(move);
   int square = get_move_to(move);
   bitboard_t attackers;
   bitboard_t own;
   board_t board;

   assert(is_capture_move(move));
   assert(get_move_player(move) == const_board->side_to_move);
   assert((const_board->bbc[get_move_player(move)] & make_bitboard_square(get_move_from(move))));

   /* Clear attackers and own bitboards */
   memset(&attackers, 0, sizeof attackers);
   memset(&own, 0, sizeof own);

   /* Make a copy of the board */
   memcpy(&board, const_board, sizeof *const_board);

   /* Initial gain: the piece on the target square */
   depth = 0;
   score[depth] = move_value(piece_value, move, board.side_to_move);

   /* Easy: if the piece we use to capture the enemy piece is worth less
    * than the piece we captured, it's always a winning capture, so return a
    * rough estimate for the SEE score.
    */
   if (score[depth] > piece_value[piece])
      return score[depth];

   /* Perform the capture on the board */
   makemove_see(&board, move);

   /* Get a new list of all attackers/defenders */
   attackers = get_all_attackers(&board, square);

   if (attackers) {
      do {
         /* Consider next move - one ply deeper, flip side */
         depth++;
         assert(depth < 128);
         /* Update score (speculative) */
         score[depth] = piece_value[get_piece(&board, square)] - score[depth-1];
         /* Perform LVA capture */
         own = attackers & board.bbc[board.side_to_move];
         if (own) {
            /* Find the least valued attacker */
            int origin = find_least_valued_piece(&board, own);

            assert(board.bbc[board.side_to_move] & make_bitboard_square(origin));

            int piece = piece_for_side(board.piece[origin], board.side_to_move);
            int ptaken = piece_for_side(board.piece[square], next_side[board.side_to_move]);
            move_t move = encode_normal_capture(piece, origin, square, ptaken);

            /* Perform the capture on the board */
            makemove_see(&board, move);

            /* Update the list of attackers/defenders */
            //attackers = get_all_attackers(&board, square);
            attackers &= ~make_bitboard_square(origin);
         }
      } while(own);

      /* Compute final score */
      while (--depth) {
         score[depth-1] = -max(-score[depth-1], score[depth]);
      }
   }
   return score[depth];
}

