/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <inttypes.h>
#include <stdio.h>

#include "leonidas.h"

#define streq(s1, s2) (strcmp((s1), (s2)) == 0)

static game_t *(*new_variant_game)(void) = create_spartan_game;

uint64_t perft(game_t *game, int depth, int root)
{
   movelist_t movelist;
   sides me = game->board.side_to_move;
   uint64_t nodes = 0;
   bool check = game->board.in_check;
   int n;

   if (depth == 0) return 1;

   /* Check if previous move left the player in check */
   generate_moves(&movelist, &game->board, me);

   for (n=0; n<movelist.num_moves; n++) {
      uint64_t count = 0;
      playmove(game, movelist.move[n]);
      //if (!player_in_check_after_move(game, movelist.move[n], me))  /* Don't count illegal moves */
      if (check || !move_leaves_player_in_check(&game->board, movelist.move[n], me))  /* Don't count illegal moves */
         count = perft(game, depth-1, root - 1);
      nodes += count;
      if (root > 0)
         printf("%8s %10"PRIu64" %10"PRIu64"\n", move_string(movelist.move[n], NULL), count, nodes);
      takeback(game);
   }
   return nodes;
}


int main(int argc, char **argv)
{
   int depth = 6;
   game_t *game = NULL;

   initialise_leonidas();

   if (argc>1) {
      int n;
      for (n=1; n<argc; n++) {
         if (strstr(argv[n], "-variant")) {
            if (n+1 >= argc) {
               fprintf(stderr, "error: -variant passed, but no variant specified");
               exit(0);
            }
            n++;

            char *input = argv[n];
            if (streq(input, "normal")) {
               new_variant_game = create_standard_game;
            } else if (streq(input, "spartan")) {
               new_variant_game = create_spartan_game;
            } else if (streq(input, "berolina")) {
               new_variant_game = create_berolina_game;
            } else if (strstr(input, "seirawan")) {
               new_variant_game = create_seirawan_game;
            } else {
               fprintf(stderr, "error: variant '%s' unknown\n", input);
               exit(0);
            }
            if (game) {
               end_game(game);
            }
            game = new_variant_game();
            start_new_game(game);
         } else if (strstr(argv[n], ".epd")) {
            char fen[4096];
            FILE *f;

            f = fopen(argv[n], "r");
            fgets(fen, sizeof fen, f);
            fclose(f);

            if (game) {
               end_game(game);
            }
            game = new_variant_game();
            start_new_game(game);
            setup_fen_position(game, fen);
         } else {
            int new_depth, result;
            
            result = sscanf(argv[n], "%d", &new_depth);
            if (result == 1)
               depth = new_depth;
         }
      }
   }

   if (!game) {
      /* Start a new game */
      game = new_variant_game();
      start_new_game(game);
   }
   //setup_fen_position(game, "rnabqkbcnr/p1pppppppp/10/1p8/P9/10/1PPPPPPPPP/RNABQKBCNR w KQkq -");
   //setup_fen_position(game, "rnabqkbcnr/pppppppppp/10/10/10/P9/1PPPPPPPPP/RNABQKBCNR b KQkq -");
   //setup_fen_position(game, "rnabqkbcnr/p1pppppppp/1p8/10/10/P9/1PPPPPPPPP/RNABQKBCNR w KQkq -");
   //setup_fen_position(game, "rnabqkbcnr/p1pppppppp/1p8/10/10/PP8/2PPPPPPPP/RNABQKBCNR b KQkq -");
   //setup_fen_position(game, "rn1bqkbcnr/p1pppppppp/ap8/10/10/PP8/2PPPPPPPP/RNABQKBCNR w KQkq -");

   /* Show position */
   print_bitboards(&game->board);

   /* Perft timing */
   uint64_t t;
   int n;
   t = get_timer();
   for (n = 1; n<depth+1; n++) {
      uint64_t nodes = perft(game, n, 0);
      uint64_t tt = get_timer();

      if (tt == t) tt++;

      printf("%2d %10"PRIu64" %5.2f %12.2fnps\n", n, nodes,
         (tt - t)/1000000.0,nodes*1.0e6/(tt-t));

      t = tt;
   }


   end_game(game);

   return 0;
}
