/*  Leonidas, a program for playing chess variants
 *  Copyright (C) 2013  Evert Glebbeek
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <string.h>
#include "movegen.h"

int white_pawn_type, black_pawn_type;

bitboard_t king_attack[64];
bitboard_t king_attack2[64];
bitboard_t ferz_attack[64];
bitboard_t knight_attack[64];
bitboard_t pawn_attack[64];
bitboard_t pawn_attacker[64];
bitboard_t pawn_step[64];
bitboard_t pawn_step2[64];
bitboard_t captain_attack[64];
bitboard_t lieutenant_attack[64];
bitboard_t lieutenant_step[64];
bitboard_t hoplite_attack[64];
bitboard_t hoplite_attacker[64];
bitboard_t hoplite_step[64];
bitboard_t hoplite_step2[64];
bitboard_t super[64];
bitboard_t rook_empty[64];
bitboard_t bishop_empty[64];
bitboard_t mslider[8][64];
bitboard_t connecting_ray[64][64];
bitboard_t joined_ray[64][64];

bitboard_t eastwest[64];
bitboard_t white_passed_pawn_mask[64];
bitboard_t black_passed_pawn_mask[64];
bitboard_t white_pawn_span[64];
bitboard_t black_pawn_span[64];

bitboard_t wpawn_attack[64];
bitboard_t wpawn_attacker[64];
bitboard_t wpawn_step[64];
bitboard_t wpawn_step2[64];
bitboard_t *bpawn_attack = wpawn_attacker;
bitboard_t *bpawn_attacker = wpawn_attack;
bitboard_t bpawn_step[64];
bitboard_t bpawn_step2[64];

bitboard_t bhoplite_attack[64];
bitboard_t bhoplite_attacker[64];
bitboard_t bhoplite_step[64];
bitboard_t bhoplite_step2[64];
bitboard_t *whoplite_attack = bhoplite_attacker;
bitboard_t *whoplite_attacker = bhoplite_attack;
bitboard_t whoplite_step[64];
bitboard_t whoplite_step2[64];

bitboard_t mrook[64][64][64];
bitboard_t mbish[64][64][64];

void initialise_movement_tables(int white_pawn, int black_pawn)
{
   int file, square;
   int occ_mask, occ;
   int n;

   white_pawn_type = white_pawn;
   black_pawn_type = black_pawn;

   memset(&mslider, 0, sizeof mslider);

   /* Rank attacks */
   for (file = 0; file < 8; file++) {
      for (occ = 0; occ < 256; occ++) {
         int mocc = (occ >> 1) & 63;

         /* South of slider position, rook and cannon moves, rook
          * attacks.
          */
         for (n=file-1; n>=0; n--) {
            mslider[file][mocc] |= make_bitboard_square(n);
            if ( occ & (1 << n) ) break;
         }

         /* North of slider position */
         for (n=file+1; n<8; n++) {
            mslider[file][mocc] |= make_bitboard_square(n);
            if ( occ & (1 << n) ) break;
         }
      }
   }

   /* Now duplicate the first rank across the board. This makes the
    * generation of rank/diagonal attacks more efficient.
    */
   for (file = 0; file < 8; file++) {
      for (occ = 0; occ < 64; occ++) {
         mslider[file][occ] *= board_afile;
      }
   }

   /* Complete rook/bishop attack tables */
   for (square = 0; square < 64; square++) {
      int file = unpack_file(square);
      int rank = unpack_rank(square);
      int diag_nr = get_a1h8_diagonal(square);
      int adiag_nr = get_a8h1_diagonal(square);
      int occ1, occ2;
      for (occ1=0; occ1<64; occ1++) {
         for (occ2=0; occ2<64; occ2++) {
            bitboard_t bb = mslider[7-rank][occ2] & board_rank1;
            mrook[square][occ1][occ2]  = mslider[file][occ1] & board_rank[rank];
            mrook[square][occ1][occ2] |= ((bb * board_a1h8_diagonal[7]) & board_hfile) >> (7-file);

            mbish[square][occ1][occ2]  = mslider[file][occ1] & board_a1h8_diagonal[diag_nr];
            mbish[square][occ1][occ2] |= mslider[file][occ2] & board_a8h1_diagonal[adiag_nr];
         }
      }
   }

   /* Generate movement tables for the king and the knight.
    * The movement tables for the knight are relatively simple,
    * because knights can't be blocked. On the discrete grid of the chess
    * board, the possible destinations of the knight lie on a circle with
    * radius 5.
    * The king likewise can't be blocked, because it can only move one
    * square at a time.
    */
   for (square=0; square<64; square++) {
      int rank = unpack_rank(square);
      int file = unpack_file(square);
      int dx, dy;

      king_attack[square] = board_empty;
      for (dx=-1; dx<2; dx++)
         for (dy=-1; dy<2; dy++) {
            if ( (dx||dy) &&
                  (rank+dx>=0 && rank+dx<=7 && file+dy>=0 && file+dy<=7) ) {
               set_bitboard(&(king_attack[square]), pack_row_file(rank+dx,file+dy));
            }
         }

      knight_attack[square] = board_empty;
      for (dx=-3; dx<4; dx++)
         for (dy=-3; dy<4; dy++) {
            if ( (dx*dx + dy*dy == 5) &&
                  (rank+dx>=0 && rank+dx<=7 && file+dy>=0 && file+dy<=7) ) {
               set_bitboard(&(knight_attack[square]), pack_row_file(rank+dx,file+dy));
            }
         }
   }

   /* Attacks by a super piece */
   for (square = A1; square <= H8; square++) {
      super[square] = get_queen_attacks(square, board_empty) | knight_attack[square];
      rook_empty[square] = get_rook_attacks(square, board_empty);
      bishop_empty[square] = get_bishop_attacks(square, board_empty);
   }

   /* Squares that can be attacked by a king in at most 2 moves, used for
    * some end game evaluation terms.
    */
   for (square = A1; square <= H8; square++) {
      bitboard_t bb = king_attack2[square] = king_attack[square];
      while (bb) {
         int s = bitscan64(bb);
         king_attack2[square] |= king_attack[s];
         bb ^= make_bitboard_square(s);
      }
   }

   for (square=0; square<64; square++) {
      captain_attack[square] = king_attack2[square] & get_rook_attacks(square, board_empty);
      lieutenant_attack[square] = king_attack2[square] & get_bishop_attacks(square, board_empty);
      ferz_attack[square] = king_attack[square] & get_bishop_attacks(square, board_empty);
      lieutenant_step[square] = king_attack[square] & get_rank_attacks(board_empty, square);
   }

   /* Pawn and hoplite tables */
   for (square=0; square<64; square++) {
      bitboard_t bb = make_bitboard_square(square);
      wpawn_attack[square] = ((bb << 9) & ~board_afile) | ((bb << 7) & ~board_hfile);
      wpawn_attacker[square] = ((bb >> 7) & ~board_afile) | ((bb >> 9) & ~board_hfile);
      wpawn_step[square] = bb << 8;
      wpawn_step2[square] = (bb << 16) & board_rank4;
      bpawn_step[square] = bb >> 8;
      bpawn_step2[square] = (bb >> 16) & board_rank5;

      bhoplite_step[square] = ((bb >> 9) & ~board_hfile) | ((bb >> 7) & ~board_afile);
      bhoplite_step2[square] = ((bb >> 18) & ~(board_hfile|board_gfile)) |
                              ((bb >> 14) & ~(board_afile|board_bfile));// | bb << 16;
      bhoplite_step2[square] &= board_rank5;
      whoplite_step[square] = ((bb << 9) & ~board_afile) | ((bb << 7) & ~board_hfile);
      whoplite_step2[square] = ((bb << 14) & ~(board_hfile|board_gfile)) |
                              ((bb << 18) & ~(board_afile|board_bfile));// | bb << 16;
      whoplite_step2[square] &= board_rank4;
      bhoplite_attack[square] = bb >> 8;
      bhoplite_attacker[square] = bb << 8;
   }

   /* east/west neighbours, for pawn structure evaluation */
   for (square=0; square<64; square++) {
      bitboard_t bb = make_bitboard_square(square);
      eastwest[square] = ((bb >> 1) & ~board_afile) | ((bb << 1 & ~board_hfile));
   }

   /* Add the initial "leap" for the hoplite.
    * This makes the "hoplite_step2" redundant, but keep that anyway: it is
    * useful to preserve the symmetry between pawns and hoplites so that we
    * can turn the hoplite into a normal pawn by just changing its table,
    * or into a Berolina pawn by omitting this step.
    */
   if (white_pawn == SPARTAN_PAWN) {
      for (square = A2; square <= H2; square++)
         whoplite_step[square] |= lieutenant_attack[square] & ~board_rank1;
   }
   if (black_pawn == SPARTAN_PAWN) {
      for (square = A7; square <= H7; square++)
         bhoplite_step[square] |= lieutenant_attack[square] & ~board_rank8;
   }

   /* Copy the appropriate movement tables for the pawns */
   for (square=0; square<64; square++) {
      switch (white_pawn) {
         case NORMAL_PAWN:
            pawn_attack[square] = wpawn_attack[square];
            pawn_attacker[square] = wpawn_attacker[square];
            pawn_step[square] = wpawn_step[square];
            pawn_step2[square] = wpawn_step2[square];
            break;

         case BEROLINA_PAWN:
            printf("Warning: two-step generation of berolina pawns is broken.\n");
         case SPARTAN_PAWN:
            pawn_attack[square] = whoplite_attack[square];
            pawn_attacker[square] = whoplite_attacker[square];
            pawn_step[square] = whoplite_step[square];
            pawn_step2[square] = whoplite_step2[square];
            break;

         default:
            break;
      }

      switch (black_pawn) {
         case NORMAL_PAWN:
            hoplite_attack[square] = bpawn_attack[square];
            hoplite_attacker[square] = bpawn_attacker[square];
            hoplite_step[square] = bpawn_step[square];
            hoplite_step2[square] = bpawn_step2[square];
            break;

         case BEROLINA_PAWN:
            printf("Warning: two-step generation of berolina pawns is broken.\n");
         case SPARTAN_PAWN:
            hoplite_attack[square] = bhoplite_attack[square];
            hoplite_attacker[square] = bhoplite_attacker[square];
            hoplite_step[square] = bhoplite_step[square];
            hoplite_step2[square] = bhoplite_step2[square];
            break;

         default:
            break;
      }
   }

   /* Calculate passed pawn masks: set bits indicate blocking enemy pawns.
    * We need to be a little bit careful with hoplites.
    */
   memset(white_passed_pawn_mask, 0, sizeof white_passed_pawn_mask);
   memset(black_passed_pawn_mask, 0, sizeof black_passed_pawn_mask);
   memset(white_pawn_span, 0, sizeof white_pawn_span);
   memset(black_pawn_span, 0, sizeof black_pawn_span);

   /* Mark locations where enemy pawns attack a pawn on this square, or
    * block its advance directly.
    * In the second step, consider the forward movement span of this pawn.
    */
   for (square=0; square<64; square++) {
      white_passed_pawn_mask[square] |= hoplite_attacker[square] | pawn_step[square];
      black_passed_pawn_mask[square] |= pawn_attacker[square] | hoplite_step[square];
      white_pawn_span[square] |= pawn_step[square];
      black_pawn_span[square] |= hoplite_step[square];

      /* consider the forward span in the mask, white */
      bitboard_t bb = pawn_step[square];
      while(bb) {
         int ds = bitscan64(bb);
         bb ^= make_bitboard_square(ds);
         bb |= pawn_step[ds];

         white_pawn_span[square] |= pawn_step[ds];
         white_passed_pawn_mask[square] |= hoplite_attacker[ds] | pawn_step[ds];
      }

      /* consider the forward span in the mask, black */
      bb = hoplite_step[square];
      while(bb) {
         int ds = bitscan64(bb);
         bb ^= make_bitboard_square(ds);
         bb |= hoplite_step[ds];

         black_pawn_span[square] |= hoplite_step[ds];
         black_passed_pawn_mask[square] |= pawn_attacker[ds] | hoplite_step[ds];
      }
   }

   for (square=0; square<64; square++) {
      int s;
      for (s=0; s<64; s++) {
         if (white_pawn_span[square] & black_pawn_span[s]) {
            white_passed_pawn_mask[square] |= make_bitboard_square(s);
            black_passed_pawn_mask[s] |= make_bitboard_square(square);
         }
      }
   }

#if 0
   printf_bitboard(get_rook_attacks(E4, board_rank7));
   printf_bitboard(get_bishop_attacks(E4, board_empty));
   printf_bitboard(get_queen_attacks(E4, board_rank7));
   printf_bitboard(get_knight_attacks(E4, board_rank7));
   printf_bitboard(get_king_attacks(E4, board_rank7));
   printf_bitboard(get_warlord_attacks(E4, board_rank7));
   printf_bitboard(get_general_attacks(E4, board_rank7));

   printf_bitboard(get_lieutenant_attacks(E4, board_rank7));
   printf_bitboard(get_captain_attacks(E4, board_rank7));
   printf_bitboard(lieutenant_step[E4]);
   printf_bitboard(pawn_attack[E4]);
   printf_bitboard(pawn_attack[A4]);
   printf_bitboard(pawn_attack[H4]);

   exit(0);
#endif

   /* Connecting rays */
   memset(connecting_ray, 0, sizeof connecting_ray);
   for (square = A1; square <= H8; square++) {
      int attack;
      for (attack = square+1; attack<=H8; attack++) {
         int row = unpack_rank(square);
         int file = unpack_file(square);
         if (row == unpack_rank(attack)) {
            int n;
            for (n=file;n<=unpack_file(attack);n++)
               connecting_ray[square][attack] |=
                  make_bitboard_square(pack_row_file(row, n));
         }
         if (file == unpack_file(attack)) {
            int n;
            for (n=row;n<=unpack_rank(attack);n++)
               connecting_ray[square][attack] |=
                  make_bitboard_square(pack_row_file(n, file));
         }
         if (get_a1h8_diagonal(square) == get_a1h8_diagonal(attack)) {
            int n;
            for (n=square;n<=(attack);n+=9)
               connecting_ray[square][attack] |= make_bitboard_square(n);
         }
         if (get_a8h1_diagonal(square) == get_a8h1_diagonal(attack)) {
            int n;
            for (n=square;n<=(attack);n+=7)
               connecting_ray[square][attack] |= make_bitboard_square(n);
         }
         connecting_ray[attack][square] = connecting_ray[square][attack];
      }
   }

   memset(joined_ray, 0, sizeof joined_ray);
   for (square = A1; square <= H8; square++) {
      int attack;
      for (attack = square+1; attack<=H8; attack++) {
         int row = unpack_rank(square);
         int file = unpack_file(square);
         if (row == unpack_rank(attack))
            joined_ray[square][attack] |= board_rank[row];
         if (file == unpack_file(attack))
            joined_ray[square][attack] |= board_file[file];
         if (get_a1h8_diagonal(square) == get_a1h8_diagonal(attack))
            joined_ray[square][attack] |= board_a1h8_diagonal[get_a1h8_diagonal(square)];
         if (get_a8h1_diagonal(square) == get_a8h1_diagonal(attack))
            joined_ray[square][attack] |= board_a8h1_diagonal[get_a8h1_diagonal(square)];
         joined_ray[attack][square] = joined_ray[square][attack];
      }
   }
}

void generate_moves_mask(movelist_t *movelist, const board_t *board, bitboard_t from_mask, bitboard_t to_mask, sides side_to_move, uint16_t allowed_promotion_pieces)
{
   bitboard_t own, enemy, own_movers;
   bitboard_t occupied;
   bitboard_t attacked;
   bitboard_t rooks;
   bitboard_t bishops;
   bitboard_t knights;
   bitboard_t kings;
   bitboard_t bb;
   bitboard_t moves;
   bitboard_t captures;
   bitboard_t en_passant;
   move_t *move;

   /* Bookkeeping: we keep a pointer to the next move in the move list, and
    * update the number of moves in the list at the end of this function
    */
   move = &(movelist->move[movelist->num_moves]);

   own = board->bbc[side_to_move];
   enemy = board->bbc[next_side[side_to_move]];

   en_passant = board_empty;
   if (board->ep) en_passant = make_bitboard_square(board->ep);

   allowed_promotion_pieces &= promotion_options[side_to_move];

   /* Only one spartan king is ever allowed */
   if (board->bbp[BASILEUS] && !onebit64(board->bbp[BASILEUS]))
      allowed_promotion_pieces &= ~KING_PROMOTION;

   occupied = get_occupied(board);
   own_movers = own & from_mask;

   /* Generate rook-like moves */
   rooks = get_rook_movers(board);
   bb = rooks & own_movers;
   while (bb) {
      int from = bitscan64(bb);
      int piece = piece_for_side(get_piece(board, from), side_to_move);
      bb ^= make_bitboard_square(from);
      moves = get_rook_attacks(from, occupied) & to_mask;
      captures = moves & enemy;
      moves &= ~occupied;

      while (moves) {
         int to = bitscan64(moves);
         moves ^= make_bitboard_square(to);
         *move = encode_normal_move(piece, from, to);
         move++;
      }

      while (captures) {
         int to = bitscan64(captures);
         int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
         captures ^= make_bitboard_square(to);
         *move = encode_normal_capture(piece, from, to, ptaken);
         move++;
      }
   }

   /* Generate all bishop-like moves */
   bishops = get_bishop_movers(board);
   bb = bishops & own_movers;
   while (bb) {
      int from = bitscan64(bb);
      int piece = piece_for_side(get_piece(board, from), side_to_move);
      bb ^= make_bitboard_square(from);
      moves = get_bishop_attacks(from, occupied) & to_mask;
      captures = moves & enemy;
      moves &= ~occupied;

      while (moves) {
         int to = bitscan64(moves);
         moves ^= make_bitboard_square(to);
         *move = encode_normal_move(piece, from, to);
         move++;
      }

      while (captures) {
         int to = bitscan64(captures);
         int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
         captures ^= make_bitboard_square(to);
         *move = encode_normal_capture(piece, from, to, ptaken);
         move++;
      }
   }

   /* Generate all knight-like moves */
   knights = get_knight_movers(board);
   bb = knights & own_movers;
   while (bb) {
      int from = bitscan64(bb);
      int piece = piece_for_side(get_piece(board, from), side_to_move);
      bb ^= make_bitboard_square(from);
      moves = get_knight_attacks(from, occupied) & to_mask;
      captures = moves & enemy;
      moves &= ~occupied;

      while (moves) {
         int to = bitscan64(moves);
         moves ^= make_bitboard_square(to);
         *move = encode_normal_move(piece, from, to);
         move++;
      }

      while (captures) {
         int to = bitscan64(captures);
         int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
         captures ^= make_bitboard_square(to);
         *move = encode_normal_capture(piece, from, to, ptaken);
         move++;
      }
   }

   /* Ferz-like moves */
   bb = board->bbp[GENERAL] & own_movers;
   while (bb) {
      int from = bitscan64(bb);
      int piece = piece_for_side(get_piece(board, from), side_to_move);
      bb ^= make_bitboard_square(from);
      moves = ferz_attack[from] & to_mask;
      captures = moves & enemy;
      moves &= ~occupied;

      while (moves) {
         int to = bitscan64(moves);
         moves ^= make_bitboard_square(to);
         *move = encode_normal_move(piece, from, to);
         move++;
      }

      while (captures) {
         int to = bitscan64(captures);
         int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
         captures ^= make_bitboard_square(to);
         *move = encode_normal_capture(piece, from, to, ptaken);
         move++;
      }
   }

   /* Lieutenant moves */
   bb = board->bbp[LIEUTENANT] & own_movers;
   while (bb) {
      int from = bitscan64(bb);
      int piece = piece_for_side(get_piece(board, from), side_to_move);
      bb ^= make_bitboard_square(from);
      moves = get_lieutenant_attacks(from, occupied) & to_mask;
      captures = moves & enemy;
      moves |= lieutenant_step[from] & to_mask;
      moves &= ~occupied;

      while (moves) {
         int to = bitscan64(moves);
         moves ^= make_bitboard_square(to);
         *move = encode_normal_move(piece, from, to);
         move++;
      }

      while (captures) {
         int to = bitscan64(captures);
         int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
         captures ^= make_bitboard_square(to);
         *move = encode_normal_capture(piece, from, to, ptaken);
         move++;
      }
   }

   /* Captain moves */
   bb = board->bbp[CAPTAIN] & own_movers;
   while (bb) {
      int from = bitscan64(bb);
      int piece = piece_for_side(get_piece(board, from), side_to_move);
      bb ^= make_bitboard_square(from);
      moves = get_captain_attacks(from, occupied) & to_mask;
      captures = moves & enemy;
      moves &= ~occupied;

      while (moves) {
         int to = bitscan64(moves);
         moves ^= make_bitboard_square(to);
         *move = encode_normal_move(piece, from, to);
         move++;
      }

      while (captures) {
         int to = bitscan64(captures);
         int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
         captures ^= make_bitboard_square(to);
         *move = encode_normal_capture(piece, from, to, ptaken);
         move++;
      }
   }

   /* Normal king moves */
   kings = get_royal(board);
   bb = kings & own_movers;
   while (bb) {
      int from = bitscan64(bb);
      int piece = piece_for_side(get_piece(board, from), side_to_move);
      bb ^= make_bitboard_square(from);
      moves = get_king_attacks(from, occupied) & to_mask;
      captures = moves & enemy;
      moves &= ~occupied;

      while (moves) {
         int to = bitscan64(moves);
         moves ^= make_bitboard_square(to);
         *move = encode_normal_move(piece, from, to);
         move++;
      }

      while (captures) {
         int to = bitscan64(captures);
         int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
         captures ^= make_bitboard_square(to);
         *move = encode_normal_capture(piece, from, to, ptaken);
         move++;
      }
   }

   /* North-bound pawn moves and captures */
   bb = board->bbp[WPAWN] & own_movers;
   while (bb) {
      int from = bitscan64(bb);
      int piece = piece_for_side(get_piece(board, from), side_to_move);
      bb ^= make_bitboard_square(from);
      captures = get_wpawn_attacks(from, occupied) & (enemy | en_passant);
      moves = pawn_step[from] & ~occupied;
      if (moves) moves |= pawn_step2[from] & ~occupied;
      moves &= to_mask;
      captures &= to_mask;

      /* Promotions */
      if ((moves | captures) & board_rank8) {
         while (moves & board_rank8) {
            int to = bitscan64(moves & board_rank8);
            moves ^= make_bitboard_square(to);
            uint16_t tpm = allowed_promotion_pieces;
            while (tpm) {
               int t = bitscan16(tpm);
               int tpiece = piece_for_side(t, side_to_move);
               tpm ^= 1<<t;
               *move = encode_normal_promotion(piece, from, to, tpiece) | MOVE_RESET50;
               move++;
            }
         }

         while (captures & board_rank8) {
            int to = bitscan64(captures & board_rank8);
            int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
            captures ^= make_bitboard_square(to);
            uint16_t tpm = allowed_promotion_pieces;
            while (tpm) {
               int t = bitscan16(tpm);
               int tpiece = piece_for_side(t, side_to_move);
               tpm ^= 1<<t;
               *move = encode_capture_promotion(piece, from, to, ptaken, tpiece) | MOVE_RESET50;
               move++;
            }
         }
      }

      /* En-passant */
      if (captures & en_passant) {
         int ptaken = piece_for_side(get_piece(board, board->ep_capture), next_side[side_to_move]);
         int to = board->ep;
         *move = encode_en_passant_capture(piece, from, to, ptaken, board->ep_capture);
         move++;
         captures ^= en_passant;
      }

      /* Now exclude promotions */
      moves &= ~board_rank8;
      captures &= ~board_rank8;

      while (moves) {
         int to = bitscan64(moves);
         moves ^= make_bitboard_square(to);
         *move = encode_normal_move(piece, from, to) | MOVE_RESET50;
         if (make_bitboard_square(to) & pawn_step2[from] & ~pawn_step[from])
            *move |= MOVE_SET_ENPASSANT;
         move++;
      }

      while (captures) {
         int to = bitscan64(captures);
         int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
         captures ^= make_bitboard_square(to);
         *move = encode_normal_capture(piece, from, to, ptaken) | MOVE_RESET50;
         move++;
      }
   }


   /* South-bound pawn (hoplite) moves and captures */
   bb = board->bbp[BHOPLITE] & own_movers;
   while (bb) {
      int from = bitscan64(bb);
      int piece = piece_for_side(get_piece(board, from), side_to_move);
      bb ^= make_bitboard_square(from);
      captures = get_bhoplite_attacks(from, occupied) & (enemy | en_passant);
      moves = hoplite_step[from] & ~occupied;
      if (moves) moves |= hoplite_step2[from] & ~occupied;
      moves &= to_mask;
      captures &= to_mask;

      /* Promotions */
      if ((moves | captures) & board_rank1) {
         while (moves & board_rank1) {
            int to = bitscan64(moves & board_rank1);
            moves ^= make_bitboard_square(to);
            uint16_t tpm = allowed_promotion_pieces;
            while (tpm) {
               int t = bitscan16(tpm);
               int tpiece = piece_for_side(t, side_to_move);
               tpm ^= 1<<t;
               *move = encode_normal_promotion(piece, from, to, tpiece) | MOVE_RESET50;
               move++;
            }
         }

         while (captures & board_rank1) {
            int to = bitscan64(captures & board_rank1);
            int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
            captures ^= make_bitboard_square(to);
            uint16_t tpm = allowed_promotion_pieces;
            while (tpm) {
               int t = bitscan16(tpm);
               int tpiece = piece_for_side(t, side_to_move);
               tpm ^= 1<<t;
               *move = encode_capture_promotion(piece, from, to, ptaken, tpiece) | MOVE_RESET50;
               move++;
            }
         }
      }

      /* En-passant */
      if (captures & en_passant) {
         int ptaken = piece_for_side(get_piece(board, board->ep_capture), next_side[side_to_move]);
         int to = board->ep;
         *move = encode_en_passant_capture(piece, from, to, ptaken, board->ep_capture);
         move++;
         captures ^= en_passant;
      }

      /* Now exclude promotions */
      moves &= ~board_rank1;
      captures &= ~board_rank1;

      while (moves) {
         int to = bitscan64(moves);
         moves ^= make_bitboard_square(to);
         *move = encode_normal_move(piece, from, to) | MOVE_RESET50;
         if (make_bitboard_square(to) & hoplite_step2[from] & ~hoplite_step[from])
            *move |= MOVE_SET_ENPASSANT;
         move++;
      }

      while (captures) {
         int to = bitscan64(captures);
         int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
         captures ^= make_bitboard_square(to);
         *move = encode_normal_capture(piece, from, to, ptaken) | MOVE_RESET50;
         move++;
      }
   }

   /* Castling */
   bb = board->bbp[KING] & own_movers & board->init;
   if (bb) {
      int from = bitscan64(bb);
      int piece = piece_for_side(get_piece(board, from), side_to_move);
      if ((board->init & short_castle_mask[side_to_move]) == short_castle_mask[side_to_move]) {
         if (!short_castle_free[side_to_move] || (occupied & short_castle_free[side_to_move]) == board_empty && (to_mask & short_castle_free[side_to_move])) {
            bitboard_t test             = short_castle_safe[side_to_move];
            bitboard_t mask             = get_super_attacks_for_squares(test);
            bitboard_t attacked_squares = generate_attack_bitboard(board, test, mask, next_side[side_to_move]);
            if ((attacked_squares & short_castle_safe[side_to_move]) == board_empty) {
               int from1 = bitscan64(short_castle_mask[side_to_move] & bb);
               int from2 = bitscan64(short_castle_mask[side_to_move] & ~bb);
               int piece2 = piece_for_side(get_piece(board, from2), side_to_move);
               int to1 = short_castle_king_dest[side_to_move];
               *move = encode_castle_move(piece, from1, to1, piece2, from2, to1-1); move++;
            }
         }
      }

      if ((board->init & long_castle_mask[side_to_move]) == long_castle_mask[side_to_move]) {
         if (!long_castle_free[side_to_move] || (occupied & long_castle_free[side_to_move]) == board_empty && (to_mask & long_castle_free[side_to_move])) {
            bitboard_t test             = long_castle_safe[side_to_move];
            bitboard_t mask             = get_super_attacks_for_squares(test);
            bitboard_t attacked_squares = generate_attack_bitboard(board, test, mask, next_side[side_to_move]);
            if ((attacked_squares & long_castle_safe[side_to_move]) == board_empty) {
               int from1 = bitscan64(long_castle_mask[side_to_move] & bb);
               int from2 = bitscan64(long_castle_mask[side_to_move] & ~bb);
               int piece2 = piece_for_side(get_piece(board, from2), side_to_move);
               int to1 = long_castle_king_dest[side_to_move];
               *move = encode_castle_move(piece, from1, to1, piece2, from2, to1+1); move++;
            }
         }
      }
   }

   /* Store the number of moves currently in the list */
   movelist->num_moves = (move - &(movelist->move[0]));
}

bool verify_move(movelist_t *movelist, const board_t *board, sides side_to_move, move_t move)
{
   assert(movelist);
   assert(board);

   movelist->num_moves = 0;

   bitboard_t from = make_bitboard_square(get_move_from(move));
   bitboard_t to = make_bitboard_square(get_move_from(to));
   generate_moves_mask(movelist, board, from, to, side_to_move, ANY_PROMOTION);
   if (movelist->num_moves) {
      return movelist->move[0] == move;
   }
   return false;
}

/* Add gating moves to the move list.
 * Intended for Seirawan chess.
 */
static void generate_gate_moves(movelist_t *movelist, const board_t *board, sides side_to_move)
{
   bitboard_t king = get_royal(board) & board->bbc[side_to_move];
   bitboard_t rank = board_rank8;
   int n_last, n;

   if (side_to_move == WHITE)
      rank = board_rank1;

   /* We only care about pieces that have not yet moved */
   if (!(rank & board->init))
      return;

   bitboard_t pinned = board_empty;
   if (rank & get_rook_movers(board) & board->bbc[next_side[side_to_move]])
      pinned = get_pinned_pieces(board, king, side_to_move);
   rank &= board->init;
   pinned &= rank;

   if (pinned) {
      /* Filter out moves of pinned pieces that are not along the same rank:
       * they are illegal because they expose the king to check.
       */
      n_last = movelist->num_moves-1;
      n = 0;
      while (n<movelist->num_moves) {
         move_t move = movelist->move[n];
         bitboard_t from = make_bitboard_square(get_move_from(move));
         bitboard_t to = make_bitboard_square(get_move_to(move));
         if ((pinned & from) && !(rank & to)) {
            movelist->move[n] = movelist->move[n_last];
            movelist->move[n_last] = move;
            movelist->num_moves--;
            n_last--;
         }
         n++;
      }
   }

   /* Go through the move list and add appropriate gating moves */
   rank &= board->init;
   move_t *move = &(movelist->move[movelist->num_moves]);
   n_last = movelist->num_moves;
   for (n=0; n<n_last; n++) {
      move_t base = movelist->move[n];
      bitboard_t from = make_bitboard_square(get_move_from(base));

      if (from & rank) {
         uint16_t tpm = board->holdings[side_to_move];
         int from = get_move_from(base);
         while (tpm) {
            int t = bitscan16(tpm);
            int tpiece = piece_for_side(t, side_to_move);
            tpm ^= 1<<t;
            *move = add_move_gate(base, tpiece, from) | MOVE_RESET50;
            move++;
         }
      }

      if (is_castle_move(base)) {
         int from = decode_pickup_square(get_move_pickup(base, 1));
         bitboard_t bb_from = make_bitboard_square(from);

         if (bb_from & rank) {
            uint16_t tpm = board->holdings[side_to_move];
            while (tpm) {
               int t = bitscan16(tpm);
               int tpiece = piece_for_side(t, side_to_move);
               tpm ^= 1<<t;
               *move = add_move_gate(base, tpiece, from) | MOVE_RESET50;
               move++;
            }
         }
      }
   }
   movelist->num_moves = (move - &(movelist->move[0]));
}

void generate_evasion_moves(movelist_t *movelist, const board_t *board, sides side_to_move)
{
   assert(board->in_check);

   movelist->num_moves = 0;
   movelist->cur_move = 0;

   /* The following moves are legal check evasions:
    *  1. Interpose between any king and the checking piece (or capture the
    *     piece). Exclude pinned pieces.
    *  2. King moves to squares that are not attacked.
    *  3. Promotion of pawn to king on a safe square (if legal)
    */
    bitboard_t kings = get_royal(board) & board->bbc[side_to_move];
    bitboard_t pinned = board_empty;
    bitboard_t block = board_empty;
    bitboard_t bb;

    /* Step 1.
     * Generate legal evasions for all kings.
     * If a king has only one attacker, then we can generate
     * interpositions as well, so we need to keep track of that.
     */
   bb = kings;
   while (bb) {
      int king_square = bitscan64(bb);
      bitboard_t king = make_bitboard_square(king_square);
      bitboard_t mypin;
      bb ^= king;

      bitboard_t attackers = get_all_attackers_for_side(board, king_square, next_side[side_to_move]);
      mypin = get_pinned_pieces(board, king, side_to_move);
      pinned |= mypin;

      assert(attackers);
      if (onebit64(attackers)) {
         bitboard_t bb = attackers;
         if (attackers & get_sliders(board)) {
            int square = bitscan64(attackers);
            bb |= connecting_ray[king_square][square];
         }
         block |= bb;

         /* If there is another king, there are a few peculiar options for
          * getting out of check:
          *
          *  - The other king can capture the checking piece, even if it is
          *    defended, or block the check. However, this doesn't work if
          *    the other king is pinned!
          *  - A piece pinned against the other king can still move to
          *    break the check against this king.
          */
         bitboard_t cking = kings ^ king;
         if (cking) {
            cking &= ~mypin;
            if (cking)
               generate_moves_mask(movelist, board, cking, bb, side_to_move, NO_PROMOTION);

            /* A piece pinned on the enemy king can still capture the current
             * piece or block the check.
             */
            bitboard_t pp = get_pinned_pieces(board, kings^king, side_to_move) & ~mypin;
            pp &= ~kings;
            generate_moves_mask(movelist, board, pp, bb, side_to_move, ANY_PROMOTION);
            pinned |= pp;
         }
      }

      /* Generate evasions */
      bitboard_t dest = get_king_attacks(king_square, board_empty) & ~board->bbc[side_to_move];

      /* Mask out other squares on the ray of the checking pieces */
      dest &= ~generate_attack_bitboard_mask(board, 0, ~king, attackers, next_side[side_to_move]);

      /* Mask out other attacked squares */
      bitboard_t bp = dest;
      while (bp) {
         int square = bitscan64(bp);
         bp ^= make_bitboard_square(square);

         if (square_is_attacked_by_side(board, square, next_side[side_to_move]))
            dest ^= make_bitboard_square(square);
      }

      /* Generate evasions */
      generate_moves_mask(movelist, board, make_bitboard_square(king_square), dest, side_to_move, NO_PROMOTION);
   }

   /* Step 2.
    * Generates interpositions/captures of checking piece.
    * We have a list of squares for which blocking is possible.
    */
   if (block) {
      /* Get pinned pieces, these have to be excluded from move generation. */
      //pinned = get_pinned_pieces(board, kings, side_to_move);
      pinned &= board->bbc[side_to_move];

      generate_moves_mask(movelist, board, ~(pinned|kings), block, side_to_move, ANY_PROMOTION);
   }

   /* Step 3.
    * Generate pawn promotions to king.
    * We need to be careful here: it's possible that there is an X-ray
    * attack to the promotion square through the pawn. In this case the
    * promotion is illegal.
    */
   if (onebit64(kings) && kings & board->bbp[BASILEUS]) {
      move_t *move = &(movelist->move[movelist->num_moves]);
      bitboard_t frombb = board_rank2 & board->bbp[BHOPLITE] & ~pinned;
      bitboard_t dest = ~(block | board->bbc[side_to_move]);
      dest &= board_rank1;

      while (frombb) {
         int from = bitscan64(frombb);
         bitboard_t bp = make_bitboard_square(from);
         frombb ^= bp;

         bitboard_t tobb = hoplite_step[from] & dest & ~board->bbc[next_side[side_to_move]];
         tobb |= hoplite_attack[from] & dest & board->bbc[next_side[side_to_move]];

         if (tobb) {
            int to = bitscan64(tobb);
            tobb ^= make_bitboard_square(to);

            if (!square_is_attacked_by_side(board, to, next_side[side_to_move]) &&
                !masked_square_is_attacked_by_side(board, to, ~bp, next_side[side_to_move]) ) {
               int tpiece = piece_for_side(BASILEUS, side_to_move);
               int piece = piece_for_side(get_piece(board, from), side_to_move);

               if (board->bbc[next_side[side_to_move]] & make_bitboard_square(to)) {
                  int ptaken = piece_for_side(get_piece(board, to), next_side[side_to_move]);
                  *move = encode_capture_promotion(piece, from, to, ptaken, tpiece) | MOVE_RESET50;
               } else {
                  *move = encode_normal_promotion(piece, from, to, tpiece) | MOVE_RESET50;
               }
               move++;
            }
         }
      }

      movelist->num_moves = (move - &(movelist->move[0]));
   }

   if (board->holdings[side_to_move])
      generate_gate_moves(movelist, board, side_to_move);
}

void generate_moves(movelist_t *movelist, const board_t *board, sides side_to_move)
{
   assert(movelist);
   assert(board);

   assert(board->in_check == player_in_check(board, board->side_to_move));

   if (board->in_check) {
      generate_evasion_moves(movelist, board, side_to_move);
      return;
   }

   movelist->num_moves = 0;
   movelist->cur_move = 0;

   generate_moves_mask(movelist, board, board_all, board_all, side_to_move, ANY_PROMOTION);
   if (board->holdings[side_to_move])
      generate_gate_moves(movelist, board, side_to_move);
}

void generate_quiescence_moves(movelist_t *movelist, const board_t *board, sides side_to_move)
{
   assert(movelist);
   assert(board);
   int n;

   assert(board->in_check == player_in_check(board, board->side_to_move));
   assert(!board->in_check);

   movelist->num_moves = 0;
   movelist->cur_move = 0;

   bitboard_t mask = board->bbc[next_side[side_to_move]];

   /* Captures */
   generate_moves_mask(movelist, board, board_all, mask, side_to_move, QUE_PROMOTION);

   /* Promotion moves (non-captures) */
   generate_moves_mask(movelist, board, board->bbp[WPAWN]|board->bbp[BHOPLITE],
               (~mask)&(board_rank1|board_rank8), side_to_move, QUE_PROMOTION);

   if (board->holdings[side_to_move])
      generate_gate_moves(movelist, board, side_to_move);

#if 0
   print_bitboards(board);
   printf("%d\n", movelist->num_moves);
   for (n=0; n<movelist->num_moves; n++) {
      printf("%s  ", move_string(movelist->move[n], NULL));
   }
   printf("\n");
   printf_bitboard(board->bbp[next_side[side_to_move]]);
#endif
}

/* Returns the index in the movelist of a valid move corresponding to the
 * given origin/destination. Intended for player interaction.
 * Returns -1 if no moves in the list match (ie, the requested move is
 * invalid). In the case of a promotion, there will be more than one move
 * that matches.
 */
int validate_move(const movelist_t *movelist, int from, int to)
{
   int n;
   if (!movelist)
      return -1;

   for (n=0; n<movelist->num_moves; n++) {
      if (get_move_from(movelist->move[n]) == from && get_move_to(movelist->move[n]) == to)
         return n;
   }
   return -1;
}

